package org.lsst.ccs.drivers.canopenjni;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 * An interface defining interaction with the CANOpen layer
 * @author emarin
 */
public interface CanOpenInterface {
    
    /**
     * Initialization of the master node. This command has to be issued first
     * for the subsequent one to be successful.
     *
     * @param master 1 for master, 0 for slave.
     * @param baud "125K","1M",...
     * @param busName the bus name (0 for can0, 1 for can1)
     * @param nodeID the nodeID
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    @Command
    public void init(int master, String baud, String busName, int nodeID) throws DriverException ;
    
    /**
     * Configures the pdo with the given cob_id for reception on the master node.
     * @param cobId 
     */
    public void addReceivedPDO(int cobId) throws DriverException;
    
    /**
     * Sends a sync message on the CANBus and synchronously waits for the received 
     * PDO configured in the master to be received.
     * For it to work, all received PDOs configured on the master node must have 
     * a transmission type equal to Ox1, otherwise the behavior is undefined.
     * @return the PDOs.
     * @throws DriverException 
     */
    public PDOData sync() throws DriverException;
    
    /**
     * Sends a scan on the can bus.
     * @return 
     */
    public int scan() throws DriverException ;
    
    /**
     * Returns the information for the given nodeID.
     * @param nodeID the node id
     * @return 
     * @throws org.lsst.ccs.drivers.commons.DriverException 
     */
    public String info(int nodeID) throws DriverException;
    
    /**
     * 
     * @param nodeId the node id
     * @param index the index
     * @param subindex the sub index
     * @param size the size
     * @param data the data
     * @throws DriverException 
     */
    public void wsdo(int nodeId, int index, int subindex, int size, long data) throws DriverException;
    
    /**
     * Read device entry
     * @param nodeId the node id
     * @param index the index
     * @param subindex the subindex
     * @return an unsigned long. Cast to int to treat it as signed, leave to long to treat it as unsigned
     * @throws org.lsst.ccs.drivers.commons.DriverException
     */
    public long rsdo(int nodeId, int index, int subindex) throws DriverException;
    
    /**
     * Slave start
     * @param nodeId
     */
    public void ssta(int nodeId) throws DriverException;
    
    /**
     * Slave stop
     * @param nodeId
     */
    public void ssto(int nodeId) throws DriverException;
    
    public void reset(int nodeId) throws DriverException;

    public void quit() throws DriverException;
    
    // Lifecycle
    public void init() throws DriverException;
    
    public void start() throws DriverException;
    
    public void stop() throws DriverException;
    
    public boolean isReady() throws DriverException;
    
    // Call backs for asynchronous information
    public void setEmergencyMessageListener(EmergencyMessageListener eml) throws DriverException;
    
    public void setBootMessageListener(BootMessageListener bml) throws DriverException;
    
}
