package org.lsst.ccs.drivers.canopenjni;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *
 * @author emarin
 */
public class CanFestivalJNI implements CanOpenInterface {
    
    private static final Logger log = Logger.getLogger("org.lssst.ccs.drivers.canopenjni");

    static {
        System.out.println("**** Loading the library ");
        try {
            System.loadLibrary("canopenJNI");
        } catch (Exception e) {
            e.printStackTrace();
        }
        System.out.println("**** Done Loading the library ");
    }
    
    private long timeout = 5000;
    
    private int index = 0x2000;
    
    private BootMessageListener bml = (nodeId) -> {System.out.println("boot : " + nodeId);};
    
    private EmergencyMessageListener eml = (nodeId, errCode, errReg) -> 
    {System.out.println("emergency message for " + nodeId + " :(" + errCode + ":"+errReg+")");};
    
    @Override
    @Command
    public void addReceivedPDO(int cobId) {
         if (index >= 0x2008) {
            log.warning("cannot add transmit pdos : max number of received pdos reached");
        } else {
             addReceivedPDO(cobId, index);
             index ++;
        }
    }
    
    /**
     * Enables reception of the PDO identified by cob_id at the specified index.
     * @param cob_id
     * @param index starts at 0x2000 and increments by 1.
     */
    private native void addReceivedPDO(int index, int cobId);
    
    @Command
    @Override
    public native void init(int master, String baud, String busName, int nodeID);
    
    @Override
    public void setEmergencyMessageListener(EmergencyMessageListener eml) {
        this.eml = eml;
    }
    
    // Called back from jni
    private void onEmergencyMessage(int nodeid, int errCode, int errReg) {
         eml.onEmergencyMessage(nodeid, errCode, errReg);
    }

    @Override
    public void setBootMessageListener(BootMessageListener bml) {
        this.bml = bml;
    }
    
    // Called back from jni
    private void onSlaveBootup(int nodeid) {
        bml.onBootMessage(nodeid);
    }

    /**
     * Sends a sync message on the canbus and synchronously waits for the received 
     * pdo configured in the master to be received.
     * For it to work, all received pdos configured on the master node must have 
     * a transmission type equal to Ox1, otherwise the behaviour is undefined.
     * @return The PDOs
     * @throws DriverException 
     */
    @Override
    @Command
    public PDOData sync() throws DriverException {
        PDOData rd = new PDOData();
        
        rd = new SyncCommandWrapper<PDOData>(
                (pdocb, r) -> {
                    syncAsync(pdocb, r);
                }, rd).get();
        if (rd.errMsg!=null && !rd.errMsg.isEmpty()) {
            throw new DriverException(rd.errMsg);
        }
        
        return rd;
    }
    
    /**
     * Sends a SYNC request.
     */
    private synchronized native void syncAsync(CallbackListener cb, PDOData data);

    @Command
    @Override
    public native int scan();

    @Override
    @Command
    public String info(int nodeID) throws DriverException {
        long deviceType = rsdo(nodeID, 0x1000, 0x00);
        long vendorID   = rsdo(nodeID, 0x1018, 0x01);
        long productCode= rsdo(nodeID, 0x1018, 0x02);
        long revNumber  = rsdo(nodeID, 0x1018, 0x03);
        long serialNb   = rsdo(nodeID, 0x1018, 0x04);
        
        StringBuilder res = new StringBuilder(nodeID);
        
        return res.append(": type :").append(Long.toHexString(deviceType))
                .append(", vendor :").append(Long.toHexString(vendorID))
                .append(", product :").append(Long.toHexString(productCode))
                .append(", revision :").append(Long.toHexString(revNumber))
                .append(", serialNB :").append(Long.toHexString(serialNb))
                .toString();
    }
    
    @Override
    @Command
    public void wsdo(int nodeId, int index, int subindex, int size, long data) throws DriverException {
        ReturnData rd = new ReturnData();
        
        rd = new SyncCommandWrapper<ReturnData>(
                (sdocb, r) -> {
                    wsdoAsync(nodeId, index, subindex, size, data, sdocb, r);
                }, rd).get();
        
        if (rd.errMsg != null && !rd.errMsg.isEmpty()) {
            throw new DriverException(rd.errMsg);
        }
    }

    /**
     * Write device entry
     * @param nodeID the node id 
     * @param index the index
     * @param subindex the subindex
     * @param size the size of the data
     * @param data the data.
     * @param cb the callback 
     */
    private synchronized native void wsdoAsync(int nodeID, int index, int subindex, int size, long data, CallbackListener cb, ReturnData rd);

    @Override
    @Command
    public long rsdo(int nodeId, int index, int subindex) throws DriverException {   
        SDOData ret = new SDOData();
        
        new SyncCommandWrapper<SDOData>(
                (sdocb, rd) -> {
                    rsdoAsync(nodeId, index, subindex, sdocb, rd);   
                }, ret).get();
        
        if (ret.errMsg != null && !ret.errMsg.isEmpty()) {
            throw new SDOException(ret.errMsg);
        } else {
            return ret.data;
        }
    }
    
    private synchronized native void rsdoAsync(int nodeId, int index, int subindex, CallbackListener cb, ReturnData rd);

    /**
     * Slave start
     * @param nodeId
     */
    @Override
    public native void ssta(int nodeId);

    /**
     * Slave stop
     * @param nodeId
     */
    @Override
    public native void ssto(int nodeId);
    
    /**
     * Slave reset
     * @param nodeId 
     */
    @Override
    public native void reset(int nodeId);

    @Override
    public native void quit();

    // Lifecycle methods.
    @Override
    public void init() {

    }

    @Override
    public void start() {

    }

    @Override
    public void stop() {

    }

    @Override
    public boolean isReady() {
        return true;
    }
    
    @Command
    public void setTimeoutMillis(long to) {
        this.timeout = to;
    }

    private interface CommandWithCallback<T extends ReturnData> {
        
        public void call(CallbackListener cb, T rd);
        
    }
    
    private class SyncCommandWrapper<T extends ReturnData> {
        
        private final CommandWithCallback callable;
        private final T rd;
        
        SyncCommandWrapper(CommandWithCallback<T> asyncCommand, T rd) {
            this.callable = asyncCommand;
            this.rd = rd;
        }
        
        T get() throws DriverException {
            final Lock cblock = new ReentrantLock();
            Condition cb = cblock.newCondition();
            
            CallbackListener sdoCB = new CallbackListener() {
                
                @Override
                public void callback() {
                    cblock.lock();
                    try {
                        cb.signalAll();
                    } finally {
                        cblock.unlock();
                    }
                }
            };
            cblock.lock();
            try {
                callable.call(sdoCB, rd);
                while(!rd.set){
                    if (!cb.await(timeout, TimeUnit.MILLISECONDS)) {
                        clearCallBack();
                        throw new DriverTimeoutException("command timeout");
                    }
                }
            } catch (InterruptedException ex) {
                clearCallBack();
                throw new DriverException("interrupted", ex);
            } catch (Exception ex) {
                throw new DriverException(ex);
            } finally {
                cblock.unlock();
            }
            return rd;
        }
        
    }
    
    /**
     * to be used in case of timeout. Native call is not reentrant.
     */
    @Command
    public synchronized native void clearCallBack();
    
}
