package org.lsst.ccs.drivers.bonnshutter;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 * A driver for the Bonn Shutter used for ComCam.
 * @author tonyj
 */
public class BonnShutter extends Ascii {

    private final Pattern OPEN_CLOSE_PATTERN = Pattern.compile("(\\d+).*",Pattern.DOTALL);
    private final int BAUD_RATE = 19200;
    public enum Blade { A, B };
    public enum Mode { A, B };
    public enum OpenCloseStatus { UNKNOWN, OPEN, CLOSED_A, CLOSED_B };

    public void open(String serialName) throws DriverException {
        int characteristics = makeDataCharacteristics(Ascii.DataBits.EIGHT, Ascii.StopBits.ONE, Ascii.Parity.NONE, Ascii.FlowCtrl.NONE);
        open(ConnType.SERIAL, serialName, BAUD_RATE, characteristics);
        setTerminator(Ascii.Terminator.CR);
        this.setTimeout(1.0);
        // Send an initial command and wait for response
        setInteractive(false);
    }

    public String getVersion() throws DriverException {
        return sendCommandAndWaitForSingleLineResponse("ve");
    }

    public void setInteractive(boolean mode) throws DriverException {
        sendCommandAndWaitForPrompt("ia "+(mode?0:1));
    }
    
    public List<String> getMotorFirmwareVersion() throws DriverException {
        return sendCommandAndWaitForMultiLineResponse("gv bl");
    }

    public List<String> getVerboseVelocityProfileParameters() throws DriverException {
        return sendCommandAndWaitForMultiLineResponse("sh");
    }

    public ProfileParameters getVelocityProfileParameters() throws DriverException {
        String result = sendCommandAndWaitForSingleLineResponse("pp");
        return new ProfileParameters(result);
    }
    
    public OpenCloseStatus getOpenCloseStatus() throws DriverException {
        String result = sendCommandAndWaitForSingleLineResponse("ss");
        final Matcher matcher = OPEN_CLOSE_PATTERN.matcher(result);
        if (!matcher.matches()) throw new DriverException("Unexpected response: "+result);
        int ordinal = Integer.parseInt(matcher.group(1));
        return OpenCloseStatus.values()[ordinal];
    }

    public void openShutter() throws DriverException {
        sendCommandAndWaitForPrompt("os");
    }

    public void closeShutter() throws DriverException {
        sendCommandAndWaitForPrompt("cs");
    }

    public void expose(int exposeMillis) throws DriverException {
        sendCommandAndWaitForPrompt("ex "+exposeMillis);
    }
    
    public void exposeSeries(int n, int exposeMillis, int waitMillis) throws DriverException {
        sendCommandAndWaitForPrompt("xx "+exposeMillis+" "+waitMillis+" "+n);
    }
    
    public void reset() throws DriverException {
        sendCommandAndWaitForPrompt("rs");
    }

    public Voltages getVoltages() throws DriverException {
        String result = sendCommandAndWaitForSingleLineResponse("vo");
        return new Voltages(result);
    }

    public void moveAbsolute(Blade blade, int velocity, int position) throws DriverException {
        sendCommandAndWaitForPrompt("ma "+" "+velocity+" "+position+" "+blade.ordinal());
    }

    public void moveRelative(Blade blade, int velocity, int position) throws DriverException {
        sendCommandAndWaitForPrompt("mr "+" "+velocity+" "+position+" "+blade.ordinal());
    }
    
    public String getMotorPosition(Blade blade) throws DriverException {
        return sendCommandAndWaitForSingleLineResponse("sp "+blade.ordinal());
    }
    
    public List<String> executeGenericCommand(String command) throws DriverException {
        return sendCommandAndWaitForMultiLineResponse(command);
    }

    private void sendCommandAndWaitForPrompt(String command) throws DriverException {
        List<String> result = sendCommandAndWaitForMultiLineResponse(command);
        if (!result.isEmpty()) {
            throw new DriverException("Unexpected response for command: " + command);
        }
    }

    private String sendCommandAndWaitForSingleLineResponse(String command) throws DriverException {
        List<String> result = sendCommandAndWaitForMultiLineResponse(command);
        if (result.size() != 1) {
            throw new DriverException("Unexpected response for command: " + command);
        }
        return result.get(0);
    }

    private synchronized List<String> sendCommandAndWaitForMultiLineResponse(String command) throws DriverException {
        write(command);
        List<String> result = new ArrayList<>();
        byte[] buff = new byte[1024];
        int startOfLine = 0;
        int offset = 0;
        for (;;) {
            int l = this.readBytes(buff, offset);
            for (int i = offset; i < offset + l; i++) {
                if (buff[i] == '\r') {
                    String line = new String(buff, startOfLine, i);
                    //System.out.println("line="+line);
                    result.add(line);
                    startOfLine = i + 1;
                } else if (i - startOfLine == 0 && buff[i] == '\n') {
                    startOfLine++;
                } else if (i - startOfLine == 1 && buff[i - 1] == 'c') {
                    if (buff[i] == '>') {
                        return result;
                    } else if (buff[i] == '?') {
                        throw new CommandRejected(command);
                    }
                }
            }
            offset += l;
            //Copy any unused characters to beginning of buffer
            if (startOfLine > 0) {
                System.arraycopy(buff, startOfLine, buff, 0, offset - startOfLine);
                offset -= startOfLine;
                startOfLine = 0;
            }
        }
    }

    public static class ProfileParameters {

        private final int startBladeA;
        private final int startBladeB;
        private final int bladeTravelDistance;
        private final double startVelocity;
        private final double acceleration;
        private final double maxVelocity;
        private final double threshold;
        private final double bladeTravelTime;

        private ProfileParameters(String result) {
            String[] tokens = result.split("\\s+");
            startBladeA = Integer.parseInt(tokens[0]);
            startBladeB = Integer.parseInt(tokens[1]);
            bladeTravelDistance = Integer.parseInt(tokens[2]);
            startVelocity = Double.parseDouble(tokens[3]);
            acceleration = Double.parseDouble(tokens[4]);
            maxVelocity = Double.parseDouble(tokens[5]);
            threshold = Double.parseDouble(tokens[6]);
            bladeTravelTime = Double.parseDouble(tokens[7]);
        }

        public int getStartBladeA() {
            return startBladeA;
        }

        public int getStartBladeB() {
            return startBladeB;
        }

        public int getBladeTravelDistance() {
            return bladeTravelDistance;
        }

        public double getStartVelocity() {
            return startVelocity;
        }

        public double getAcceleration() {
            return acceleration;
        }

        public double getMaxVelocity() {
            return maxVelocity;
        }

        public double getThreshold() {
            return threshold;
        }

        public double getBladeTravelTime() {
            return bladeTravelTime;
        }

        @Override
        public String toString() {
            return "ProfileParameters{" + "startBladeA=" + startBladeA + ", startBladeB=" + startBladeB + ", bladeTravelDistance=" + bladeTravelDistance + ", startVelocity=" + startVelocity + ", acceleration=" + acceleration + ", maxVelocity=" + maxVelocity + ", threshold=" + threshold + ", bladeTravelTime=" + bladeTravelTime + '}';
        }
    }

    public static class CommandRejected extends DriverException {

        private CommandRejected(String command) {
            super("Command rejected: " + command);
        }
    }

    public static class Voltages {

        private static Pattern pattern = Pattern.compile(".* ([0-9.]+)V ([0-9.]+)V");
        private double v5;
        private double v36;

        private Voltages(String result) throws DriverException {
            Matcher matcher = pattern.matcher(result);
            if (!matcher.find()) {
                throw new DriverException("Unexpected response: " + result);
            }
            v5 = Double.parseDouble(matcher.group(1));
            v36 = Double.parseDouble(matcher.group(2));
        }

        public double getV5() {
            return v5;
        }

        public double getV36() {
            return v36;
        }

        @Override
        public String toString() {
            return "Voltages{" + "v5=" + v5 + ", v36=" + v36 + '}';
        }
    }
}
