package org.lsst.ccs.drivers.bk;

import java.util.Timer;
import java.util.TimerTask;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.PowerSupplyDriver;
import org.lsst.ccs.drivers.scpi.Scpi;

/**
 *  Routines for controlling a B&K Precision XLN series power supply
 *
 *  @author  Owen Saxton
 */
public class XLNSeries extends Scpi implements PowerSupplyDriver {

    public static final int
        DEFAULT_BAUDRATE = 57600,
        DEFAULT_PORT     = 5025,
        STATUS_LCD_ON    = 0x0002,
        STATUS_OUTPUT_ON = 0x0004,
        STATUS_CVCC_ENAB = 0x0008,
        STATUS_CCCV_ENAB = 0x0010,
        STATUS_OPP_ENAB  = 0x0020,
        STATUS_OCP_ENAB  = 0x0040,
        STATUS_OVP_ENAB  = 0x0080,
        STATUS_OTP       = 0x0200,
        STATUS_AC_LOW    = 0x0400,
        STATUS_CVCC      = 0x0800,
        STATUS_CCCV      = 0x1000,
        STATUS_OPP       = 0x2000,
        STATUS_OCP       = 0x4000,
        STATUS_OVP       = 0x8000;

    private final Timer timer = new Timer("Timer_XLN", true);
    private long onDelay = 0, offDelay = 0;
    private SetOutput outTask;


    /**
     *  Timer task to implement the delayed setting of the output state.
     */
    class SetOutput extends TimerTask {

        private final String command;

        SetOutput(String command)
        {
            this.command = command;
        }
    
        @Override
        public void run()
        {
            try {
                writeCommand(command);
            }
            catch (DriverException e) {
            }
            outTask = null;
        }
    }


    /**
     *  Constructor.
     */
    public XLNSeries()
    {
        setDefaultBaud(DEFAULT_BAUDRATE);
        setDefaultPort(DEFAULT_PORT);
    }


    /**
     *  Opens a connection.
     *
     *  This overrides the implementation in Scpi/Ascii and is called by all other open methods.
     * 
     *  @param  connType  The enumerated connection type: NET or SERIAL
     *  @param  ident     The IP address (NET) or serial port name (SERIAL)
     *  @param  connParm  The port number or baud rate (not used)
     *  @param  commParm  The communications parameters (not used)
     *  @throws  DriverException
     */
    @Override
    public void open(ConnType connType, String ident, int connParm, int commParm) throws DriverException
    {
        super.open(connType, ident, 0, 0);
        setTerminator(Terminator.CRLF);
        try {
            checkIdentification("B&K PRECISION", CHECK_EQUALS, "XLN", CHECK_STARTS_WITH);
        }
        catch (DriverException e) {
            closeSilent();
            throw e;
        }
    }


    /**
     *  Turns output on or off for a channel.
     *
     *  @param  on    Turn on if true, off if false
     *  @param  chan  The channel number (ignored)
     *  @throws  DriverException
     */
    @Override
    public void setOutput(boolean on, int chan) throws DriverException
    {
        long delay = on ? onDelay : offDelay;
        String command = "OUT " + (on ? "ON" : "OFF");
        SetOutput task = outTask;
        if (task != null) {
            task.cancel();
            outTask = null;
        }
        if (delay == 0) {
            writeCommand(command);
        }
        else {
            outTask = new SetOutput(command);
            timer.schedule(outTask, delay);
        }
    }


    /**
     *  Gets the output state of a channel.
     *
     *  @param  chan  The channel number (ignored)
     *  @return  The output state
     *  @throws  DriverException
     */
    @Override
    public boolean getOutput(int chan) throws DriverException
    {
        return readString("OUT?").equals("ON");
    }


    /**
     *  Sets the power-on delay for a channel.
     *
     *  @param  time  The delay (sec)
     *  @param  chan  The channel number (ignored)
     *  @throws  DriverException
     */
    @Override
    public void setOnDelay(double time, int chan) throws DriverException
    {
        onDelay = (long)(1000 * time);
    }


    /**
     *  Sets the power-off delay for a channel.
     *
     *  @param  time  The delay (sec)
     *  @param  chan  The channel number (ignored)
     *  @throws  DriverException
     */
    @Override
    public void setOffDelay(double time, int chan) throws DriverException
    {
        offDelay = (long)(1000 * time);
    }


    /**
     *  Sets the voltage for a channel.
     *
     *  @param  value  The value to set
     *  @param  chan   The channel number (ignored)
     *  @throws  DriverException
     */
    @Override
    public void setVoltage(double value, int chan) throws DriverException
    {
        writeCommand("VOLT " + value);
    }


    /**
     *  Gets the set voltage for a channel.
     *
     *  @param  chan   The channel number (ignored)
     *  @return  The voltage
     *  @throws  DriverException
     */
    @Override
    public double getVoltage(int chan) throws DriverException
    {
        return readDouble("VOLT?");
    }


    /**
     *  Reads the voltage for a channel.
     *
     *  @param  chan  The channel number (ignored)
     *  @return  The voltage
     *  @throws  DriverException
     */
    @Override
    public double readVoltage(int chan) throws DriverException
    {
        return readDouble("MEAS:VOLT?");
    }


    /**
     *  Sets the current for a channel.
     *
     *  @param  value  The value to set
     *  @param  chan   The channel number (ignored)
     *  @throws  DriverException
     */
    @Override
    public void setCurrent(double value, int chan) throws DriverException
    {
        writeCommand("CURR " + value);
    }


    /**
     *  Gets the set current for a channel.
     *
     *  @param  chan   The channel number (ignored)
     *  @return  The current
     *  @throws  DriverException
     */
    @Override
    public double getCurrent(int chan) throws DriverException
    {
        return readDouble("CURR?");
    }


    /**
     *  Reads the current for a channel.
     *
     *  @param  chan  The channel number (ignored)
     *  @return  The current
     *  @throws  DriverException
     */
    @Override
    public double readCurrent(int chan) throws DriverException
    {
        return readDouble("MEAS:CURR?");
    }


    /**
     *  Sets the voltage limit for a channel.
     *
     *  @param  value  The value to set
     *  @param  chan   The channel number (ignored)
     *  @throws  DriverException
     */
    public void setVoltageLimit(double value, int chan) throws DriverException
    {
        writeCommand("OUT:LIM:VOLT" + value);
    }


    /**
     *  Gets the voltage limit for a channel.
     *
     *  @param  chan  The channel number (ignored)
     *  @return  The voltage
     *  @throws  DriverException
     */
    public double getVoltageLimit(int chan) throws DriverException
    {
        return readDouble("OUT:LIM:VOLT?");
    }


    /**
     *  Sets the current limit for a channel.
     *
     *  @param  value  The value to set
     *  @param  chan   The channel number (ignored)
     *  @throws  DriverException
     */
    public void setCurrentLimit(double value, int chan) throws DriverException
    {
        writeCommand("OUT:LIM:CURR" + value);
    }


    /**
     *  Gets the current limit for a channel.
     *
     *  @param  chan  The channel number (ignored)
     *  @return  The current
     *  @throws  DriverException
     */
    public double getCurrentLimit(int chan) throws DriverException
    {
        return readDouble("OUT:LIM:CURR?");
    }


    /**
     *  Gets the status word.
     *
     *  @return  The status word
     *  @throws  DriverException
     */
    public int getStatusWord() throws DriverException
    {
        try {
            return Integer.valueOf(readString("STATUS?"), 16);
        }
        catch (NumberFormatException e) {
            throw new DriverException(e);
        }
    }


    /**
     *  Locks or unlocks the front panel.
     *
     *  @param  lock  True to lock the panel; false to unlock it
     *  @throws  DriverException
     */
    public void lockPanel(boolean lock) throws DriverException
    {
        writeCommand("SYS:KEY:LOCK " + (lock ? 1 : 0));
    }


    /**
     *  Checks for command error.
     *
     *  @throws  DriverException
     */
    @Override
    protected void checkError() throws DriverException
    {
        try {
            int error = Integer.valueOf(getError());
            if (error == 0) return;
            String text;
            switch (error) {
            case 1:
                text = "Unknown command"; break;
            case 2:
                text = "Execution failed"; break;
            case 3:
                text = "Query failed"; break;
            case 4:
                text = "Parameter out of range"; break;
            default:
                text = "Unexpected error";
            }
            throw new DriverException("SCPI error: " + error + ": " + text);
        }
        catch (NumberFormatException e) {
            throw new DriverException(e);
        }
    }


    /**
     *  Gets the event status enable register value.
     *
     *  @return  0
     */
    @Override
    public int getEventStatusEnable()
    {
        return 0;
    }


    /**
     *  Gets the event status register value.
     *
     *  @return  0
     */
    @Override
    public int getEventStatus()
    {
        return 0;
    }


    /**
     *  Gets the service request enable register value.
     *
     *  @return  0
     */
    @Override
    public int getServiceRequestEnable()
    {
        return 0;
    }


    /**
     *  Gets the status byte value.
     *
     *  @return  0
     */
    @Override
    public int getStatusByte()
    {
        return 0;
    }


    /**
     *  Gets the operation complete state.
     *
     *  @return  0
     */
    @Override
    public int getOperationComplete()
    {
        return 0;
    }

}
