package org.lsst.ccs.drivers.bk;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.TestScpi;

/**
 *  Program to test the B&K Precision XLN series device driver
 * 
 *  @author Owen Saxton
 */
public class TestXLN extends TestScpi {

    private final XLNSeries bk;
    private final int chanNum = 1;

    public enum OnOff {
        OFF, ON;
    }


   /**
    *  Constructor
    */
    public TestXLN()
    {
        super(new XLNSeries());
        bk = (XLNSeries)dev;
    }


   /**
    *  Sets the voltage.
    * 
    *  @param  value  The value to set
    *  @throws  DriverException
    */
    @Command(name="setvoltage", description="Set the voltage")
    public void setVoltage(@Argument(name="value", description="Voltage to set")
                           double value) throws DriverException
    {
        bk.setVoltage(value, chanNum);
    }


   /**
    *  Shows the set voltage.
    * 
    *  @return  The result string
    *  @throws  DriverException
    */
    @Command(name="showvoltage", description="Show the set voltage")
    public String showVoltage() throws DriverException
    {
        return "Voltage = " + bk.getVoltage(chanNum);
    }


   /**
    *  Reads the actual voltage.
    * 
    *  @return  The result string
    *  @throws  DriverException
    */
    @Command(name="readvoltage", description="Read the actual voltage")
    public String readVoltage() throws DriverException
    {
        return "Voltage = " + bk.readVoltage(chanNum);
    }


   /**
    *  Sets the soft voltage limit.
    * 
    *  @param  value  The value to set
    *  @throws  DriverException
    */
    @Command(name="setvoltlim", description="Set the soft voltage limit")
    public void setVoltageLimit(@Argument(name="value", description="Voltage limit to set")
                                double value) throws DriverException
    {
        bk.setVoltageLimit(value, chanNum);
    }


   /**
    *  Shows the soft voltage limit.
    * 
    *  @return  The result string
    *  @throws  DriverException
    */
    @Command(name="showvoltlim", description="Show the soft voltage limit")
    public String showVoltageLimit() throws DriverException
    {
        return "Voltage limit = " + bk.getVoltageLimit(chanNum);
    }


   /**
    *  Sets the current.
    * 
    *  @param  value  The value to set
    *  @throws  DriverException
    */
    @Command(name="setcurrent", description="Set the current")
    public void setCurrent(@Argument(name="value", description="Current to set")
                           double value) throws DriverException
    {
        bk.setCurrent(value, chanNum);
    }


   /**
    *  Shows the set current.
    * 
    *  @return  The result string
    *  @throws  DriverException
    */
    @Command(name="showcurrent", description="Show the set current")
    public String showCurrent() throws DriverException
    {
        return "Current = " + bk.getCurrent(chanNum);
    }


   /**
    *  Reads the actual current.
    * 
    *  @return  The result string
    *  @throws  DriverException
    */
    @Command(name="readcurrent", description="Read the actual current")
    public String readCurrent() throws DriverException
    {
        return "Current = " + bk.readCurrent(chanNum);
    }


   /**
    *  Sets the soft current limit.
    * 
    *  @param  value  The value to set
    *  @throws  DriverException
    */
    @Command(name="setcurrlim", description="Set the soft current limit")
    public void setCurrentLimit(@Argument(name="value", description="Current limit to set")
                                double value) throws DriverException
    {
        bk.setCurrentLimit(value, chanNum);
    }


   /**
    *  Shows the soft current limit.
    * 
    *  @return  The result string
    *  @throws  DriverException
    */
    @Command(name="showcurrlim", description="Show the soft current limit")
    public String showCurrentLimit() throws DriverException
    {
        return "Current limit = " + bk.getCurrentLimit(chanNum);
    }


   /**
    *  Sets the output on delay.
    * 
    *  @param  delay  The delay to power on (sec)
    *  @throws  DriverException
    */
    @Command(name="setondelay", description="Set the output on delay")
    public void setOnDelay(@Argument(name="delay", description="Delay time (sec)")
                           double delay) throws DriverException
    {
        bk.setOnDelay(delay, chanNum);
    }


   /**
    *  Sets the output off delay.
    * 
    *  @param  delay  The delay to power off (sec)
    *  @throws  DriverException
    */
    @Command(name="setoffdelay", description="Set the output off delay")
    public void setOffDelay(@Argument(name="delay", description="Delay time (sec)")
                            double delay) throws DriverException
    {
        bk.setOffDelay(delay, chanNum);
    }


   /**
    *  Turns the output on or off.
    * 
    *  @param  state  The state to set: on or off
    *  @throws  DriverException
    */
    @Command(name="setoutput", description="Turn output on or off")
    public void setOutput(@Argument(name="state", description="Output state: on or off")
                          OnOff state) throws DriverException
    {
        bk.setOutput(state == OnOff.ON, chanNum);
    }


   /**
    *  Shows the output state.
    * 
    *  @return  The result string
    *  @throws  DriverException
    */
    @Command(name="showoutput", description="Show the output state")
    public String showOutput() throws DriverException
    {
        return "Output state = " + getOnOff(bk.getOutput(chanNum));
    }


   /**
    *  Locks the front panel.
    * 
    *  @throws  DriverException
    */
    @Command(name="lock", description="Lock the front panel")
    public void lock() throws DriverException
    {
        bk.lockPanel(true);
    }


   /**
    *  Unlocks the front panel.
    * 
    *  @throws  DriverException
    */
    @Command(name="unlock", description="Unlock the front panel")
    public void unlock() throws DriverException
    {
        bk.lockPanel(false);
    }


   /**
    *  Shows the status word.
    * 
    *  @return  The result string
    *  @throws  DriverException
    */
    @Command(name="showstatusword", description="Show the status word")
    public String showStatusWord() throws DriverException
    {
        return String.format("Status word = 0x%06x", bk.getStatusWord());
    }


   /**
    *  Converts a boolean to on/off.
    */
    private String getOnOff(boolean on)
    {
        return on ? "on" : "off";
    }

}
