package org.lsst.ccs.drivers.bk;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.ascii.TestAscii;

/**
 *  Program to test the B&K Precision model 1696 device driver
 * 
 *  @author Owen Saxton
 */
public class Test1696 extends TestAscii {

    private final Model1696 bk;
    private final int chanNum = 1;

    public enum OnOff {
        OFF, ON;
    }


   /**
    *  Constructor
    */
    public Test1696()
    {
        super(new Model1696());
        bk = (Model1696)dev;
    }


   /**
    *  Opens connection to a device.
    * 
    *  @param  connType  The type of connection: ftdi or serial
    *  @param  ident     The identification: serial no. (ftdi) or port name (serial)
    *  @throws  DriverException
    */
    @Command(name="open", description="Open connection to device")
    public void open(@Argument(name="type", description="Connection type")
                     Model1696.ConnType connType,
                     @Argument(name="ident", description="Identification")
                     String ident) throws DriverException
    {
        bk.open(connType, ident);
    }


   /**
    *  Sets the voltage.
    * 
    *  @param  value  The value to set
    *  @throws  DriverException
    */
    @Command(name="setvoltage", description="Set the voltage")
    public void setVoltage(@Argument(name="value", description="Voltage to set")
                           double value) throws DriverException
    {
        bk.setVoltage(value, chanNum);
    }


   /**
    *  Shows the set voltage.
    * 
    *  @return  The result string
    *  @throws  DriverException
    */
    @Command(name="showvoltage", description="Show the set voltage")
    public String showVoltage() throws DriverException
    {
        return "Voltage = " + bk.getVoltage(chanNum);
    }


   /**
    *  Reads the actual voltage.
    * 
    *  @return  The result string
    *  @throws  DriverException
    */
    @Command(name="readvoltage", description="Read the actual voltage")
    public String readVoltage() throws DriverException
    {
        return "Voltage = " + bk.readVoltage(chanNum);
    }


   /**
    *  Sets the soft voltage limit.
    * 
    *  @param  value  The value to set
    *  @throws  DriverException
    */
    @Command(name="setvoltlim", description="Set the soft voltage limit")
    public void setVoltageLimit(@Argument(name="value", description="Voltage limit to set")
                                double value) throws DriverException
    {
        bk.setVoltageLimit(value, chanNum);
    }


   /**
    *  Shows the soft voltage limit.
    * 
    *  @return  The result string
    *  @throws  DriverException
    */
    @Command(name="showvoltlim", description="Show the soft voltage limit")
    public String showVoltageLimit() throws DriverException
    {
        return "Voltage limit = " + bk.getVoltageLimit(chanNum);
    }


   /**
    *  Sets the current.
    * 
    *  @param  value  The value to set
    *  @throws  DriverException
    */
    @Command(name="setcurrent", description="Set the current")
    public void setCurrent(@Argument(name="value", description="Current to set")
                           double value) throws DriverException
    {
        bk.setCurrent(value, chanNum);
    }


   /**
    *  Shows the set current.
    * 
    *  @return  The result string
    *  @throws  DriverException
    */
    @Command(name="showcurrent", description="Show the set current")
    public String showCurrent() throws DriverException
    {
        return "Current = " + bk.getCurrent(chanNum);
    }


   /**
    *  Reads the actual current.
    * 
    *  @return  The result string
    *  @throws  DriverException
    */
    @Command(name="readcurrent", description="Read the actual current")
    public String readCurrent() throws DriverException
    {
        return "Current = " + bk.readCurrent(chanNum);
    }


   /**
    *  Turns the output on or off.
    * 
    *  @param  state  The state to set: on or off
    *  @throws  DriverException
    */
    @Command(name="setoutput", description="Turn output on or off")
    public void setOutput(@Argument(name="state", description="Output state: on or off")
                          OnOff state) throws DriverException
    {
        bk.setOutput(state == OnOff.ON, chanNum);
    }


   /**
    *  Shows the output state.
    * 
    *  @return  The result string
    *  @throws  DriverException
    */
    @Command(name="showoutput", description="Show the output state")
    public String showOutput() throws DriverException
    {
        return "Output state = " + getOnOff(bk.getOutput(chanNum));
    }


   /**
    *  Locks the front panel.
    * 
    *  @throws  DriverException
    */
    @Command(name="lock", description="Lock the front panel")
    public void lock() throws DriverException
    {
        bk.lockPanel(true, chanNum);
    }


   /**
    *  Unlocks the front panel.
    * 
    *  @throws  DriverException
    */
    @Command(name="unlock", description="Unlock the front panel")
    public void unlock() throws DriverException
    {
        bk.lockPanel(false, chanNum);
    }


   /**
    *  Shows the front panel state.
    * 
    *  @return  The result string
    *  @throws  DriverException
    */
    @Command(name="showpanel", description="Show the front panel state")
    public String showPanel() throws DriverException
    {
        Model1696.Panel panel = bk.getPanel(chanNum);
        return   " Read voltage = " + String.format("%5s", panel.readVoltage)
            +  "   Read current = " + String.format("%5s", panel.readCurrent)
            +  "\n Read power   = " + String.format("%5s", panel.readPower)
            +  "   Set voltage  = " + String.format("%5s", panel.setVoltage)
            +  "\n Set current  = " + String.format("%5s", panel.setCurrent)
            +  "   Timer mins   = " + String.format("%5s", panel.minutes)
            +  "\n Timer secs   = " + String.format("%5s", panel.seconds)
            +  "   Program no.  = " + String.format("%5s", panel.program)
            +  "\n Timer disp   = " + String.format("%5s", panel.timerDisplay)
            +  "   Colon disp   = " + String.format("%5s", panel.colonDisplay)
            +  "\n Vconst disp  = " + String.format("%5s", panel.vConstDisplay)
            +  "   Vset disp    = " + String.format("%5s", panel.vSetDisplay)
            +  "\n Iconst disp  = " + String.format("%5s", panel.iConstDisplay)
            +  "   Iset disp    = " + String.format("%5s", panel.iSetDisplay)
            +  "\n Program disp = " + String.format("%5s", panel.programDisplay)
            +  "   Setting disp = " + String.format("%5s", panel.settingDisplay)
            +  "\n Lock disp    = " + String.format("%5s", panel.lockDisplay)
            +  "   Unlock disp  = " + String.format("%5s", panel.unlockDisplay)
            +  "\n Fault disp   = " + String.format("%5s", panel.faultDisplay)
            +  "   Out on disp  = " + String.format("%5s", panel.onDisplay)
            +  "\n Out off disp = " + String.format("%5s", panel.offDisplay)
            +  "   Remote disp  = " + String.format("%5s", panel.remoteDisplay);
    }


   /**
    *  Converts a boolean to on/off.
    */
    private String getOnOff(boolean on)
    {
        return on ? "on" : "off";
    }

}
