package org.lsst.ccs.drivers.bk;

import java.util.Timer;
import java.util.TimerTask;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.PowerSupplyDriver;
import org.lsst.ccs.drivers.scpi.Scpi;

/**
 *  Routines for controlling a B&K Precision model 9130 power supply
 *
 *  @author  Owen Saxton
 */
public class Model9130 extends Scpi implements PowerSupplyDriver {

    public static final int
        DEFAULT_BAUDRATE = 9600,
        MIN_CHANNEL      = 1,
        MAX_CHANNEL      = 3,
        NUM_CHANNELS     = MAX_CHANNEL - MIN_CHANNEL + 1;
    public static final double MAX_VALUE = Double.POSITIVE_INFINITY;

    private final Timer timer = new Timer("Timer_9130", true);
    private final long[]
        onDelay = new long[NUM_CHANNELS],
        offDelay = new long[NUM_CHANNELS];
    private final SetOutput[] outTask = new SetOutput[NUM_CHANNELS];


    /**
     *  Timer task to implement the delayed setting of the output state.
     */
    class SetOutput extends TimerTask {

        private final int chan;
        private final String command;

        SetOutput(String command, int chan)
        {
            this.command = command;
            this.chan = chan;
        }
    
        @Override
        public void run()
        {
            try {
                writeChan(command, chan);
            }
            catch (DriverException e) {
            }
            outTask[chan - MIN_CHANNEL] = null;
        }
    }


    /**
     *  Constructor.
     */
    public Model9130() {
        super(Option.NO_NET);   // Disallow network connections
    }


    /**
     *  Opens a connection.
     *
     *  @param  connType  The enumerated connection type: FTDI or SERIAL
     *  @param  ident     The USB ID (FTDI) or port name (SERIAL)
     *  @param  baudRate  The baud rate, or 0 for the default (9600)
     *  @param  commParm  The communications parameters
     *  @throws  DriverException
     */
    @Override
    public void open(ConnType connType, String ident, int baudRate, int commParm)
        throws DriverException
    {
        super.open(connType, ident, baudRate == 0 ? DEFAULT_BAUDRATE : baudRate, commParm);
        setTerminator(Terminator.LF);
        try {
            checkIdentification("BK", CHECK_STARTS_WITH, "9130", CHECK_EQUALS);
        }
        catch (DriverException e) {
            closeSilent();
            throw e;
        }
    }


    /**
     *  Opens a connection.
     *
     *  @param  connType  The enumerated connection type: FTDI or SERIAL
     *  @param  ident     The USB ID (FTDI) or port name (SERIAL)
     *  @throws  DriverException
     */
    public void open(ConnType connType, String ident) throws DriverException
    {
        open(connType, ident, 0);
    }


    /**
     *  Opens a connection.
     *
     *  @param  connType  The connection type: FTDI or SERIAL
     *  @param  ident     The USB ID (FTDI) or port name (SERIAL)
     *  @throws  DriverException
     */
    @Deprecated
    public void open(int connType, String ident) throws DriverException
    {
        open(connType, ident, 0);
    }


    /**
     *  Turns output on or off for a channel.
     *
     *  @param  on    Turn on if true, off if false
     *  @param  chan  The channel number
     *  @throws  DriverException
     */
    @Override
    public void setOutput(boolean on, int chan) throws DriverException
    {
        checkChannel(chan);
        int index = chan - MIN_CHANNEL;
        long delay = on ? onDelay[index] : offDelay[index];
        String command = "OUTP " + (on ? "1" : "0");
        SetOutput task = outTask[index];
        if (task != null) {
            task.cancel();
            outTask[index] = null;
        }
        if (delay == 0) {
            writeChan(command, chan);
        }
        else {
            outTask[index] = new SetOutput(command, chan);
            timer.schedule(outTask[index], delay);
        }
    }


    /**
     *  Turns output on or off for all channels.
     *
     *  @param  on    Three-element array of on actions
     *  @throws  DriverException
     */
    public void setOutput(boolean[] on) throws DriverException
    {
        if (on.length != NUM_CHANNELS) {
            throw new DriverException("Invalid value array size");
        }
        boolean delay = false;
        for (int j = 0; j < NUM_CHANNELS; j++) {
            if ((on[j] ? onDelay[j] : offDelay[j]) != 0) {
                delay = true;
            }
        }
        if (!delay) {
            StringBuilder str = new StringBuilder();
            for (int j = 0; j < on.length; j++) {
                if (j != 0) {
                    str.append(",");
                }
                str.append(on[j] ? "1" : "0");
            }
            writeCommand("APP:OUT " + str);
        }
        else {
            for (int j = 0; j < NUM_CHANNELS; j++) {
                setOutput(on[j], MIN_CHANNEL + j);
            }
        }
    }


    /**
     *  Gets the output state of a channel.
     *
     *  @param  chan  The channel number
     *  @return  The output state
     *  @throws  DriverException
     */
    @Override
    public boolean getOutput(int chan) throws DriverException
    {
        return readIntegerChan("OUTP?", chan) != 0;
    }


    /**
     *  Gets the output state of all channels.
     *
     *  @return  A three-element array of output states
     *  @throws  DriverException
     */
    public boolean[] getOutput() throws DriverException
    {
        int[] iState = readIntegerArray("APP:OUT?");
        boolean[] state = new boolean[iState.length];
        for (int j = 0; j < state.length; j++) {
            state[j] = iState[j] != 0;
        }

        return state;
    }


    /**
     *  Sets the power-on delay for a channel.
     *
     *  @param  time  The delay (sec)
     *  @param  chan  The channel number
     *  @throws  DriverException
     */
    @Override
    public void setOnDelay(double time, int chan) throws DriverException
    {
        checkChannel(chan);
        onDelay[chan - MIN_CHANNEL] = (long)(1000 * time);
    }


    /**
     *  Sets the power-off delay for a channel.
     *
     *  @param  time  The delay (sec)
     *  @param  chan  The channel number
     *  @throws  DriverException
     */
    @Override
    public void setOffDelay(double time, int chan) throws DriverException
    {
        checkChannel(chan);
        offDelay[chan - MIN_CHANNEL] = (long)(1000 * time);
    }


    /**
     *  Sets the voltage for a channel.
     *
     *  @param  value  The value to set
     *  @param  chan   The channel number
     *  @throws  DriverException
     */
    @Override
    public void setVoltage(double value, int chan) throws DriverException
    {
        writeChan("VOLT " + (Double.isFinite(value) ? value : "MAX"), chan);
    }


    /**
     *  Sets the voltage for all channels.
     *
     *  @param  values  The three-element array of values to set
     *  @throws  DriverException
     */
    public void setVoltage(double[] values) throws DriverException
    {
        writeCommand("APP:VOLT " + makeString(values));
    }


    /**
     *  Gets the set voltage for a channel.
     *
     *  @param  chan   The channel number
     *  @return  The voltage
     *  @throws  DriverException
     */
    @Override
    public double getVoltage(int chan) throws DriverException
    {
        return readDoubleChan("VOLT?", chan);
    }


    /**
     *  Gets the set voltages for all channels.
     *
     *  @return  A three-element array of voltages
     *  @throws  DriverException
     */
    public double[] getVoltage() throws DriverException
    {
        return readDoubleArray("APP:VOLT?");
    }


    /**
     *  Gets the maximum voltage for a channel.
     *
     *  @param  chan   The channel number
     *  @return  The maximum voltage
     *  @throws  DriverException
     */
    public double getMaximumVoltage(int chan) throws DriverException
    {
        return readDoubleChan("VOLT? MAX", chan);
    }


    /**
     *  Gets the maximum voltages for all channels.
     *
     *  @return  A three-element array of voltages
     *  @throws  DriverException
     */
    public double[] getMaximumVoltage() throws DriverException
    {
        return readDoubleArray("APP:VOLT? MAX");
    }


    /**
     *  Reads the voltage for a channel.
     *
     *  @param  chan  The channel number
     *  @return  The voltage
     *  @throws  DriverException
     */
    @Override
    public double readVoltage(int chan) throws DriverException
    {
        return readDoubleChan("MEAS:VOLT?", chan);
    }


    /**
     *  Reads the voltage for all channels.
     *
     *  @return  A three-element array of voltages
     *  @throws  DriverException
     */
    public double[] readVoltage() throws DriverException
    {
        return readDoubleArray("MEAS:VOLT:ALL?");
    }


    /**
     *  Sets the current for a channel.
     *
     *  @param  value  The value to set
     *  @param  chan   The channel number
     *  @throws  DriverException
     */
    @Override
    public void setCurrent(double value, int chan) throws DriverException
    {
        writeChan("CURR " + (Double.isFinite(value) ? value : "MAX"), chan);
    }


    /**
     *  Sets the current for all channels.
     *
     *  @param  values  The three-element array of values to set
     *  @throws  DriverException
     */
    public void setCurrent(double[] values) throws DriverException
    {
        writeCommand("APP:CURR " + makeString(values));
    }


    /**
     *  Gets the set current for a channel.
     *
     *  @param  chan   The channel number
     *  @return  The current
     *  @throws  DriverException
     */
    @Override
    public double getCurrent(int chan) throws DriverException
    {
        return readDoubleChan("CURR?", chan);
    }


    /**
     *  Gets the set current for all channels.
     *
     *  @return  The three-element array of currents
     *  @throws  DriverException
     */
    public double[] getCurrent() throws DriverException
    {
        return readDoubleArray("APP:CURR?");
    }


    /**
     *  Gets the maximum current for a channel.
     *
     *  @param  chan   The channel number
     *  @return  The maximum current
     *  @throws  DriverException
     */
    public double getMaximumCurrent(int chan) throws DriverException
    {
        return readDoubleChan("CURR? MAX", chan);
    }


    /**
     *  Gets the maximum current for all channels.
     *
     *  @return  The three-element array of currents
     *  @throws  DriverException
     */
    public double[] getMaximumCurrent() throws DriverException
    {
        return readDoubleArray("APP:CURR? MAX");
    }


    /**
     *  Reads the current for a channel.
     *
     *  @param  chan  The channel number
     *  @return  The current
     *  @throws  DriverException
     */
    @Override
    public double readCurrent(int chan) throws DriverException
    {
        return readDoubleChan("MEAS:CURR?", chan);
    }


    /**
     *  Reads the current for all channels.
     *
     *  @return  The three-element array of currents
     *  @throws  DriverException
     */
    public double[] readCurrent() throws DriverException
    {
        return readDoubleArray("MEAS:CURR:ALL?");
    }


    /**
     *  Sets the voltage limit for a channel.
     *
     *  @param  value  The value to set
     *  @param  chan   The channel number
     *  @throws  DriverException
     */
    public void setVoltageLimit(double value, int chan) throws DriverException
    {
        writeChan("VOLT:PROT " + (Double.isFinite(value) ? value : "MAX"), chan);
    }


    /**
     *  Sets the voltage limit for all channels.
     *
     *  @param  values  The three-element array of values to set
     *  @throws  DriverException
     */
    public void setVoltageLimit(double[] values) throws DriverException
    {
        writeCommand("APP:PROT " + makeString(values));
    }


    /**
     *  Gets the voltage limit for a channel.
     *
     *  @param  chan  The channel number
     *  @return  The voltage
     *  @throws  DriverException
     */
    public double getVoltageLimit(int chan) throws DriverException
    {
        return readDoubleChan("VOLT:PROT?", chan);
    }


    /**
     *  Gets the voltage limit for all channels.
     *
     *  @return  The three-element array of voltages
     *  @throws  DriverException
     */
    public double[] getVoltageLimit() throws DriverException
    {
        return readDoubleArray("APP:PROT?");
    }


    /**
     *  Locks or unlocks the front panel.
     *
     *  @param  lock  True to lock the panel; false to unlock it
     *  @throws  DriverException
     */
    public void lockPanel(boolean lock) throws DriverException
    {
        writeCommand(lock ? "SYST:RWL" : "SYST:LOC");
    }


    /**
     *  Checks a channel for validity.
     */
    private void checkChannel(int chan) throws DriverException
    {
        if (chan < MIN_CHANNEL || chan > MAX_CHANNEL) {
            throw new DriverException("Invalid channel number");
        }
    }


    /**
     *  Writes a channel select command.
     */
    private void writeSelect(int chan) throws DriverException
    {
        checkChannel(chan);
        writeCommand("INST:NSEL " + chan);
    }


    /**
     *  Writes a command to a channel.
     */
    private synchronized void writeChan(String instr, int chan)
        throws DriverException
    {
        writeSelect(chan);
        writeCommand(instr);
    }


    /**
     *  Writes a command to a channel and reads the double result.
     */
    private synchronized double readDoubleChan(String instr, int chan)
        throws DriverException
    {
        writeSelect(chan);
        return readDouble(instr);
    }


    /**
     *  Writes a command to a channel and reads the integer result.
     */
    private synchronized int readIntegerChan(String instr, int chan)
        throws DriverException
    {
        writeSelect(chan);
        return readInteger(instr);
    }


    /**
     *  Makes a comma-separated string from an array of doubles.
     */
    private String makeString(double[] values) throws DriverException
    {
        if (values.length != MAX_CHANNEL) {
            throw new DriverException("Invalid value array size");
        }
        StringBuilder str = new StringBuilder();
        for (int j = 0; j < values.length; j++) {
            if (j != 0) {
                str.append(",");
            }
            str.append(Double.isFinite(values[j]) ? values[j] : "MAX");
        }

        return str.toString();
    }
}
