package org.lsst.ccs.drivers.bk;

import java.util.HashMap;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import org.lsst.ccs.drivers.ascii.Ascii;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.PowerSupplyDriver;

/**
 *************************************************************************
 *
 *  Routines for controlling a B&K Precision model 1696/7/8 power supply.
 *
 *  @author  Owen Saxton
 *
 *************************************************************************
 */
public class Model1696 extends Ascii implements PowerSupplyDriver {

    public static final int
        DEFAULT_BAUDRATE = 9600;

    private final Timer timer = new Timer("Timer_1696", true);
    private long onDelay = 0, offDelay = 0;
    private SetOutput outTask;

   /**
    *  Inner class to hold front panel data.
    */
    public static class Panel {
        public boolean
            timerDisplay,
            colonDisplay,
            vConstDisplay,
            vSetDisplay,
            iConstDisplay,
            iSetDisplay,
            programDisplay,
            settingDisplay,
            lockDisplay,
            unlockDisplay,
            faultDisplay,
            onDisplay,
            offDisplay,
            remoteDisplay;
        public double
            readVoltage,
            readCurrent,
            readPower,
            setVoltage,
            setCurrent;
        public int
            minutes,
            seconds,
            program;
    }

   /**
    *  Timer task to implement the delayed setting of the output state.
    */
    class SetOutput extends TimerTask {

        private final String command;

        SetOutput(String command)
        {
            this.command = command;
        }
    
        @Override
        public void run()
        {
            try {
                writeCmnd(command);
            }
            catch (DriverException e) {
            }
            outTask = null;
        }
    }


   /**
    *  Constructor.
    */
    public Model1696() {
        super(Option.NO_NET);   // Disallow network connections
    }


   /**
    *  Opens a connection.
    *
    *  @param  connType  The enumerated connection type: FTDI or SERIAL
    *
    *  @param  ident     The USB ID (FTDI) or port name (SERIAL)
    *
    *  @param  baudRate  The baud rate
    *
    *  @param  commParm  The communications parameters
    *
    *  @throws  DriverException
    */
    @Override
    public void open(ConnType connType, String ident, int baudRate,
                     int commParm) throws DriverException
    {
        super.open(connType, ident, baudRate, commParm);
        setTerminator("\r");
    }


   /**
    *  Opens a connection.
    *
    *  @param  connType  The enumerated connection type: FTDI or SERIAL
    *
    *  @param  ident     The USB ID (FTDI) or port name (SERIAL)
    *
    *  @throws  DriverException
    */
    public void open(ConnType connType, String ident) throws DriverException
    {
        open(connType, ident, DEFAULT_BAUDRATE);
    }


   /**
    *  Opens a connection.
    *
    *  @param  connType  The connection type: FTDI or SERIAL
    *
    *  @param  ident     The USB ID (FTDI) or port name (SERIAL)
    *
    *  @throws  DriverException
    */
    @Deprecated
    public void open(int connType, String ident) throws DriverException
    {
        open(connType, ident, DEFAULT_BAUDRATE);
    }


   /**
    *  Turns output on or off for a channel.
    *
    *  @param  on    Turn on if true, off if false
    *
    *  @param  chan  The channel number
    *
    *  @throws  DriverException
    */
    @Override
    public void setOutput(boolean on, int chan) throws DriverException
    {
        long delay = on ? onDelay : offDelay;
        String command = "SOUT" + chanString(chan) + (on ? "0" : "1");
        SetOutput task = outTask;
        if (task != null) {
            task.cancel();
            outTask = null;
        }
        if (delay == 0) {
            writeCmnd(command);
        }
        else {
            outTask = new SetOutput(command);
            timer.schedule(outTask, delay);
        }
    }


   /**
    *  Gets the output state of a channel.
    *
    *  @param  chan  The channel number
    *
    *  @return  The output state
    *
    *  @throws  DriverException
    */
    @Override
    public boolean getOutput(int chan) throws DriverException
    {
        return getPanel(chan).onDisplay;
    }


   /**
    *  Sets the power-on delay for a channel.
    *
    *  @param  time  The delay (sec)
    *
    *  @param  chan  The channel number
    *
    *  @throws  DriverException
    */
    @Override
    public void setOnDelay(double time, int chan) throws DriverException
    {
        onDelay = (long)(1000 * time);
    }


   /**
    *  Sets the power-off delay for a channel.
    *
    *  @param  time  The delay (sec)
    *
    *  @param  chan  The channel number
    *
    *  @throws  DriverException
    */
    @Override
    public void setOffDelay(double time, int chan) throws DriverException
    {
        offDelay = (long)(1000 * time);
    }


   /**
    *  Sets the voltage for a channel.
    *
    *  @param  value  The value to set
    *
    *  @param  chan   The channel number
    *
    *  @throws  DriverException
    */
    @Override
    public void setVoltage(double value, int chan) throws DriverException
    {
        writeCmnd("VOLT" + chanString(chan) + valString(value, 10.0));
    }


   /**
    *  Gets the set voltage for a channel.
    *
    *  @param  chan   The channel number
    *
    *  @return  The voltage
    *
    *  @throws  DriverException
    */
    @Override
    public double getVoltage(int chan) throws DriverException
    {
        String reply = readCmnd("GETS" + chanString(chan));
        return Double.valueOf(reply.substring(0, 3)) / 10.0;
    }


   /**
    *  Reads the voltage for a channel.
    *
    *  @param  chan   The channel number
    *
    *  @return  The voltage
    *
    *  @throws  DriverException
    */
    @Override
    public double readVoltage(int chan) throws DriverException
    {
        String reply = readCmnd("GETD" + chanString(chan));
        return Double.valueOf(reply.substring(0, 4)) / 100.0;
    }


   /**
    *  Sets the current for a channel.
    *
    *  @param  value  The value to set
    *
    *  @param  chan   The channel number
    *
    *  @throws  DriverException
    */
    @Override
    public void setCurrent(double value, int chan) throws DriverException
    {
        writeCmnd("CURR" + chanString(chan) + valString(value, 100.0));
    }


   /**
    *  Gets the set current for a channel.
    *
    *  @param  chan   The channel number
    *
    *  @return  The current
    *
    *  @throws  DriverException
    */
    @Override
    public double getCurrent(int chan) throws DriverException
    {
        String reply = readCmnd("GETS" + chanString(chan));
        return Double.valueOf(reply.substring(3, 6)) / 100.0;
    }


   /**
    *  Reads the current for a channel.
    *
    *  @param  chan   The channel number
    *
    *  @return  The current
    *
    *  @throws  DriverException
    */
    @Override
    public double readCurrent(int chan) throws DriverException
    {
        String reply = readCmnd("GETD" + chanString(chan));
        return Double.valueOf(reply.substring(4, 8)) / 1000.0;
    }


   /**
    *  Sets the voltage limit for a channel.
    *
    *  @param  value  The value to set
    *
    *  @param  chan   The channel number
    *
    *  @throws  DriverException
    */
    public void setVoltageLimit(double value, int chan) throws DriverException
    {
        writeCmnd("SOVP" + chanString(chan) + valString(value, 10.0));
    }


   /**
    *  Gets the voltage limit for a channel.
    *
    *  @param  chan  The channel number
    *
    *  @return  The voltage
    *
    *  @throws  DriverException
    */
    public double getVoltageLimit(int chan) throws DriverException
    {
        return Double.valueOf(readCmnd("GOVP" + chanString(chan))) / 10.0;
    }


   /**
    *  Locks or unlocks the front panel.
    *
    *  @param  lock   True to lock the panel; false to unlock it
    *
    *  @param  chan   The channel number
    *
    *  @throws  DriverException
    */
    public void lockPanel(boolean lock, int chan) throws DriverException
    {
        writeCmnd((lock ? "SESS" : "ENDS") + chanString(chan));
    }


   /**
    *  Gets the state of the front panel lock.
    *
    *  @param  chan  The channel number
    *
    *  @return  Whether the front panel is locked
    *
    *  @throws  DriverException
    */
    public boolean isPanelLocked(int chan) throws DriverException
    {
        return getPanel(chan).lockDisplay;
    }


   /**
    *  Gets the front panel indicators.
    *
    *  @param  chan  The channel number
    *
    *  @return  The front panel data
    *
    *  @throws  DriverException
    */
    public Panel getPanel(int chan) throws DriverException
    {
        String reply = readCmnd("GPAL" + chanString(chan));
        Panel data = new Panel();
        data.timerDisplay = getLcdBoolean(reply, 36);
        data.colonDisplay = getLcdBoolean(reply, 37);
        data.vConstDisplay = getLcdBoolean(reply, 46);
        data.vSetDisplay = getLcdBoolean(reply, 47);
        data.iConstDisplay = getLcdBoolean(reply, 55);
        data.iSetDisplay = getLcdBoolean(reply, 56);
        data.programDisplay = getLcdBoolean(reply, 60);
        data.settingDisplay = getLcdBoolean(reply, 62);
        data.lockDisplay = getLcdBoolean(reply, 63);
        data.unlockDisplay = getLcdBoolean(reply, 64);
        data.faultDisplay = getLcdBoolean(reply, 65);
        data.onDisplay = getLcdBoolean(reply, 66);
        data.offDisplay = getLcdBoolean(reply, 67);
        data.remoteDisplay = getLcdBoolean(reply, 68);
        data.readVoltage = getLcdValue(reply, 1, 4);
        data.readCurrent = getLcdValue(reply, 10, 4);
        data.readPower = getLcdValue(reply, 19, 4);
        data.setVoltage = getLcdValue(reply, 40, 3);
        data.setCurrent = getLcdValue(reply, 49, 3);
        data.minutes = (int)getLcdValue(reply, 28, 2);
        data.seconds = (int)getLcdValue(reply, 30, 2);
        data.program = (int)getLcdValue(reply, 58, 1);
        return data;
    }


   /**
    *  Writes a command and waits for acknowledgment.
    */
    private synchronized void writeCmnd(String instr) throws DriverException
    {
        write(instr);
        String ack = read();
        if (!ack.equals("OK")) {
            throw new DriverException("Unrecognized command response (" + ack
                                        + ")");
        }
    }


   /**
    *  Writes a command and waits for reply and acknowledgment.
    */
    private synchronized String readCmnd(String instr) throws DriverException
    {
        write(instr);
        String reply = read();
        String ack = read();
        if (!ack.equals("OK")) {
            throw new DriverException("Unrecognized command response (" + ack
                                        + ")");
        }
        return reply;
    }


   /**
    *  Converts a channel number to a string.
    */
    private String chanString(int chan) throws DriverException
    {
        String str = String.format("%02d", chan);
        if (str.length() != 2) {
            throw new DriverException("Invalid channel number (" + chan + ")");
        }
        return str;
    }


   /**
    *  Converts a voltage or current value to a string.
    */
    private String valString(double value, double mult) throws DriverException
    {
        int ival = (int)(mult * value + 0.5);
        String str = String.format("%03d", ival);
        if (str.length() != 3) {
            throw new DriverException("Invalid value (" + value + ")");
        }
        return str;
    }


   /**
    *  Gets a boolean from a character in a string.
    */
    private boolean getLcdBoolean(String data, int index)
    {
        return data.substring(index - 1, index).equals("0");
    }


   /**
    *  Gets an LCD display value from encoded data.
    */
    private final static Map<String, Integer> lcdMap = new HashMap<>();
    static {
        lcdMap.put("00", 0);
        lcdMap.put("3?", 0);
        lcdMap.put("06", 1);
        lcdMap.put("5;", 2);
        lcdMap.put("4?", 3);
        lcdMap.put("66", 4);
        lcdMap.put("6=", 5);
        lcdMap.put("7=", 6);
        lcdMap.put("07", 7);
        lcdMap.put("7?", 8);
        lcdMap.put("6?", 9);
        lcdMap.put("80", 10);
        lcdMap.put(";?", 10);
        lcdMap.put("86", 11);
        lcdMap.put("=;", 12);
        lcdMap.put("<?", 13);
        lcdMap.put(">6", 14);
        lcdMap.put(">=", 15);
        lcdMap.put("?=", 16);
        lcdMap.put("87", 17);
        lcdMap.put("??", 18);
        lcdMap.put(">?", 19);
    }

    private double getLcdValue(String data, int index, int leng)
        throws DriverException
    {
        double value = 0, div = 1;
        int point = leng;
        index--;
        for (int j = 0; j < leng; j++, index += 2) {
            Integer digit = lcdMap.get(data.substring(index, index + 2));
            if (digit == null) {
                throw new DriverException("Invalid LCD string");
            }
            if (digit > 9) {
                digit -= 10;
                point = j + 1;
            }
            value = 10 * value + digit;
        }
        for (int j = 0; j < leng - point; j++) {
            div *= 10;
        }

        return value / div;
    }

}
