package org.lsst.ccs.drivers.bk;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.scpi.Scpi;
import org.lsst.ccs.drivers.scpi.TestScpi;

/**
 ******************************************************************************
 ** 
 **  Program to test the B&K Precision model 9130 device driver
 ** 
 **  @author Owen Saxton
 ** 
 ******************************************************************************
 */
public class Test9130 extends TestScpi {

    private final Model9130 bk;
    private int chanNum = 1;

    public enum OnOff {
        OFF, ON;
    }


   /**
    *************************************************************************** 
    ** 
    **  Constructor
    ** 
    *************************************************************************** 
    */
    public Test9130()
    {
        super(new Model9130());
        bk = (Model9130)scpi;
    }


   /**
    *************************************************************************** 
    ** 
    **  Opens connection to a device.
    ** 
    **  @param  connType  The type of connection: ftdi or serial
    **
    **  @param  ident     The identification: serial no. (ftdi) or port name
    **                    (serial)
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="open", description="Open connection to device")
    public void open(@Argument(name="type", description="Connection type")
                     Scpi.ConnType connType,
                     @Argument(name="ident", description="Identification")
                     String ident) throws DriverException
    {
        bk.open(connType, ident);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the channel.
    ** 
    **  @param  chan  The channel number
    **
    *************************************************************************** 
    */
    @Command(name="setchannel", description="Set the current channel")
    public void setChannel(@Argument(name="channel", description="Channel number")
                           int chan)
    {
        chanNum = chan;
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the set channel.
    ** 
    **  @return  The result string
    **
    *************************************************************************** 
    */
    @Command(name="showchannel", description="Show the set channel")
    public String showChannel()
    {
        return "Channel = " + chanNum;
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the voltage.
    ** 
    **  @param  value  The value to set
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setvoltage", description="Set the voltage")
    public void setVoltage(@Argument(name="value", description="Voltage to set")
                           double value) throws DriverException
    {
        bk.setVoltage(value, chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the voltage.
    ** 
    **  @param  value  The value to set
    **
    **  @param  chan   The channel number
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setvoltage", description="Set the voltage")
    public void setVoltage(@Argument(name="value", description="Voltage to set")
                           double value,
                           @Argument(name="channel",
                                     description="Channel number")
                           int chan) throws DriverException
    {
        bk.setVoltage(value, chan);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets all the voltages.
    ** 
    **  @param  values  The values to set
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setvoltages", description="Set all the voltages")
    public void setVoltages(@Argument(name="value",
                                      description="Voltages to set")
                            double... values) throws DriverException
    {
        bk.setVoltage(values);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the voltage to the maximum.
    ** 
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setmaxvoltage", description="Set the voltage to maximum")
    public void setMaxVoltage() throws DriverException
    {
        bk.setVoltage(Model9130.MAX_VALUE, chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the voltage to the maximum.
    ** 
    **  @param  chan   The channel number
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setmaxvoltage", description="Set the voltage to maximum")
    public void setMaxVoltage(@Argument(name="channel",
                                        description="Channel number")
                              int chan) throws DriverException
    {
        bk.setVoltage(Model9130.MAX_VALUE, chan);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the maximum voltage.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showmaxvoltage", description="Show the maximum voltage")
    public String showMaxVoltage() throws DriverException
    {
        return "Maximum voltage = " + bk.getMaximumVoltage(chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the maximum voltage.
    ** 
    **  @param  chan  The channel number
    **
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showmaxvoltage", description="Show the maximum voltage")
    public String showMaxVoltage(@Argument(name="channel",
                                           description="Channel number")
                                 int chan) throws DriverException
    {
        return "Maximum voltage = " + bk.getMaximumVoltage(chan);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows all the maximum voltages.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showmaxvoltages", description="Show all the maximum voltages")
    public String showMaxVoltages() throws DriverException
    {
        return "Maximum voltages = " + getString(bk.getMaximumVoltage());
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the set voltage.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showvoltage", description="Show the set voltage")
    public String showVoltage() throws DriverException
    {
        return "Voltage = " + bk.getVoltage(chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the set voltage.
    ** 
    **  @param  chan  The channel number
    **
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showvoltage", description="Show the set voltage")
    public String showVoltage(@Argument(name="channel",
                                        description="Channel number")
                              int chan) throws DriverException
    {
        return "Voltage = " + bk.getVoltage(chan);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows all the set voltage(s).
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showvoltages", description="Show all the set voltages")
    public String showVoltages() throws DriverException
    {
        return "Voltages = " + getString(bk.getVoltage());
    }


   /**
    *************************************************************************** 
    ** 
    **  Reads the actual voltage.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="readvoltage", description="Read the actual voltage")
    public String readVoltage() throws DriverException
    {
        return "Voltage = " + bk.readVoltage(chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Reads the actual voltage.
    ** 
    **  @param  chan  The channel number
    **
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="readvoltage", description="Read the actual voltage")
    public String readVoltage(@Argument(name="channel",
                                        description="Channel number")
                              int chan) throws DriverException
    {
        return "Voltage = " + bk.readVoltage(chan);
    }


   /**
    *************************************************************************** 
    ** 
    **  Reads all the actual voltages.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="readvoltages", description="Read all the actual voltages")
    public String readVoltages() throws DriverException
    {
        return "Voltages = " + getString(bk.readVoltage());
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the soft voltage limit.
    ** 
    **  @param  value  The value to set
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setvoltlim", description="Set the soft voltage limit")
    public void setVoltageLimit(@Argument(name="value",
                                          description="Voltage limit to set")
                                double value) throws DriverException
    {
        bk.setVoltageLimit(value, chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the soft voltage limit.
    ** 
    **  @param  value  The value to set
    **
    **  @param  chan   The channel number
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setvoltlim", description="Set the soft voltage limit")
    public void setVoltageLimit(@Argument(name="value",
                                          description="Voltage limit to set")
                                double value,
                                @Argument(name="channel",
                                          description="Channel number")
                                int chan) throws DriverException
    {
        bk.setVoltageLimit(value, chan);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets all the soft voltage limits.
    ** 
    **  @param  values  The values to set
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setvoltlims", description="Set all the voltage limits")
    public void setVoltlimits(@Argument(name="values",
                                        description="Voltage limits to set")
                              double... values) throws DriverException
    {
        bk.setVoltageLimit(values);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the soft voltage limit.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showvoltlim", description="Show the soft voltage limit")
    public String showVoltageLimit() throws DriverException
    {
        return "Voltage limit = " + bk.getVoltageLimit(chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the soft voltage limit.
    ** 
    **  @param  chan  The channel number
    **
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showvoltlim", description="Show the soft voltage limit")
    public String showVoltageLimit(@Argument(name="channel",
                                             description="Channel number")
                                   int chan) throws DriverException
    {
        return "Voltage limit = " + bk.getVoltageLimit(chan);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows all the soft voltage limits.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showvoltlims",
             description="Show all the soft voltage limits")
    public String showVoltageLimits() throws DriverException
    {
        return "Voltage limits = " + getString(bk.getVoltageLimit());
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the current.
    ** 
    **  @param  value  The value to set
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setcurrent", description="Set the current")
    public void setCurrent(@Argument(name="value", description="Current to set")
                           double value) throws DriverException
    {
        bk.setCurrent(value, chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the current.
    ** 
    **  @param  value  The value to set
    **
    **  @param  chan   The channel number
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setcurrent", description="Set the current")
    public void setCurrent(@Argument(name="value", description="Current to set")
                           double value,
                           @Argument(name="channel",
                                     description="Channel number")
                           int chan) throws DriverException
    {
        bk.setCurrent(value, chan);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets all the currents.
    ** 
    **  @param  values  The values to set
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setcurrents", description="Set all the currents")
    public void setCurrents(@Argument(name="values",
                                      description="Currents to set")
                            double... values) throws DriverException
    {
        bk.setCurrent(values);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the current to the maximum.
    ** 
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setmaxcurrent", description="Set the current to maximum")
    public void setMaxCurrent() throws DriverException
    {
        bk.setVoltage(Model9130.MAX_VALUE, chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the current to the maximum.
    ** 
    **  @param  chan   The channel number
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setmaxcurrent", description="Set the current to maximum")
    public void setMaxCurrent(@Argument(name="channel",
                                        description="Channel number")
                              int chan) throws DriverException
    {
        bk.setVoltage(Model9130.MAX_VALUE, chan);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the maximum current.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showmaxcurrent", description="Show the set current")
    public String showMaxCurrent() throws DriverException
    {
        return "Maximum current = " + bk.getMaximumCurrent(chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the maximum current.
    ** 
    **  @param  chan  The channel number
    **
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showmaxcurrent", description="Show the set current")
    public String showMaxCurrent(@Argument(name="channel",
                                           description="Channel number")
                                 int chan) throws DriverException
    {
        return "Maximum current = " + bk.getMaximumCurrent(chan);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows all the maximum currents.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showmaxcurrents", description="Show all the maximum currents")
    public String showMaxCurrents() throws DriverException
    {
        return "Maximum currents = " + getString(bk.getMaximumCurrent());
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the set current.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showcurrent", description="Show the set current")
    public String showCurrent() throws DriverException
    {
        return "Current = " + bk.getCurrent(chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the set current.
    ** 
    **  @param  chan  The channel number
    **
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showcurrent", description="Show the set current")
    public String showCurrent(@Argument(name="channel",
                                        description="Channel number")
                              int chan) throws DriverException
    {
        return "Current = " + bk.getCurrent(chan);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows all the set currents.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showcurrents", description="Show the set current")
    public String showCurrents() throws DriverException
    {
        return "Currents = " + getString(bk.getCurrent());
    }


   /**
    *************************************************************************** 
    ** 
    **  Reads the actual current.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="readcurrent", description="Read the actual current")
    public String readCurrent() throws DriverException
    {
        return "Current = " + bk.readCurrent(chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Reads the actual current.
    ** 
    **  @param  chan  The channel number
    **
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="readcurrent", description="Read the actual current")
    public String readCurrent(@Argument(name="channel",
                                        description="Channel number")
                              int chan) throws DriverException
    {
        return "Current = " + bk.readCurrent(chan);
    }


   /**
    *************************************************************************** 
    ** 
    **  Reads all the actual currents.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="readcurrents", description="Read the actual current")
    public String readCurrents() throws DriverException
    {
        return "Current = " + getString(bk.readCurrent());
    }


   /**
    *************************************************************************** 
    ** 
    **  Turns the output on or off.
    ** 
    **  @param  state  The state to set: on or off
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setoutput", description="Turn output on or off")
    public void setOutput(@Argument(name="state",
                                    description="Output state: on or off")
                          OnOff state) throws DriverException
    {
        bk.setOutput(state == OnOff.ON, chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Turns the output on or off.
    ** 
    **  @param  state  The state to set: on or off
    **
    **  @param  chan   The channel number
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setoutput", description="Turn output on or off")
    public void setOutput(@Argument(name="state",
                                    description="Output state: on or off")
                          OnOff state,
                          @Argument(name="channel",
                                    description="Channel number")
                          int chan) throws DriverException
    {
        bk.setOutput(state == OnOff.ON, chan);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets all the output states on or off.
    ** 
    **  @param  states  The states to set: on or off for each
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="setoutputs", description="Set all the output states")
    public void setOutputs(@Argument(name="states",
                                     description="States (on or off) to set")
                           OnOff... states) throws DriverException
    {
        boolean[] bstates = new boolean[states.length];
        for (int j = 0; j < bstates.length; j++) {
            bstates[j] = states[j] == OnOff.ON;
        }
        bk.setOutput(bstates);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the output state.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showoutput", description="Show the output state")
    public String showOutput() throws DriverException
    {
        return "Output state = " + getOnOff(bk.getOutput(chanNum));
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the output state.
    ** 
    **  @param  chan  The channel number
    **
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showoutput", description="Show the output state")
    public String showOutput(@Argument(name="channel",
                                       description="Channel number")
                             int chan) throws DriverException
    {
        return "Output state = " + getOnOff(bk.getOutput(chan));
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows all the output states.
    ** 
    **  @return  The result string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="showoutputs", description="Show the output state")
    public String showOutputs() throws DriverException
    {
        return "Output states = " + getString(bk.getOutput());
    }


   /**
    *************************************************************************** 
    ** 
    **  Locks the front panel.
    ** 
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="lock", description="Lock the front panel")
    public void lock() throws DriverException
    {
        bk.lockPanel(true);
    }


   /**
    *************************************************************************** 
    ** 
    **  Unlocks the front panel.
    ** 
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="unlock", description="Unlock the front panel")
    public void unlock() throws DriverException
    {
        bk.lockPanel(false);
    }


   /**
    *************************************************************************** 
    ** 
    **  Converts an array of numbers to a string.
    ** 
    *************************************************************************** 
    */
    private StringBuilder getString(double[] values)
    {
        StringBuilder text = new StringBuilder();
        text.append(values[0]);
        for (int j = 1; j < values.length; j++) {
            text.append(", ").append(values[j]);
        }

        return text;
    }


   /**
    *************************************************************************** 
    ** 
    **  Converts an array of booleans to a string.
    ** 
    *************************************************************************** 
    */
    private StringBuilder getString(boolean[] values)
    {
        StringBuilder text = new StringBuilder();
        text.append(values[0] ? "on" : "off");
        for (int j = 1; j < values.length; j++) {
            text.append(", ").append(values[j] ? "on" : "off");
        }

        return text;
    }


   /**
    *************************************************************************** 
    ** 
    **  Converts a boolean to on/off.
    ** 
    *************************************************************************** 
    */
    private String getOnOff(boolean on)
    {
        return on ? "on" : "off";
    }

}
