package org.lsst.ccs.drivers.bk;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.ascii.TestAscii;

/**
 *************************************************************************** 
 ** 
 **  Program to test the B&K Precision model 1696 device driver
 ** 
 **  @author Owen Saxton
 ** 
 *************************************************************************** 
 */
public class Test1696 extends TestAscii {

    private Model1696 bk = new Model1696();
    private int chanNum = 1;

    public enum OnOff {
        OFF, ON;
    }


   /**
    *************************************************************************** 
    ** 
    **  Constructor
    ** 
    *************************************************************************** 
    */
    public Test1696()
    {
        super(new Model1696());
        bk = (Model1696)dev;
    }


   /**
    *************************************************************************** 
    ** 
    **  Opens connection to a device.
    ** 
    *************************************************************************** 
    */
    @Command(name="open", description="Open connection to device")
    public void open(@Argument(name="type", description="Connection type")
                     TestAscii.ConnType connType,
                     @Argument(name="ident", description="Identification")
                     String ident) throws DriverException
    {
        bk.open(connType.getValue(), ident);
    }


   /**
    *************************************************************************** 
    ** 
    **  Opens connection to a device.
    ** 
    *************************************************************************** 
    */
    @Override
    @Command(name="open", description="Open connection to device")
    public void open(@Argument(name="type", description="Connection type")
                     TestAscii.ConnType connType,
                     @Argument(name="ident", description="Identification")
                     String ident,
                     @Argument(name="rate", description="Baud rate")
                     int baudRate) throws DriverException
    {
        bk.open(connType.getValue(), ident, baudRate);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the voltage.
    ** 
    *************************************************************************** 
    */
    @Command(name="setvoltage", description="Set the voltage")
    public void setVoltage(@Argument(name="value", description="Voltage to set")
                           double value) throws DriverException
    {
        bk.setVoltage(value, chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the set voltage.
    ** 
    *************************************************************************** 
    */
    @Command(name="showvoltage", description="Show the set voltage")
    public String getVoltage() throws DriverException
    {
        return "Voltage = " + bk.getVoltage(chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Reads the actual voltage.
    ** 
    *************************************************************************** 
    */
    @Command(name="readvoltage", description="Read the actual voltage")
    public String readVoltage() throws DriverException
    {
        return "Voltage = " + bk.readVoltage(chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the soft voltage limit.
    ** 
    *************************************************************************** 
    */
    @Command(name="setvoltlim", description="Set the soft voltage limit")
    public void setVoltageLimit(@Argument(name="value",
                                          description="Voltage limit to set")
                                double value) throws DriverException
    {
        bk.setVoltageLimit(value, chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the soft voltage limit.
    ** 
    *************************************************************************** 
    */
    @Command(name="showvoltlim", description="Show the soft voltage limit")
    public String getVoltageLimit() throws DriverException
    {
        return "Voltage limit = " + bk.getVoltageLimit(chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the current.
    ** 
    *************************************************************************** 
    */
    @Command(name="setcurrent", description="Set the current")
    public void setCurrent(@Argument(name="value", description="Current to set")
                           double value) throws DriverException
    {
        bk.setCurrent(value, chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the set current.
    ** 
    *************************************************************************** 
    */
    @Command(name="showcurrent", description="Show the set current")
    public String getCurrent() throws DriverException
    {
        return "Current = " + bk.getCurrent(chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Reads the actual current.
    ** 
    *************************************************************************** 
    */
    @Command(name="readcurrent", description="Read the actual current")
    public String readCurrent() throws DriverException
    {
        return "Current = " + bk.readCurrent(chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Turns the output on or off.
    ** 
    *************************************************************************** 
    */
    @Command(name="setoutput", description="Turn output on or off")
    public void setOutput(@Argument(name="state",
                                    description="Output state: on or off")
                          OnOff state) throws DriverException
    {
        bk.setOutput(state == OnOff.ON, chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the output state.
    ** 
    *************************************************************************** 
    */
    @Command(name="showoutput", description="Show the output state")
    public String showOutput() throws DriverException
    {
        return "Output state = " + getOnOff(bk.getOutput(chanNum));
    }


   /**
    *************************************************************************** 
    ** 
    **  Locks the front panel.
    ** 
    *************************************************************************** 
    */
    @Command(name="lock", description="Lock the front panel")
    public void lock() throws DriverException
    {
        bk.lockPanel(true, chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Unlocks the front panel.
    ** 
    *************************************************************************** 
    */
    @Command(name="unlock", description="Unlock the front panel")
    public void unlock() throws DriverException
    {
        bk.lockPanel(false, chanNum);
    }


   /**
    *************************************************************************** 
    ** 
    **  Shows the front panel state.
    ** 
    *************************************************************************** 
    */
    @Command(name="showpanel", description="Show the front panel state")
    public String showPanel() throws DriverException
    {
        Model1696.Panel panel = bk.getPanel(chanNum);
        return   " Read voltage = " + String.format("%5s", panel.readVoltage)
            +  "   Read current = " + String.format("%5s", panel.readCurrent)
            +  "\n Read power   = " + String.format("%5s", panel.readPower)
            +  "   Set voltage  = " + String.format("%5s", panel.setVoltage)
            +  "\n Set current  = " + String.format("%5s", panel.setCurrent)
            +  "   Timer mins   = " + String.format("%5s", panel.minutes)
            +  "\n Timer secs   = " + String.format("%5s", panel.seconds)
            +  "   Program no.  = " + String.format("%5s", panel.program)
            +  "\n Timer disp   = " + String.format("%5s", panel.timerDisplay)
            +  "   Colon disp   = " + String.format("%5s", panel.colonDisplay)
            +  "\n Vconst disp  = " + String.format("%5s", panel.vConstDisplay)
            +  "   Vset disp    = " + String.format("%5s", panel.vSetDisplay)
            +  "\n Iconst disp  = " + String.format("%5s", panel.iConstDisplay)
            +  "   Iset disp    = " + String.format("%5s", panel.iSetDisplay)
            +  "\n Program disp = " + String.format("%5s", panel.programDisplay)
            +  "   Setting disp = " + String.format("%5s", panel.settingDisplay)
            +  "\n Lock disp    = " + String.format("%5s", panel.lockDisplay)
            +  "   Unlock disp  = " + String.format("%5s", panel.unlockDisplay)
            +  "\n Fault disp   = " + String.format("%5s", panel.faultDisplay)
            +  "   Out on disp  = " + String.format("%5s", panel.onDisplay)
            +  "\n Out off disp = " + String.format("%5s", panel.offDisplay)
            +  "   Remote disp  = " + String.format("%5s", panel.remoteDisplay);
    }


   /**
    *************************************************************************** 
    ** 
    **  Converts a boolean to on/off.
    ** 
    *************************************************************************** 
    */
    private String getOnOff(boolean on)
    {
        return on ? "on" : "off";
    }

}
