package org.lsst.ccs.drivers.auxelex;

import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Routines for controlling the high-voltage (24V & 48V) SLAC PDUs 
 *
 *  @author  Owen Saxton
 */
public class PduHV extends PduCommon {

    /**
     *  Constants and data.
     */
    private static final double
        SHUNT_040 = 0.0040;

    private static final int
        REG_LAMBDA_BASE = REG_DEVICE_BASE + 0x3400,
        REG_TEMP_BASE = REG_DEVICE_BASE + 0x3800,
        REG_BOARD_TEMP = REG_TEMP_BASE,
        REG_FPGA_TEMP = REG_TEMP_BASE + 1;

    private final LambdaPS lambda;


    /**
     *  Constructor.
     *
     *  @param  numChans   The number of channels
     *  @param  mainModel  The model of the main (Lambda) power supply
     */
    public PduHV(int numChans, int mainModel)
    {
        double[] currScales = new double[numChans];
        for (int j = 0; j < numChans; j++) {
            currScales[j] = 1.0 / SHUNT_040;
        }
        setCurrentScales(currScales);
        lambda = new LambdaPS(this, REG_LAMBDA_BASE, mainModel);
    }


    /**
     *  Reads the board temperature
     *
     *  @return The board temperature (Celsius)
     *  @throws  DriverException
     */
    public double readBoardTemperature() throws DriverException
    {
        return readReg(REG_BOARD_TEMP);
    }


    /**
     *  Reads the FPGA temperature
     *
     *  @return The FPGA temperature (Celsius)
     *  @throws  DriverException
     */
    public double readFpgaTemperature() throws DriverException
    {
        return readReg(REG_FPGA_TEMP);
    }


    /**
     *  Gets the main PS model type.
     *
     *  @return  The model type
     *  @throws  DriverException
     */
    public int getMainModel() throws DriverException
    {
        return lambda.getModel();
    }


    /**
     *  Gets the main PS serial number.
     *
     *  @return  The serial number
     *  @throws  DriverException
     */
    public String getMainSerialNo() throws DriverException
    {
        return lambda.getSerialNo();
    }


    /**
     *  Gets the main PS firmware version.
     *
     *  @return  The firmware revision
     *  @throws  DriverException
     */
    public String getMainFwVersion() throws DriverException
    {
        return lambda.getFwVersion();
    }


    /**
     *  Gets the main PS product version.
     *
     *  @return  The product revision
     *  @throws  DriverException
     */
    public String getMainPrVersion() throws DriverException
    {
        return lambda.getPrVersion();
    }


    /**
     *  Gets the main PS part number.
     *
     *  @return  The unit part number
     *  @throws  DriverException
     */
    public String getMainPartNo() throws DriverException
    {
        return lambda.getPartNo();
    }


    /**
     *  Gets the main PS manufacturing date.
     *
     *  @return  The manufacturing date
     *  @throws  DriverException
     */
    public String getMainManuDate() throws DriverException
    {
        return lambda.getManuDate();
    }


    /**
     *  Gets the main PS manufacturing location.
     *
     *  @return  The manufacturing location
     *  @throws  DriverException
     */
    public String getMainManuLocn() throws DriverException
    {
        return lambda.getManuLocn();
    }


    /**
     *  Reads the main PS status.
     *
     *  @return  The contents of the status register
     *  @throws  DriverException
     */
    public int readMainStatus() throws DriverException
    {
        return lambda.readStatus();
    }


    /**
     *  Reads the main PS voltage.
     *
     *  @return  The measured voltage
     *  @throws  DriverException
     */
    public double readMainVoltage() throws DriverException
    {
        return lambda.readVoltage();
    }


    /**
     *  Reads the main PS current.
     *
     *  @return  The measured current
     *  @throws  DriverException
     */
    public double readMainCurrent() throws DriverException
    {
        return lambda.readCurrent();
    }


    /**
     *  Reads the main PS baseplate temperature.
     *
     *  @return  The measured temperature (C)
     *  @throws  DriverException
     */
    public double readMainTemperature() throws DriverException
    {
        return lambda.readTemperature();
    }

}
