package org.lsst.ccs.drivers.auxelex;

import java.io.EOFException;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.net.InetAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Server for reading BFR currents
 * 
 *  @author saxton
 */
public class BfrServer extends Thread {

    /*
     *  Class for containing incoming requests.
     */
    public static class Request implements Serializable {

        int address;   // Modbus address
        int sensor;    // First sensor number
        int count;     // Sensor count

        public Request(int address, int sensor, int count) {
            this.address = address;
            this.sensor = sensor;
            this.count = count;
        }

        private static final long serialVersionUID = 1104232295274389306L;
    }

    /*
     *  Class for containing outgoing replies.
     */
    public static class Reply implements Serializable {

        double[] values;  // Read values
        DriverException ex;  // Exception

        Reply (double[] values) {
            this.values = values;
            this.ex = null;
        }

        Reply (DriverException ex) {
            this.values = null;
            this.ex = ex;
        }

        private static final long serialVersionUID = -3339762080987937239L;
    }

    /*
     *  Thread for processing incoming requests and sending the reply.
     */
    class RequestThread extends Thread {

        private ObjectInputStream inStream;
        private ObjectOutputStream outStream;
        private volatile boolean stopping = false;

        public RequestThread(Socket socket) {
            try {
                outStream = new ObjectOutputStream(socket.getOutputStream());
                outStream.flush();
                inStream = new ObjectInputStream(socket.getInputStream());
            }
            catch (IOException e) {
                LOG.log(Level.SEVERE, "Error creating BFR server request streams", e);
            }
        }

        @Override
        public void run() {
            while (true) {
                try {
                    Request rqst = (Request)inStream.readObject();
                    Object reply;
                    try {
                        double[] values = bfr.readSensorDirect(rqst.address, rqst.sensor, rqst.count);
                        reply = new Reply(values);
                    }
                    catch (DriverException e) {
                        reply = new Reply(e);
                    }
                    try {
                        outStream.writeObject(reply);
                    }
                    catch (IOException e) {
                        LOG.log(Level.SEVERE, "Error writing BFR server reply", e);
                        break;
                    }
                }
                catch (IOException | ClassNotFoundException e) {
                    if (!(e instanceof EOFException || stopping)) {
                        LOG.log(Level.SEVERE, "Error reading BFR server request", e);
                    }
                    break;
                }
            }
            rqstThreads.remove(this);
        }

        public void stopRun() {
            try {
                stopping = true;
                inStream.close();
            }
            catch (IOException e) {}
        }

    }

    /*
     *  Constants & data.
     */
    public static final int
        PORT = 8000,
        NUM_PORTS = 4;
    public static final InetAddress IP = InetAddress.getLoopbackAddress();

    private static final Logger LOG = Logger.getLogger(BfrServer.class.getName());
    private static Map<Integer, BfrServer> servers = Collections.synchronizedMap(new HashMap<>());
    private Bfr bfr;
    private ServerSocket socket;
    private volatile boolean stopping = false;
    private Set<RequestThread> rqstThreads = Collections.synchronizedSet(new HashSet<>());


    /**
     *  Constructor.
     * 
     *  @param  bfr  The BFR being served
     *  @throws  DriverException 
     */
    public BfrServer(Bfr bfr) throws DriverException {
        this.bfr = bfr;
        try {
            socket = new ServerSocket(getServerPort(bfr.getNode()), NUM_PORTS, IP);
        }
        catch (IOException e) {
            throw new DriverException("Error creating BFR server socket", e);
        }
    }


    /**
     *  Starts a new server thread.
     * 
     *  @param  bfr  The BFR being served
     *  @throws DriverException 
     */
    public static void startServer(Bfr bfr) throws DriverException {
        if (!servers.keySet().contains(bfr.getNode())) {
            BfrServer server = new BfrServer(bfr);
            server.setDaemon(true);
            server.start();
            servers.put(bfr.getNode(), server);
        }
    }


    /**
     *  Stops a server thread.
     * 
     *  @param node 
     */
    public static void stopServer(int node) {
        BfrServer server = servers.get(node);
        if (server != null) {
            server.stopRun();
            servers.remove(node);
        }
    }


    /**
     *  Provides the port number for a node.
     * 
     *  @param  node  The node number of the BFR
     *  @return  The port number to use
     */
    public static int getServerPort(int node) {
        return PORT + node;
    }


    /**
     *  Runs the server thread.
     */
    @Override
    public void run() {
        while (true) {
            try {
                Socket rqstSocket = socket.accept();
                RequestThread thread = new RequestThread(rqstSocket);
                rqstThreads.add(thread);
                thread.setDaemon(true);
                thread.start();
            }
            catch (IOException e) {
                if (!stopping) {
                    LOG.log(Level.SEVERE, "Error accepting incoming BFR server connection", e);
                }
                break;
            }
        }
        for (RequestThread rqstThread : rqstThreads) {
            rqstThread.stopRun();
        }
    }


    /**
     *  Stops the run.
     */
    public void stopRun() {
        try {
            stopping = true;
            socket.close();
        }
        catch (IOException e) {}
    }

}
