package org.lsst.ccs.drivers.auxelex;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.SocketTimeoutException;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Supplier;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.utilities.conv.Convert;

/**
 *  Routines for implementing the SLAC register protocol
 *
 *  @author  Owen Saxton
 */
public class Srp {

    private static final Logger LOG = Logger.getLogger(Srp.class.getName());

    /**
     *  Constants and data.
     */
    public static final int
        DEFAULT_PORT = 8192,
        PORT_0 = DEFAULT_PORT,
        PORT_1 = DEFAULT_PORT + 1,
        PORT_2 = DEFAULT_PORT + 2,
        PORT_3 = DEFAULT_PORT + 3;

    public enum BoardType {UNKNOWN, SIMULATED, REB_PS_PROTO, REB_PS_UPDATE, REB_PS_PROD_SR,
                           REB_PS_PROD_CR, PDU_5V, PDU_24V_DIRTY, PDU_24V_CLEAN, PDU_48V, BFR,
                           ION_PUMP, HEATER}

    private static final int
        MAX_REGS = 64,
        MAX_REGS_3 = 1024,

        READ_WARN = 500,      // Default read time warning
        READ_TIMEOUT = 1000,  // Default read timeout

        SRP_PKT_LENG    = 20,
        SRP_OFF_HEADER  = 4,
        SRP_OFF_OC      = 8,
        SRP_OFF_ADDR    = 8,
        SRP_OFF_DATA    = 12,
        SRP_OFF_FOOTER  = 16,

        SRP3_PKT_LENG    = 20,
        SRP3_OFF_HEADER  = 0,
        SRP3_OFF_VERSION = 0,
        SRP3_OFF_OPCODE  = 1,
        SRP3_OFF_TIMEOUT = 3,
        SRP3_OFF_TID     = 4,
        SRP3_OFF_ADDR_LO = 8,
        SRP3_OFF_ADDR_HI = 12,
        SRP3_OFF_SIZE    = 16,
        SRP3_OFF_DATA    = 20,
        SRP3_OFF_FOOTER  = 20,
        SRP3_OPC_NP_READ  = 0,
        SRP3_OPC_NP_WRITE = 1,
        SRP3_OPC_P_WRITE  = 2;

    private static final byte
        SRP_OC_READ  = 0x00,
        SRP_OC_WRITE = 0x40;

    private static final int
        REG_BOARD_ID = 0x70101,
        ID_REB_PS_SCIENCE_A = 0x00,
        ID_REB_PS_SCIENCE_B = 0x02,
        ID_REB_PS_CORNER_A = 0x01,
        ID_REB_PS_CORNER_B = 0x03,
        ID_PDU_5V_A = 0x20,
        ID_PDU_5V_B = 0x21,
        ID_PDU_5V_C = 0x22,
        ID_PDU_24V_DIRTY_A = 0x11,
        ID_PDU_24V_DIRTY_B = 0x14,
        ID_PDU_24V_DIRTY_C = 0x17,
        ID_PDU_24V_CLEAN_A = 0x12,
        ID_PDU_24V_CLEAN_B = 0x15,
        ID_PDU_24V_CLEAN_C = 0x18,
        ID_PDU_48V_A = 0x10,
        ID_PDU_48V_B = 0x13,
        ID_PDU_48V_C = 0x16,
        ID_ION_PUMP_A = 0x30,
        ID_ION_PUMP_B = 0x31,
        ID_HEATER_A = 0x40,
        ID_BFR_A = 0x50,
        REG_BUILD_STAMP = 0x70200,
        LEN_BUILD_STAMP = 64;

    private static final Map<Integer, BoardType> boardIdMap = new HashMap<>();
    static {
        boardIdMap.put(ID_REB_PS_SCIENCE_A, BoardType.REB_PS_PROD_SR);
        boardIdMap.put(ID_REB_PS_SCIENCE_B, BoardType.REB_PS_PROD_SR);
        boardIdMap.put(ID_REB_PS_CORNER_A, BoardType.REB_PS_PROD_CR);
        boardIdMap.put(ID_REB_PS_CORNER_B, BoardType.REB_PS_PROD_CR);
        boardIdMap.put(ID_PDU_5V_A, BoardType.PDU_5V);
        boardIdMap.put(ID_PDU_5V_B, BoardType.PDU_5V);
        boardIdMap.put(ID_PDU_5V_C, BoardType.PDU_5V);
        boardIdMap.put(ID_PDU_24V_CLEAN_A, BoardType.PDU_24V_CLEAN);
        boardIdMap.put(ID_PDU_24V_CLEAN_B, BoardType.PDU_24V_CLEAN);
        boardIdMap.put(ID_PDU_24V_CLEAN_C, BoardType.PDU_24V_CLEAN);
        boardIdMap.put(ID_PDU_24V_DIRTY_A, BoardType.PDU_24V_DIRTY);
        boardIdMap.put(ID_PDU_24V_DIRTY_B, BoardType.PDU_24V_DIRTY);
        boardIdMap.put(ID_PDU_24V_DIRTY_C, BoardType.PDU_24V_DIRTY);
        boardIdMap.put(ID_PDU_48V_A, BoardType.PDU_48V);
        boardIdMap.put(ID_PDU_48V_B, BoardType.PDU_48V);
        boardIdMap.put(ID_PDU_48V_C, BoardType.PDU_48V);
        boardIdMap.put(ID_ION_PUMP_A, BoardType.ION_PUMP);
        boardIdMap.put(ID_ION_PUMP_B, BoardType.ION_PUMP);
        boardIdMap.put(ID_HEATER_A, BoardType.HEATER);
        boardIdMap.put(ID_BFR_A, BoardType.BFR);
    }

    protected BoardType boardType;
    private DatagramSocket sock;
    private int seqno, nSeqErr, nTimeout;
    private int srpVersion = 3, maxRegs = MAX_REGS_3;
    private byte[]
        inBuff = new byte[SRP3_PKT_LENG + 4 * (maxRegs - 1)],
        outBuff = new byte[SRP3_PKT_LENG + 4 * (maxRegs - 1)];
    private DatagramPacket
        inPkt = new DatagramPacket(inBuff, inBuff.length),
        outPkt = new DatagramPacket(outBuff, outBuff.length);
    private byte[] ipAddr;
    private boolean simulated, debug;
    private final Map<Integer, Integer> simRegMap = new HashMap<>();
    private Collection<BoardType> validBoardTypes;
    private int readTimeout = READ_TIMEOUT;
    private int readWarn = READ_WARN;
    private String logName;


    /**
     * Create an SRP and give a name to be included in log messages and exceptions
     * @param logName 
     */
    public Srp(String logName) {
        this.logName = logName;
    }
    
    /**
     * For backwards compatibility
     */
    public Srp() {
        this(null);
    }
    
    public void setLogName(String logName) {
        this.logName = logName;
    }

    /**
     *  Sets the collection of valid board types.
     *
     *  @param  types  The collection of valid types
     */
    public void setValidBoardTypes(Collection types)
    {
        validBoardTypes = types;
    }


    /**
     *  Sets the SRP protocol version.
     *
     *  @param  version  The protocol version
     */
    public void setSrpVersion(int version)
    {
        if (version == srpVersion) return;
        srpVersion = version;
        maxRegs = srpVersion == 3 ? MAX_REGS_3 : MAX_REGS;
        int pktLeng = srpVersion == 3 ? SRP3_PKT_LENG : SRP_PKT_LENG;
        inBuff = new byte[pktLeng + 4 * (maxRegs - 1)];
        outBuff = new byte[pktLeng + 4 * (maxRegs - 1)];
        inPkt = new DatagramPacket(inBuff, inBuff.length);
        outPkt = new DatagramPacket(outBuff, outBuff.length);
    }


    /**
     *  Opens a connection to a board.
     *
     *  @param  host  The host name or IP address, or null or empty for simulation
     *  @param  port  The port number, or 0 for default
     *  @throws  DriverException
     */
    public synchronized void open(String host, int port) throws DriverException
    {
        if (sock != null) {
            throw new SrpException(logName, "Connection is already open");
        }
        nSeqErr = 0;
        nTimeout = 0;
        try {
            DatagramSocket newSock = new DatagramSocket();
            if (host == null || host.isEmpty()) {
                simulated = true;
                ipAddr = new byte[]{0, 0, 0, 0};
                simInitialize();
            }
            else {
                simulated = false;
                int actPort = port == 0 ? DEFAULT_PORT : port;
                InetAddress inetAddr = InetAddress.getByName(host);
                newSock.connect(inetAddr, actPort);
                newSock.setSoTimeout(readTimeout);
                outPkt.setAddress(inetAddr);
                outPkt.setPort(actPort);
                ipAddr = inetAddr.getAddress();
            }
            sock = newSock;
            try {
                setBoardType();
            }
            catch (DriverException e) {
                close();
                throw e;
            }
        }
        catch (IOException e) {
            throw new SrpException(logName, "IOException during open", e);
        }
    }


    /**
     *  Opens a connection to a board.
     *
     *  @param  host  The IP address, or null or empty for simulation
     *  @throws  DriverException
     */
    public void open(String host) throws DriverException
    {
        open(host, 0);
    }


    /**
     *  Opens a connection to a board.
     *
     *  @param  node  The IP private network node number, or 0 for simulation
     *  @param  port  The port number, or 0 for default
     *  @throws  DriverException
     */
    public void open(int node, int port) throws DriverException
    {
        open(node == 0 ? null : "192.168.1." + node, port);
    }


    /**
     *  Opens a connection to a board.
     *
     *  @param  node  The IP private network node number, or 0 for simulation
     *  @throws  DriverException
     */
    public void open(int node) throws DriverException
    {
        open(node, 0);
    }


    /**
     *  Closes the connection.
     *
     *  This method isn't synchronized in order to allow a connection to be
     *  closed while another thread is waiting for a read to complete.  But
     *  some synchronization does need to be implemented to avoid race
     *  conditions leading to null pointer exceptions..
     *
     *  @throws  DriverException
     */
    public void close() throws DriverException
    {
        checkOpen();
        sock.close();
        sock = null;
    }


    /**
     *  Sets the read timeout.
     * 
     *  @param  timeout  The timeout (ms)
     */
    public void setReadTimeout(int timeout) // throws DriverException
    {
        readTimeout = timeout;
        if (sock != null) {
            try {
                sock.setSoTimeout(timeout);
            }
            catch (IOException e) {
                LOG.log(Level.WARNING, "Error setting socket timeout: {0}", e.getMessage());
            }
        }
    }


    /**
     *  Gets the read timeout
     * 
     *  @return  The read timeout (ms) 
     */
    public int getReadTimeout()
    {
        return readTimeout;
    }


    /**
     *  Sets the read warning time.
     * 
     *  @param  time  The time (ms)
     */
    public void setReadWarning(int time) // throws DriverException
    {
        readWarn = time;
    }


    /**
     *  Gets the read warning time.
     * 
     *  @return  The time (ms) 
     */
    public int getReadWarning()
    {
        return readWarn;
    }


    /**
     *  Sets the board type.
     *
     *  @throws  DriverException
     */
    private void setBoardType() throws DriverException
    {
        if (simulated) {
            boardType = BoardType.SIMULATED;
        }
        else {
            int type = readReg(REG_BOARD_ID);
            boardType = type != 0 ? boardIdMap.get(type & 0xff) : null;
            //System.out.println("Board type from ID = " + boardType);
            if (boardType == null) {
                boardType = BoardType.UNKNOWN;
            }
        }
        if (validBoardTypes != null && !validBoardTypes.contains(boardType)) {
            throw new SrpException(logName, "Invalid board type: " + boardType);
        }
    }


    /**
     *  Gets the board type.
     *
     *  @return  The board type
     */
    public BoardType getBoardType()
    {
        return boardType;
    }


    /**
     *  Gets the build stamp
     *
     *  @return  The build stamp
     *  @throws  DriverException
     */
    public String getBuildStamp() throws DriverException
    {
        int data[] = readRegs(REG_BUILD_STAMP, LEN_BUILD_STAMP);
        byte[] bData = new byte[4 * LEN_BUILD_STAMP];
        for (int j = 0; j < data.length; j++) {
            Convert.intToBytes(data[j], bData, 4 * j);
        }
        int len = 0;
        for (; len < bData.length; len++) {
            if (bData[len] == 0) break;
        }
        return new String(bData, 0, len);
    }


    /**
     *  Sets the debug state.
     *
     *  @param  on  The debug on state, true or false
     */
    public void setDebug(Boolean on)
    {
        debug = on;
    }


    /**
     *  Gets the IP address.
     *
     *  @return  The IP address byte array
     */
    public byte[] getIpAddress()
    {
        return ipAddr;
    }


    /**
     *  Gets whether simulated
     *
     *  @return  Whether simulated
     */
    public boolean isSimulated()
    {
        return simulated;
    }


    /**
     *  Writes a register.
     *
     *  @param  addr   The register address
     *  @param  value  The value to write
     *  @throws  DriverException
     */
    public void writeReg(int addr, int value) throws DriverException
    {
        writeRegs(addr, new int[]{value});
    }


    /**
     *  Writes registers.
     *
     *  @param  addr   The first register address
     *  @param  value  The array of values to write.  If it contains more than the allowed
     *                 maximum number (1024 for SRP V3, 64 otherwise) of values, only the
     *                 allowed number are written.
     *  @throws  DriverException
     */
    public synchronized void writeRegs(int addr, int[] value) throws DriverException
    {
        checkOpen();
        int count = Math.min(value.length, maxRegs);
        try {
            if (simulated) {
                simWriteRegs(addr, value, count);
            }
            else if (srpVersion == 3) {
                writeRegs3(addr, value, count);
            }
            else {
                writeRegs0(addr, value, count);
            }
        } catch (SrpException x) {
            x.addDetails(String.format("While writing addr=%x values=%s", addr, Arrays.toString(value)));
            throw x;            
        }
    }


    /**
     *  Writes registers using new (version 3) SRP protocol.
     *
     *  @param  addr   The first register address
     *  @param  value  The array of values to write
     *  @param  count  The number of values to write
     *  @throws  DriverException
     */
    private void writeRegs3(int addr, int[] value, int count) throws DriverException
    {
        Convert.intToBytes(4 * addr, outBuff, SRP3_OFF_ADDR_LO);
        Convert.intToBytes(0, outBuff, SRP3_OFF_ADDR_HI);
        Convert.intToBytes(4 * count - 1, outBuff, SRP3_OFF_SIZE);
        for (int j = 0; j < count; j++) {
            Convert.intToBytes(value[j], outBuff, SRP3_OFF_DATA + 4 * j);
        }
        outBuff[SRP3_OFF_VERSION] = 3;
        outBuff[SRP3_OFF_OPCODE] = SRP3_OPC_NP_WRITE;
        outBuff[SRP3_OFF_TIMEOUT] = 10;
        for (int j = 0; j < 2; j++) {
            try {
                send(SRP3_PKT_LENG + 4 * count);
                receive();
                break;
            }
            catch (DriverTimeoutException e) {
                if (j == 1) {
                    throw e;
                }
            }
        }
    }


    /**
     *  Writes registers using old SRP protocol.
     *
     *  @param  addr   The first register address
     *  @param  value  The array of values to write.
     *  @param  count  The number of registers to write
     *  @throws  DriverException
     */
    private void writeRegs0(int addr, int[] value, int count) throws DriverException
    {
        Convert.intToBytesBE(addr, outBuff, SRP_OFF_ADDR);
        for (int j = 0; j < count; j++) {
            Convert.intToBytesBE(value[j], outBuff, SRP_OFF_DATA + 4 * j);
        }
        outBuff[SRP_OFF_OC] = SRP_OC_WRITE;
        for (int j = 0; j < 2; j++) {
            try {
                send(SRP_PKT_LENG + 4 * (count - 1));
                receive();
                break;
            }
            catch (DriverTimeoutException e) {
                if (j == 1) {
                    throw e;
                }
            }
        }
    }


    /**
     *  Reads a register.
     *
     *  @param  addr  The register address
     *  @return  The read value
     *  @throws  DriverException
     */
    public int readReg(int addr) throws DriverException
    {
        return readRegs(addr, 1)[0];
    }


    /**
     *  Reads registers.
     *
     *  @param  addr   The first register address
     *  @param  count  The number of registers to read.  If greater than the allowed
     *                 maximum number (1024 for SRP V3, 64 otherwise) of registers, only the
     *                 allowed number are read.
     *  @return  The array of read values
     *  @throws  DriverException
     */
    public synchronized int[] readRegs(int addr, int count) throws DriverException
    {
        checkOpen();
        if (count <= 0) {
            return new int[0];
        }
        count = Math.min(count, maxRegs);
        try {
            if (simulated) {
                return simReadRegs(addr, count);
            }
            else if (srpVersion == 3) {
                return readRegs3(addr, count);
            }
            else {
                return readRegs0(addr, count);
            }
        } catch (SrpException x) {
            x.addDetails(String.format("While reading addr=%x count=%d", addr, count));
            throw x;
        }
    }


    /**
     *  Reads registers using new (version 3) SRP protocol.
     *
     *  @param  addr   The first register address
     *  @param  count  The number of registers to read.  If greater than
     *                 MAX_REGS (currently 64), is set to MAX_REGS.
     *  @return  The array of read values
     *  @throws  DriverException
     */
    private int[] readRegs3(int addr, int count) throws DriverException
    {
        outBuff[SRP3_OFF_VERSION] = 3;
        outBuff[SRP3_OFF_OPCODE] = SRP3_OPC_NP_READ;
        outBuff[SRP3_OFF_TIMEOUT] = 10;
        Convert.intToBytes(4 * addr, outBuff, SRP3_OFF_ADDR_LO);
        Convert.intToBytes(0, outBuff, SRP3_OFF_ADDR_HI);
        Convert.intToBytes(4 * count - 1, outBuff, SRP3_OFF_SIZE);
        for (int j = 0; j < 2; j++) {  // Single retry on timeout
            try {
                send(SRP3_PKT_LENG);
                receive();
                break;
            }
            catch (DriverTimeoutException e) {
                if (j == 1) {
                    throw e;
                }
            }
        }
        int nRead = (inPkt.getLength() - SRP3_PKT_LENG) / 4 - 1;
        int[] data = new int[nRead];
        for (int j = 0; j < nRead; j++) {
            data[j] = Convert.bytesToInt(inBuff, SRP3_OFF_DATA + 4 * j);
        }
        return data;
    }


    /**
     *  Reads registers using old SRP protocol.
     *
     *  @param  addr   The first register address
     *  @param  count  The number of registers to read.  If greater than
     *                 MAX_REGS (currently 64), is set to MAX_REGS.
     *  @return  The array of read values
     *  @throws  DriverException
     */
    private int[] readRegs0(int addr, int count) throws DriverException
    {
        Convert.intToBytesBE(addr, outBuff, SRP_OFF_ADDR);
        Convert.intToBytesBE(count - 1, outBuff, SRP_OFF_DATA);
        outBuff[SRP_OFF_OC] = SRP_OC_READ;
        for (int j = 0; j < 2; j++) {   // Single retry on timeout
            try {
                send(SRP_PKT_LENG);
                receive();
                break;
            }
            catch (DriverTimeoutException e) {
                if (j == 1) {
                    throw e;
                }
            }
        }
        int nRead = (inPkt.getLength() - SRP_PKT_LENG) / 4 + 1;
        int[] data = new int[nRead];
        for (int j = 0; j < nRead; j++) {
            data[j] = Convert.bytesToIntBE(inBuff, SRP_OFF_DATA + 4 * j);
        }
        return data;
    }


    /**
     *  Updates a register.
     *
     *  @param  addr   The register address
     *  @param  mask   The mask of bits to be updated
     *  @param  value  The value to use for updating
     *  @return  Previous register value
     *  @throws  DriverException
     */
    public synchronized int updateReg(int addr, int mask, int value) throws DriverException
    {
        int prevValue = readReg(addr);
        writeReg(addr, (prevValue & ~mask) | (value & mask));
        return prevValue;
    }


    /**
     *  Checks whether a connection is open.
     *
     *  @throws  DriverException
     */
    private void checkOpen() throws DriverException
    {
        if (sock == null) {
            throw new DriverException("Connection is not open");
        }
    }


    /**
     *  Sends the "out" packet.
     *
     *  @throws  DriverException
     */
    private void send(int leng) throws DriverException
    {
        seqno += 4;
        Convert.intToBytes(seqno, outBuff, srpVersion == 3 ? SRP3_OFF_TID : SRP_OFF_HEADER);
        if (srpVersion != 3) {
            Convert.intToBytes(0, outBuff, leng - SRP_PKT_LENG + SRP_OFF_FOOTER);
        }
        showData("Sent:", leng, outBuff);
        outPkt.setLength(leng);
        try {
            sock.send(outPkt);
        }
        catch (IOException e) {
            throw new SrpException(logName, "IOException during send", e);
        }
    }


    /**
     *  Receives the "in" packet.
     *
     *  @throws  DriverException
     */
    private void receive() throws DriverException
    {
        long start = System.currentTimeMillis();
        while (true) {
            inPkt.setLength(inBuff.length);
            try {
                sock.receive(inPkt);
                int rSeqno = Convert.bytesToInt(inBuff, srpVersion == 3 ? SRP3_OFF_TID : SRP_OFF_HEADER);
                if (seqno == rSeqno) break;
                else {
                   LOG.log(Level.WARNING, () -> String.format("%sDiscarded unexpected packet seqno %x", formatLogName(), rSeqno)); 
                }
                nSeqErr++;
            }
            catch (SocketTimeoutException e) {
                nTimeout++;
                throw new SrpException.SrpTimeoutException(logName, "Timeout during receive", e);
            }
            catch (IOException e) {
                throw new SrpException(logName, "IOException during receive", e);
            }
        }
        long timeToRead = System.currentTimeMillis() - start;
        if (timeToRead > readWarn) {
           LOG.log(Level.WARNING, () -> String.format("%sRead of seqno %x took unexpectedly long time %dms", formatLogName(), seqno, timeToRead)); 
        }
        int leng = inPkt.getLength();
        showData("Rcvd:", leng, inBuff);
        if (srpVersion == 3) {
            int status = Convert.bytesToInt(inBuff, leng - 4);
            if (status != 0) throw SrpException.create3(logName, status);
        }
        else {
            int status = Convert.bytesToIntBE(inBuff, leng + SRP_OFF_FOOTER - SRP_PKT_LENG);
            if (status != 0) throw SrpException.create0(logName, status);
        }
    }
    
    private String formatLogName() {
        return formatLogName(logName);
    }
    
    static String formatLogName(String logName) {
        return logName == null ? "" : logName+": ";
    }

    /**
     *  Display data buffer.
     */
    private void showData(String title, int length, byte[] data)
    {
        final Supplier<String> messageBuilder = () -> {
            StringBuilder builder = new StringBuilder();
            @SuppressWarnings("ReplaceAllDot")
            String blanks = "\n" + title.replaceAll(".", " ");
            for (int j = 0; j < length; j++) {
                if ((j % 32) == 0) {
                    if (j == 0) {
                        builder.append(title);
                    }
                    else {
                        builder.append(blanks);
                    }
                }
                if ((j % 4) == 0) {
                    builder.append(" ");
                }
                builder.append(String.format("%02x", data[j] & 0xff));
            }
            return builder.toString();
        };
        if (debug) {
            System.out.println(messageBuilder.get());
        } else {
            LOG.log(Level.FINEST, () -> formatLogName()+messageBuilder.get());
        }
    }


    /**
     *  Gets the sequence error count.
     *
     *  @return  The number of sequence errors
     */
    public int getNumSeqErr()
    {
        return nSeqErr;
    }


    /**
     *  Gets the socket timeout count.
     *
     *  @return  The number of socket timeouts
     */
    public int getNumTimeout()
    {
        return nTimeout;
    }


    /**
     *  Initializes the simulation.
     */
    protected void simInitialize()
    {
        clearSimRegMap();
    }

    
    /**
     *  Writes simulated registers.
     *
     *  @param  addr   The first register address
     *  @param  value  The array of values to write.
     *  @param  count  The number of values to write.
     */
    protected void simWriteRegs(int addr, int[] value, int count)
    {
        for (int j = 0; j < count; j++, addr++) {
            putSimRegMap(addr, value[j]);
        }
    }


    /**
     *  Reads simulated registers.
     *
     *  @param  addr   The first register address
     *  @param  count  The number of registers to read.
     *  @return  The array of read values
     */
    protected int[] simReadRegs(int addr, int count)
    {
        int[] data = new int[count];
        for (int j = 0; j < count; j++) {
            data[j] = getSimRegMap(addr + j);
        }
        return data;
    }


    /**
     *  Clears the simulated register map.
     */
    protected final void clearSimRegMap()
    {
        simRegMap.clear();
    }

    
    /**
     *  Puts to the simulated register map.
     *
     *  @param  addr   The register address
     *  @param  value  The array of values to write.
     */
    protected final void putSimRegMap(int addr, int value)
    {
        simRegMap.put(addr, value);
    }


    /**
     *  Gets from the simulated register map.
     *
     *  @param  addr   The register address
     *  @return  The array of read values
     */
    protected final int getSimRegMap(int addr)
    {
        Integer value = simRegMap.get(addr);
        return value == null ? 0 : value;
    }
        
}
