package org.lsst.ccs.drivers.auxelex;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;

/**
 * Exceptions thrown by the SRP driver
 * @author tonyj
 */
public class SrpException extends DriverException {
    
    private static final int SRP_STS_TIMEOUT = 0x02;
    private static final int SRP_STS_ERROR = 0x01;
    private static final int SRP3_STS_MEMORY = 0x00ff;
    private static final int SRP3_STS_TIMEOUT = 0x0100;
    private static final int SRP3_STS_EOFE = 0x0200;
    private static final int SRP3_STS_FRAMING = 0x0400;
    private static final int SRP3_STS_VERSION = 0x0800;
    private static final int SRP3_STS_REQUEST = 0x1000;
    private static final long serialVersionUID = 2455904340900648746L;
    private final int status;
    private String extraMessage;

    SrpException(String logName, String message, int status) {
        super(Srp.formatLogName(logName)+message + " (status=" + Integer.toHexString(status) + ")");
        this.status = status;
    }

    SrpException(String logName, String message) {
        super(Srp.formatLogName(logName)+message);
        this.status = 0;
    }

    SrpException(String logName, String message, Throwable cause) {
        super(logName == null ? "" : logName+": "+message, cause);
        this.status = 0;
    }

    @Override
    public String getMessage() {
        String message = super.getMessage();
        if (extraMessage != null) {
            return message + " ("+extraMessage+")";
        } else {
            return message;
        }
    }
    
    public int getStatus() {
        return status;
    }

    void addDetails(String extraMessage) {
        this.extraMessage = extraMessage;
    }

    public static class SrpTimeoutException extends DriverTimeoutException {

        private static final long serialVersionUID = -2082721527482465140L;
        SrpTimeoutException(String logName, String message, Throwable cause) {
            super(Srp.formatLogName(logName)+message, cause);
        }
    }
    
    public static class SrpInvalidRegisterAddressException extends SrpException {

        private static final long serialVersionUID = 9025518445392473744L;

        SrpInvalidRegisterAddressException(String logName, int status) {
            super(logName, "Invalid Register Address", status);
        }
    }

    public static class SrpRegisterAccessTimeoutException extends SrpException {

        private static final long serialVersionUID = -7542864357902100945L;

        SrpRegisterAccessTimeoutException(String logName, int status) {
            super(logName, "Register access timeout (FATAL???!!!)", status);
        }
    }

    public static class SrpEndOfFrameException extends SrpException {

        private static final long serialVersionUID = 608402485101612599L;

        SrpEndOfFrameException(String logName, int status) {
            super(logName, "End of frame with error", status);
        }
    }

    public static class SrpFramingException extends SrpException {

        private static final long serialVersionUID = 6991600583438830264L;

        SrpFramingException(String logName, int status) {
            super(logName, "Framing error", status);
        }
    }

    public static class SrpVersionMismatchException extends SrpException {

        private static final long serialVersionUID = 5306785243067574205L;

        SrpVersionMismatchException(String logName, int status) {
            super(logName, "Version mismatch", status);
        }
    }

    public static class SrpInvalidRequestException extends SrpException {

        private static final long serialVersionUID = -8437270228409744211L;

        SrpInvalidRequestException(String logName, int status) {
            super(logName, "Invalid request", status);
        }
    }

    static SrpException create3(String logName, int status) {
        if ((status & SRP3_STS_MEMORY) != 0) {
            return new SrpInvalidRegisterAddressException(logName, status);
        }
        if ((status & SRP3_STS_TIMEOUT) != 0) {
            return new SrpRegisterAccessTimeoutException(logName, status);
        }
        if ((status & SRP3_STS_EOFE) != 0) {
            return new SrpEndOfFrameException(logName, status);
        }
        if ((status & SRP3_STS_FRAMING) != 0) {
            return new SrpFramingException(logName, status);
        }
        if ((status & SRP3_STS_VERSION) != 0) {
            return new SrpVersionMismatchException(logName, status);
        }
        if ((status & SRP3_STS_REQUEST) != 0) {
            return new SrpInvalidRequestException(logName, status);
        }
        return new SrpException(logName, "Unknown error", status);
    }

    static SrpException create0(String logName, int status) {
        if ((status & SRP_STS_TIMEOUT) != 0) {
            return new SrpRegisterAccessTimeoutException(logName, status);
        }
        return new SrpException(logName, "Unknown error", status);
    }
    
}
