package org.lsst.ccs.drivers.auxelex;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Common code for testing the PDU drivers
 * 
 *  @author Owen Saxton
 */
public class TestIonPump extends TestSrp {

    /**
     *  Fields
     */
    protected final IonPump ipc;

    /**
     *  Constructor
     */
    public TestIonPump()
    {
        super(new IonPump());
        ipc = (IonPump)srp;
    }

    /*
     *  Commands
     */
    @Command(description="Set a channel's voltage")
    public void setVoltage(@Argument(description="Channel number")
                           int chan,
                           @Argument(description="Voltage")
                           double volts) throws DriverException
    {
        ipc.setVoltage(chan, volts);
    }

    @Command(description="Set a named channel's voltage")
    public void setVoltageN(@Argument(description="Channel name")
                            String chan,
                            @Argument(description="Voltage")
                            double volts) throws DriverException
    {
        ipc.setVoltage(chan, volts);
    }

    @Command(description="Set a channel's current")
    public void setCurrent(@Argument(description="Channel number")
                           int chan,
                           @Argument(description="Current")
                           double amps) throws DriverException
    {
        ipc.setCurrent(chan, amps);
    }

    @Command(description="Set a named channel's current")
    public void setCurrentN(@Argument(description="Channel name")
                            String chan,
                            @Argument(description="Current")
                            double amps) throws DriverException
    {
        ipc.setCurrent(chan, amps);
    }

    @Command(description="Set a channel's power")
    public void setPower(@Argument(description="Channel number")
                         int chan,
                         @Argument(description="Power")
                         double watts) throws DriverException
    {
        ipc.setPower(chan, watts);
    }

    @Command(description="Set a named channel's power")
    public void setPowerN(@Argument(description="Channel name")
                          String chan,
                          @Argument(description="Power")
                          double watts) throws DriverException
    {
        ipc.setPower(chan, watts);
    }

    @Command(description="Turn on a channel")
    public void powerOn(@Argument(description="Channel number")
                        int chan) throws DriverException
    {
        ipc.powerOn(chan);
    }

    @Command(description="Turn on a named channel")
    public void powerOnN(@Argument(description="Channel name")
                         String chan) throws DriverException
    {
        ipc.powerOn(chan);
    }

    @Command(description="Turn off a channel")
    public void powerOff(@Argument(description="Channel number")
                         int chan) throws DriverException
    {
        ipc.powerOff(chan);
    }

    @Command(description="Turn off a named channel")
    public void powerOffN(@Argument(description="Channel name")
                          String chan) throws DriverException
    {
        ipc.powerOff(chan);
    }

    @Command(description="Show the powered state of a channel")
    public boolean isPowered(@Argument(description="Channel number")
                             int chan) throws DriverException
    {
        return ipc.isPowered(chan);
    }

    @Command(description="Show the state of a named channel")
    public boolean isPoweredN(@Argument(description="Channel name")
                              String chan) throws DriverException
    {
        return ipc.isPowered(chan);
    }

    @Command(description="Show the overall powered status")
    public String showPoweredStatus() throws DriverException
    {
        return String.format("0x%03x", ipc.getPoweredStatus());
    }

    @Command(description="Show the overall voltage limit status")
    public String showVlimitStatus() throws DriverException
    {
        return String.format("0x%03x", ipc.getVlimitStatus());
    }

    @Command(description="Show the overall current limit status")
    public String showiLimitStatus() throws DriverException
    {
        return String.format("0x%03x", ipc.getIlimitStatus());
    }

    @Command(description="Show the overall power limit status")
    public String showPlimitStatus() throws DriverException
    {
        return String.format("0x%03x", ipc.getPlimitStatus());
    }

    @Command(description="Read a channel's voltage")
    public String readVoltage(@Argument(description="Channel number")
                              int chan) throws DriverException
    {
        return String.format("%.1f", ipc.readVoltage(chan));
    }

    @Command(description="Read a named channel's voltage")
    public String readVoltageN(@Argument(description="Channel name")
                               String chan) throws DriverException
    {
        return String.format("%.1f", ipc.readVoltage(chan));
    }

    @Command(description="Read a channel's current")
    public String readCurrent(@Argument(description="Channel number")
                              int chan) throws DriverException
    {
        return String.format("%.3f", ipc.readCurrent(chan));
    }

    @Command(description="Read a named channel's current")
    public String readCurrentN(@Argument(description="Channel name")
                               String chan) throws DriverException
    {
        return String.format("%.3f", ipc.readCurrent(chan));
    }

    @Command(description="Read a channel's power")
    public String readPower(@Argument(description="Channel number")
                            int chan) throws DriverException
    {
        return String.format("%.2f", ipc.readPower(chan));
    }

    @Command(description="Read a named channel's power")
    public String readPowerN(@Argument(description="Channel name")
                             String chan) throws DriverException
    {
        return String.format("%.2f", ipc.readPower(chan));
    }

    @Command(description="Show whether a channel is voltage limited")
    public boolean isVoltageLimited(@Argument(description="Channel number")
                                    int chan) throws DriverException
    {
        return ipc.isVoltageLimited(chan);
    }

    @Command(description="Show whether a named channel is voltage limited")
    public boolean isVoltageLimitedN(@Argument(description="Channel number")
                                     String chan) throws DriverException
    {
        return ipc.isVoltageLimited(chan);
    }

    @Command(description="Show whether a channel is current limited")
    public boolean isCurrentLimited(@Argument(description="Channel number")
                                    int chan) throws DriverException
    {
        return ipc.isCurrentLimited(chan);
    }

    @Command(description="Show whether a named channel is current limited")
    public boolean isCurrentLimitedN(@Argument(description="Channel number")
                                     String chan) throws DriverException
    {
        return ipc.isCurrentLimited(chan);
    }

    @Command(description="Show whether a channel is power limited")
    public boolean isPowerLimited(@Argument(description="Channel number")
                                  int chan) throws DriverException
    {
        return ipc.isPowerLimited(chan);
    }

    @Command(description="Show whether a named channel is power limited")
    public boolean isPowerLimitedN(@Argument(description="Channel number")
                                   String chan) throws DriverException
    {
        return ipc.isPowerLimited(chan);
    }

    @Command(description="Show the channel names")
    public String showChannelNames() throws DriverException
    {
        StringBuilder text = new StringBuilder();
        String[] names = ipc.getChannelNames();
        String prefix = "";
        for (String name : names) {
            text.append(prefix).append(name == null ? "-" : name);
            prefix = " ";
        }
        return text.toString();
    }

    @Command(description="Show the complete power state")
    public String showAllPower() throws DriverException
    {
        String[] names = ipc.getChannelNames();
        int lName = 0;
        for (String name : names) {
            lName = Math.max(lName, name.length());
        }
        lName = Math.max(lName, 4);
        int lPref = (lName - 4) / 2, lSuff = lName - lPref - 4;
        StringBuilder text = new StringBuilder();
        text.append(String.format("Chan %" + ++lPref + "sName%" + ++lSuff + "s State  Vlim  Ilim  Plim   "
                                   + "Volts   mAmps  Watts", "", ""));
        int vLimits = ipc.getVlimitStatus();
        int iLimits = ipc.getIlimitStatus();
        int pLimits = ipc.getPlimitStatus();
        for (int chan = 0; chan < IonPump.NUM_CHANNELS; chan++, vLimits >>= 1, iLimits >>= 1, pLimits >>= 1) {
            text.append(String.format("\n %2s   %-" + lName + "s   %3s    %3s   %3s   %3s  %6.1f  %6.3f  %5.2f",
                        chan, names[chan], ipc.isPowered(chan) ? " on" : "off",
                        (vLimits & 1) != 0 ? " on" : "off", (iLimits & 1) != 0 ? " on" : "off",
                        (pLimits & 1) != 0 ? " on" : "off",
                        ipc.readVoltage(chan), 1000 * ipc.readCurrent(chan), ipc.readPower(chan)));
        }
        return text.toString();
    }

}
