package org.lsst.ccs.drivers.auxelex;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.utilities.conv.Convert;

/**
 *  Methods for accessing the Lambda power supply registers.
 *
 *  @author  Owen Saxton
 */
public class LambdaPS {

    public static final int
        MODEL_UNKNOWN = -1,
        MODEL_12 = 0,
        MODEL_28 = 1,
        MODEL_48 = 2;

    public static final int
        STATUS_OFF  = 0x01,
        STATUS_DSAB = 0x02,
        STATUS_IOGB = 0x04,
        STATUS_OTW  = 0x08,
        STATUS_OTP  = 0x10,
        STATUS_VBAD = 0x20,
        STATUS_IBAD = 0x40,
        STATUS_MASK = 0x7f;

    private static final int
        OFF_SERIAL_NO  = 0,
        OFF_FW_VERSION = 5,
        OFF_PR_VERSION = 6,
        OFF_VOLTAGE    = 7,
        OFF_CURRENT    = 8,
        OFF_TEMP       = 9,
        OFF_STATUS     = 10,
        OFF_PART_NO    = 11,
        OFF_MANU_DATE  = 14,
        OFF_MANU_LOCN  = 16,
        LENG_SERIAL_NO  = 20,
        LENG_FW_VERSION = 4,
        LENG_PR_VERSION = 4,
        LENG_PART_NO    = 12,
        LENG_MANU_DATE  = 8,
        LENG_MANU_LOCN  = 3,
        ADC_BAD_MASK    = 0xfffffc00;

    private static final Map<String, Integer> MODEL_MAP = new HashMap<>();
    static {
        MODEL_MAP.put("12", MODEL_12);
        MODEL_MAP.put("28", MODEL_28);
        MODEL_MAP.put("48", MODEL_48);
    }

    private static final double[]
        V_SCALES = {0.01668, 0.03973, 0.06772},
        I_SCALES = {0.0815, 0.0488, 0.0282};
    private static final double
        T_SCALE = 2.048,
        T_OFFSET = 25.0;
    private static final int
        T_ADC_OFFSET = 610;

    private final Srp srp;
    private final int baseAddr;
    private double vScale, iScale;
    private final int simModel;
    private double simVolts;


    /**
     *  Constructor
     *
     *  @param  srp       The Srp object to use for register access
     *  @param  baseAddr  The Lambda register base address
     *  @param  model     The expected Lambda model (for setting scale factors)
     */
    protected LambdaPS(Srp srp, int baseAddr, int model)
    {
        this.srp = srp;
        this.baseAddr = baseAddr;
        vScale = V_SCALES[model];
        iScale = I_SCALES[model];
        simModel = model;
    }


    /**
     *  Gets the PS serial number.
     *
     *  @return  The serial number
     *  @throws  DriverException
     */
    public String getSerialNo() throws DriverException
    {
        return readString(baseAddr + OFF_SERIAL_NO, LENG_SERIAL_NO);
    }


    /**
     *  Gets the PS firmware version.
     *
     *  @return  The firmware revision
     *  @throws  DriverException
     */
    public String getFwVersion() throws DriverException
    {
        return readString(baseAddr + OFF_FW_VERSION, LENG_FW_VERSION);
    }


    /**
     *  Gets the PS product version.
     *
     *  @return  The product revision
     *  @throws  DriverException
     */
    public String getPrVersion() throws DriverException
    {
        return readString(baseAddr + OFF_PR_VERSION, LENG_PR_VERSION);
    }


    /**
     *  Gets the PS part number.
     *
     *  @return  The unit part number
     *  @throws  DriverException
     */
    public String getPartNo() throws DriverException
    {
        return readString(baseAddr + OFF_PART_NO, LENG_PART_NO);
    }


    /**
     *  Gets the PS manufacturing date.
     *
     *  @return  The manufacturing date
     *  @throws  DriverException
     */
    public String getManuDate() throws DriverException
    {
        return readString(baseAddr + OFF_MANU_DATE, LENG_MANU_DATE);
    }


    /**
     *  Gets the PS manufacturing location.
     *
     *  @return  The manufacturing location
     *  @throws  DriverException
     */
    public String getManuLocn() throws DriverException
    {
        return readString(baseAddr + OFF_MANU_LOCN, LENG_MANU_LOCN);
    }


    /**
     *  Reads the PS status.
     *
     *  @return  The contents of the status register
     *  @throws  DriverException
     */
    public int readStatus() throws DriverException
    {
        return srp.readReg(baseAddr + OFF_STATUS) & STATUS_MASK;
    }


    /**
     *  Reads the PS voltage.
     *
     *  @return  The measured voltage
     *  @throws  DriverException
     */
    public double readVoltage() throws DriverException
    {
        return fixValue(srp.readReg(baseAddr + OFF_VOLTAGE)) * vScale;
    }


    /**
     *  Reads the PS current.
     *
     *  @return  The measured current
     *  @throws  DriverException
     */
    public double readCurrent() throws DriverException
    {
        return fixValue(srp.readReg(baseAddr + OFF_CURRENT)) * iScale;
    }


    /**
     *  Reads the PS baseplate temperature.
     *
     *  @return  The measured temperature (C)
     *  @throws  DriverException
     */
    public double readTemperature() throws DriverException
    {
        return (fixValue(srp.readReg(baseAddr + OFF_TEMP)) - T_ADC_OFFSET) / T_SCALE + T_OFFSET;
    }


    /**
     *  Gets the PS model.
     *
     *  Additionally the voltage and current scales are set appropriately.
     *
     *  @return  The model (MODEL_UNKNOWN, _12, _28, _48)
     *  @throws  DriverException
     */
    public int getModel() throws DriverException
    {
        int model = MODEL_UNKNOWN;
        String mName = getPartNo();
        if (mName.startsWith("CPFE1000FI")) {
            Integer mod = MODEL_MAP.get(mName.substring(10));
            if (mod != null) {
                model = mod;
                vScale = V_SCALES[mod];
                iScale = I_SCALES[mod];
            }
        }
        return model;
    }


    /**
     *  Fixes a value.
     *
     *  If the raw value is 2^10 or greater, NaN is returned.
     *
     *  @param  raw    The raw value
     *  @return  The calculated value
     */
    private static double fixValue(int raw)
    {
        return (raw & ADC_BAD_MASK) == 0 ? raw : Double.NaN;
    }


    /**
     *  Reads a string from the Lambda PS
     *
     *  @param  addr  The register address
     *  @param  leng  The length of the string
     *  @return  The string
     *  @throws  DriverException
     */
    private String readString(int addr, int leng) throws  DriverException
    {
        int[] words = srp.readRegs(addr, (leng + 3) / 4);
        byte[] bytes = new byte[4 * words.length];
        for (int j = 0; j < words.length; j++) {
            Convert.intToBytesBE(words[j], bytes, 4 * j);
        }
        return new String(bytes, 0, leng);
    }


    /**
     *  Initializes simulation
     */
    public void simInitialize()
    {
        simWriteString(baseAddr + OFF_SERIAL_NO, "00000000000000000000");
        simWriteString(baseAddr + OFF_FW_VERSION, "0000");
        simWriteString(baseAddr + OFF_PR_VERSION, "0000");
        simWriteString(baseAddr + OFF_MANU_DATE, "00000000");
        simWriteString(baseAddr + OFF_MANU_LOCN, "000");
        String volts = "12";
        for (Map.Entry e : MODEL_MAP.entrySet()) {
            if ((int)e.getValue() == simModel) {
                volts = (String)e.getKey();
                break;
            }
        }
        simWriteString(baseAddr + OFF_PART_NO, "CPFE1000FI" + volts);
        simVolts = Double.valueOf(volts);
        srp.putSimRegMap(baseAddr + OFF_STATUS, 0);
        srp.putSimRegMap(baseAddr + OFF_VOLTAGE, 0);
        srp.putSimRegMap(baseAddr + OFF_CURRENT, 0);
        double temp = 26.5;
        srp.putSimRegMap(baseAddr + OFF_TEMP, (int)((temp - T_OFFSET) * T_SCALE) + T_ADC_OFFSET);
    }

    
    /**
     *  Sets simulated output on or off.
     *
     *  @param  on  Whether on
     */
    public void simSetOutput(boolean on)
    {
        srp.putSimRegMap(baseAddr + OFF_VOLTAGE, on ? (int)(simVolts / vScale) : 0);
    }


    /**
     *  Gets the simulated voltage.
     * 
     *  @return  The voltage
     */
    public double simGetVoltage()
    {
        return simVolts;
    }


    /**
     *  Writes a string to a simulated Lambda PS
     *
     *  @param  addr  The register address
     *  @param  str   The string
     */
    private void simWriteString(int addr, String str)
    {
        byte[] bytes = (str + "   ").getBytes();
        for (int j = 0; j < str.length(); j += 4) {
            int word = Convert.bytesToIntBE(bytes, j);
            srp.putSimRegMap(addr + j / 4, word);
        }
    }

}
