package org.lsst.ccs.drivers.auxelex;

import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.net.Socket;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Routines for controlling the power breaker-filter-relay board (BFR)
 *
 *  @author  Owen Saxton
 */
public class Bfr extends Srp {

    /**
     *  Constants and data.
     */
    public static final int
        RELAY_HTR_1 = 0,
        RELAY_HTR_2 = 1,
        RELAY_PWS_2 = 2,
        RELAY_PWS_3 = 3,
        RELAY_PWS_4 = 4,
        RELAY_PWS_16 = 5,
        RELAY_PWS_17 = 6,
        RELAY_PWS_18 = 7,
        RELAY_PWS_19 = 8,
        RELAY_PWS_25 = 9,
        RELAY_PWS_20 = 10,
        RELAY_PWS_21 = 11,
        NUM_RELAYS = 12;

    public static final int
        SENSOR_HTRS = 0,
        SENSOR_PWS_1 = 1,
        SENSOR_PWS_3 = 2,
        SENSOR_PWS_4 = 3,
        SENSOR_PWS_16 = 4,
        SENSOR_PWS_17 = 5,
        SENSOR_PWS_18 = 6,
        SENSOR_PWS_19 = 7,
        SENSOR_PWS_25 = 8,
        SENSOR_PWS_20_21 = 9,
        SENSOR_PROT = 10,
        NUM_SENSORS = 11;

    public enum CurrentType {RMS, AC, DC}

    private static final Map<String, Integer> relayMap = new HashMap<>();
    static {
        relayMap.put("heater1", RELAY_HTR_1);
        relayMap.put("heater2", RELAY_HTR_2);
        relayMap.put("pws02-48VD", RELAY_PWS_2);
        relayMap.put("pws03-24VD", RELAY_PWS_3);
        relayMap.put("pws04-28V", RELAY_PWS_4);
        relayMap.put("pws16-ps0-2", RELAY_PWS_16);
        relayMap.put("pws17-ps3-5", RELAY_PWS_17);
        relayMap.put("pws18-ps6-8", RELAY_PWS_18);
        relayMap.put("pws19-ps9-10", RELAY_PWS_19);
        relayMap.put("pws25-psSpre", RELAY_PWS_25);
        relayMap.put("pws20-24VC", RELAY_PWS_20);
        relayMap.put("pws21-5V", RELAY_PWS_21);
    }
    private static final int
        REG_RELAYS        = 0x10000,
        REG_MBUS_CMND     = 0x20000,
        REG_MBUS_RESP     = 0x20002,
        REG_MBUS_RESP_RDY = 0x2001c,
        MBUS_ADDRESS      = 1,
        MBUS_FUNC_READ    = 0x03,
        MBUS_FUNC_WRITE   = 0x06,
        BASE_ADDR_INST    = 0x0000,
        BASE_ADDR_MIN     = 0x0400,
        BASE_ADDR_MAX     = 0x0800,
        BASE_ADDR_HOLD    = 0x0c00,
        OFF_ADDR_RMS      = 0x0000,
        OFF_ADDR_AC       = 0x0100,
        OFF_ADDR_DC       = 0x0200,
        TRIGGER_ADDR      = 0x3010,
        SENSOR_READ_TMO   = 0xaa;

    private final static Map<CurrentType, Integer> offsetMap = new HashMap<>();
    static {
        offsetMap.put(CurrentType.RMS, OFF_ADDR_RMS);
        offsetMap.put(CurrentType.AC, OFF_ADDR_AC);
        offsetMap.put(CurrentType.DC, OFF_ADDR_DC);
    }

    private static final List<BoardType> validTypes = new ArrayList<>();
    static {
        validTypes.add(BoardType.BFR);
        validTypes.add(BoardType.SIMULATED);
    }
    static String[] relayNames = new String[NUM_RELAYS];
    static {
        for (String name : relayMap.keySet()) {
            relayNames[relayMap.get(name)] = name;
        }
    }

    private static final double CURRENT_SCALE = 100.0;

    private boolean isServer = false;
    private Socket serverSock;
    private ObjectOutputStream serverOut;
    private ObjectInputStream serverIn;
    private static final int[] simRelaySensors = new int[NUM_RELAYS + 1];
    static {
        simRelaySensors[RELAY_HTR_1] = SENSOR_HTRS;
        simRelaySensors[RELAY_HTR_2] = SENSOR_HTRS;
        simRelaySensors[RELAY_PWS_2] = SENSOR_PWS_1;
        simRelaySensors[RELAY_PWS_3] = SENSOR_PWS_3;
        simRelaySensors[RELAY_PWS_4] = SENSOR_PWS_4;
        simRelaySensors[RELAY_PWS_16] = SENSOR_PWS_16;
        simRelaySensors[RELAY_PWS_17] = SENSOR_PWS_17;
        simRelaySensors[RELAY_PWS_18] = SENSOR_PWS_18;
        simRelaySensors[RELAY_PWS_19] = SENSOR_PWS_19;
        simRelaySensors[RELAY_PWS_20] = SENSOR_PWS_20_21;
        simRelaySensors[RELAY_PWS_21] = SENSOR_PWS_20_21;
        simRelaySensors[RELAY_PWS_25] = SENSOR_PWS_25;
        simRelaySensors[NUM_RELAYS] = SENSOR_PROT;
    }
    private static final int[] simSensorRelays = new int[NUM_SENSORS];
    static {
        Arrays.fill(simSensorRelays, -1);
        for (int relay = 0; relay <= NUM_RELAYS; relay++) {
            int sensor = simRelaySensors[relay];
            int oldRelay = simSensorRelays[sensor];
            simSensorRelays[sensor] = (oldRelay == -1) ? relay : oldRelay | (relay << 8);
        }
    }
    private final int[] simCurrents = new int[NUM_RELAYS + 1];
    private final Map<Integer, Integer> simModbus = new HashMap<>();


    /**
     *  Constructor
     */
    public Bfr()
    {
        setValidBoardTypes(validTypes);
    }


    /**
     *  Opens a connection to a board.
     *
     *  @param  host  The host name or IP address, or null or empty for simulation
     *  @param  port  The port number, or 0 for default
     *  @throws  DriverException
     */
    @Override
    public synchronized void open(String host, int port) throws DriverException
    {
        super.open(host, port);
        isServer = (port == DEFAULT_PORT || port == 0) && !isSimulated();
        if (isServer) {
            try {
                BfrServer.startServer(this);
            }
            catch (DriverException e) {
                close();
                throw e;
            }
        }
    }


    /**
     *  Closes the connection.
     *
     *  @throws  DriverException
     */
    @Override
    public void close() throws DriverException
    {
        if (isServer) {
            BfrServer.stopServer(getNode());
        }
        else {
            closeServer();
        }
        super.close();
    }


    /**
     *  Turns on a relay by number
     *
     *  @param  relay  The relay number (0 - 11)
     *  @throws  DriverException if the relay number is out of bounds
     */
    public void setRelayOn(int relay) throws  DriverException
    {
        writeReg(REG_RELAYS + checkRelayNumber(relay), 1);
    }


    /**
     *  Turns on a relay by name
     *
     *  @param  relay  The relay name
     *  @throws  DriverException if the relay name is invalid
     */
    public void setRelayOn(String relay) throws  DriverException
    {
        writeReg(REG_RELAYS + getRelayNumber(relay), 1);
    }


    /**
     *  Turns on several relays by number
     *
     *  @param  relays  An arg list or array of relay numbers (0 - 11)
     *  @throws  DriverException if any relay number is out of bounds
     */
    public void setRelayOn(int... relays) throws  DriverException
    {
        for (int relay : relays) {
            checkRelayNumber(relay);
        }
        for (int relay : relays) {
            writeReg(REG_RELAYS + relay, 1);
        }
    }


    /**
     *  Turns on several relays by name
     *
     *  @param  relays  An arg list or array of relay names
     *  @throws  DriverException if any relay name is invalid
     */
    public void setRelayOn(String... relays) throws  DriverException
    {
        List<Integer> numbers = new ArrayList<>();
        for (String relay : relays) {
            numbers.add(getRelayNumber(relay));
        }
        for (int number : numbers) {
            writeReg(REG_RELAYS + number, 1);
        }
    }


    /**
     *  Turns on all relays
     *
     *  @throws  DriverException  if any relay number is out of bounds
     */
    public void setRelayOn() throws  DriverException
    {
        for (int relay = 0; relay < NUM_RELAYS; relay++) {
            writeReg(REG_RELAYS + relay, 1);
        }
    }


    /**
     *  Turns off a relay by number
     *
     *  @param  relay  The relay number (0 - 11)
     *  @throws  DriverException  if the relay number is out of bounds
     */
    public void setRelayOff(int relay) throws  DriverException
    {
        writeReg(REG_RELAYS + checkRelayNumber(relay), 0);
    }


    /**
     *  Turns off a relay by name
     *
     *  @param  relay  The relay name
     *  @throws  DriverException if the relay name is invalid
     */
    public void setRelayOff(String relay) throws  DriverException
    {
        writeReg(REG_RELAYS + getRelayNumber(relay), 0);
    }


    /**
     *  Turns off several relays by number
     *
     *  @param  relays  An arg list or array of relay numbers (0 - 11)
     *  @throws  DriverException  if any relay number is out of bounds
     */
    public void setRelayOff(int... relays) throws  DriverException
    {
        for (int relay : relays) {
            checkRelayNumber(relay);
        }
        for (int relay : relays) {
            writeReg(REG_RELAYS + relay, 0);
        }
    }


    /**
     *  Turns off several relays by name
     *
     *  @param  relays  An arg list or array of relay names
     *  @throws  DriverException if any relay name is invalid
     */
    public void setRelayOff(String... relays) throws  DriverException
    {
        List<Integer> numbers = new ArrayList<>();
        for (String relay : relays) {
            numbers.add(getRelayNumber(relay));
        }
        for (int number : numbers) {
            writeReg(REG_RELAYS + number, 0);
        }
    }


    /**
     *  Turns off all relays
     *
     *  @throws  DriverException
     */
    public void setRelayOff() throws  DriverException
    {
        for (int relay = 0; relay < NUM_RELAYS; relay++) {
            writeReg(REG_RELAYS + relay, 0);
        }
    }


    /**
     *  Gets the state of a relay by number
     *
     *  @param  relay  The relay number (0 - 11)
     *  @return  Whether the relay is on
     *  @throws  DriverException if the relay number is out of bounds
     */
    public boolean isRelayOn(int relay) throws  DriverException
    {
        return (readReg(REG_RELAYS + checkRelayNumber(relay)) & 1) != 0;
    }


    /**
     *  Gets the state of a relay by name
     *
     *  @param  relay  The relay name
     *  @return  Whether the relay is on
     *  @throws  DriverException if the relay name is invalid
     */
    public boolean isRelayOn(String relay) throws  DriverException
    {
        return (readReg(REG_RELAYS + getRelayNumber(relay)) & 1) != 0;
    }


    /**
     *  Gets the state of all relays
     *
     *  @return  12-element boolean array of relay states
     *  @throws  DriverException
     */
    public boolean[] isRelayOn() throws  DriverException
    {
        int[] regs = readRegs(REG_RELAYS, NUM_RELAYS);
        boolean[] state = new boolean[NUM_RELAYS];
        for (int relay = 0; relay < NUM_RELAYS; relay++) {
            state[relay] = (regs[relay] & 1) != 0;
        }
        return state;
    }


    /**
     *  Reads a range of instantaneous current values.
     *
     *  @param  type    The type of current (RMS, AC, DC)
     *  @param  sensor  The first sensor number (0 - 10)
     *  @param  count   The number of sensors to read
     *  @return  An array of current values
     *  @throws  DriverException
     */
    public double[] readCurrent(CurrentType type, int sensor, int count) throws DriverException
    {
        return readSensor(BASE_ADDR_INST + offsetMap.get(type), sensor, count);
    }


    /**
     *  Reads an instantaneous current value.
     *
     *  @param  type    The type of current (RMS, AC, DC)
     *  @param  sensor  The sensor number (0 - 10)
     *  @return  The current value
     *  @throws  DriverException
     */
    public double readCurrent(CurrentType type, int sensor) throws DriverException
    {
        return readCurrent(type, sensor, 1)[0];
    }


    /**
     *  Reads all instantaneous current values.
     *
     *  @param  type  The type of current (RMS, AC, DC)
     *  @return  The current value
     *  @throws  DriverException
     */
    public double[] readCurrent(CurrentType type) throws DriverException
    {
        return readCurrent(type, 0, NUM_SENSORS);
    }


    /**
     *  Reads a range of minimum current values.
     *
     *  @param  type    The type of current (RMS, AC, DC)
     *  @param  sensor  The first sensor number (0 - 10)
     *  @param  count   The number of sensors to read
     *  @return  An array of current values
     *  @throws  DriverException
     */
    public double[] readMinCurrent(CurrentType type, int sensor, int count) throws DriverException
    {
        return readSensor(BASE_ADDR_MIN + offsetMap.get(type), sensor, count);
    }


    /**
     *  Reads a minimum current value.
     *
     *  @param  type    The type of current (RMS, AC, DC)
     *  @param  sensor  The sensor number (0 - 10)
     *  @return  The current value
     *  @throws  DriverException
     */
    public double readMinCurrent(CurrentType type, int sensor) throws DriverException
    {
        return readMinCurrent(type, sensor, 1)[0];
    }


    /**
     *  Reads all minimum current values.
     *
     *  @param  type  The type of current (RMS, AC, DC)
     *  @return  The current value
     *  @throws  DriverException
     */
    public double[] readMinCurrent(CurrentType type) throws DriverException
    {
        return readMinCurrent(type, 0, NUM_SENSORS);
    }


    /**
     *  Reads a range of maximum current values.
     *
     *  @param  type    The type of current (RMS, AC, DC)
     *  @param  sensor  The first sensor number (0 - 10)
     *  @param  count   The number of sensors to read
     *  @return  An array of current values
     *  @throws  DriverException
     */
    public double[] readMaxCurrent(CurrentType type, int sensor, int count) throws DriverException
    {
        return readSensor(BASE_ADDR_MAX + offsetMap.get(type), sensor, count);
    }


    /**
     *  Reads a maximum current value.
     *
     *  @param  type    The type of current (RMS, AC, DC)
     *  @param  sensor  The sensor number (0 - 10)
     *  @return  The current value
     *  @throws  DriverException
     */
    public double readMaxCurrent(CurrentType type, int sensor) throws DriverException
    {
        return readMaxCurrent(type, sensor, 1)[0];
    }


    /**
     *  Reads all maximum current values.
     *
     *  @param  type  The type of current (RMS, AC, DC)
     *  @return  The current value
     *  @throws  DriverException
     */
    public double[] readMaxCurrent(CurrentType type) throws DriverException
    {
        return readMaxCurrent(type, 0, NUM_SENSORS);
    }


    /**
     *  Reads a range of held current values.
     *
     *  @param  type    The type of current (RMS, AC, DC)
     *  @param  sensor  The first sensor number (0 - 10)
     *  @param  count   The number of sensors to read
     *  @return  An array of current values
     *  @throws  DriverException
     */
    public double[] readHeldCurrent(CurrentType type, int sensor, int count) throws DriverException
    {
        return readSensor(BASE_ADDR_HOLD + offsetMap.get(type), sensor, count);
    }


    /**
     *  Reads a held current value.
     *
     *  @param  type    The type of current (RMS, AC, DC)
     *  @param  sensor  The sensor number (0 - 10)
     *  @return  The current value
     *  @throws  DriverException
     */
    public double readHeldCurrent(CurrentType type, int sensor) throws DriverException
    {
        return readHeldCurrent(type, sensor, 1)[0];
    }


    /**
     *  Reads all held current values.
     *
     *  @param  type  The type of current (RMS, AC, DC)
     *  @return  The current value
     *  @throws  DriverException
     */
    public double[] readHeldCurrent(CurrentType type) throws DriverException
    {
        return readHeldCurrent(type, 0, NUM_SENSORS);
    }


    /**
     *  Resets all minimum and maximum values
     *
     *  @throws  DriverException
     */
    public void resetExtrema() throws DriverException
    {
        if (isSimulated()) {
            simResetExtrema();
        }
        else {
            int[] cmnd = {(MBUS_ADDRESS << 24) | (MBUS_FUNC_WRITE << 16) | TRIGGER_ADDR, 0x0001 << 16};
            writeRegs(REG_MBUS_CMND, cmnd);
        }
    }


    /**
     *  Triggers a hold operation
     *
     *  @throws  DriverException
     */
    public void triggerHold() throws DriverException
    {
        if (isSimulated()) {
            simTriggerHold();
        }
        else {
            int[] cmnd = {(MBUS_ADDRESS << 24) | (MBUS_FUNC_WRITE << 16) | TRIGGER_ADDR, 0x0010 << 16};
            writeRegs(REG_MBUS_CMND, cmnd);
        }
    }


    /**
     *  Gets the names of the relays
     *
     *  @return  12-element string array of relay names, in number order
     *  @throws  DriverException
     */
    public static String[] getRelayNames() throws  DriverException
    {
        return relayNames;
    }


    /**
     *  Reads a range of current values.
     * 
     *  This uses the BFR server if this instance is not running it.
     *
     *  @param  addr    The current modbus address
     *  @param  sensor  The first sensor number
     *  @param  count   The number of sensors to read
     *  @return  The array of current values
     *  @throws  DriverException
     */
    private double[] readSensor(int addr, int sensor, int count) throws DriverException
    {
        if (isSimulated()) {
            return simReadSensor(addr, sensor, count);
        }
        else if (isServer) {
            return readSensorDirect(addr, sensor, count);
        }
        else {
            checkOpen();
            return sendServer(addr, sensor, count);
        }
    }


    /**
     *  Directly reads a range of current values.
     *
     *  @param  addr    The current modbus address
     *  @param  sensor  The first sensor number
     *  @param  count   The number of sensors to read
     *  @return  The array of current values
     *  @throws  DriverException
     */
    synchronized double[] readSensorDirect(int addr, int sensor, int count) throws DriverException  // Also used by BfrServer
    {
        if (sensor < 0 || sensor >= NUM_SENSORS) {
            throw new DriverException("Invalid sensor number: " + sensor);
        }
        if (count <= 0 || sensor + count > NUM_SENSORS) {
            throw new DriverException("Invalid sensor count: " + count);
        }
        int cmnd[] = {(MBUS_ADDRESS << 24) | (MBUS_FUNC_READ << 16) | (addr + sensor), count << 16};
        writeRegs(REG_MBUS_CMND, cmnd);
        long endTime = System.currentTimeMillis() + 1000;
        int resp = 0;
        while (System.currentTimeMillis() < endTime) {
            resp = readReg(REG_MBUS_RESP_RDY);
            if ((resp & 1) != 0) break;
            try {
                Thread.sleep(1);
            }
            catch (InterruptedException e) {}
        }
        if ((resp & 1) == 0 || (resp >>> 24) == SENSOR_READ_TMO) {
            throw new DriverException("Current sensor read timed out");
        }
        int[] data = readRegs(REG_MBUS_RESP, (2 * count + 6) / 4);
        int func = (data[0] >> 16) & 0xff;
        int nBytes = (data[0] >> 8) & 0xff;
        if ((func >> 7) != 0) {
            throw new DriverException("Modbus read error: " + nBytes);
        }
        if (nBytes != 2 * count) {
            throw new DriverException("Incorrect returned data length: " + nBytes);
        }
        double[] values = new double[count];
        for (int j = 0; j < count; j++) {
            if ((j & 1) == 0) {
                values[j] = (((data[j / 2] << 24) >> 16) | (data[j / 2 + 1] >>> 24)) / CURRENT_SCALE;
            }
            else {
                values[j] = ((data[j / 2 + 1] << 8) >> 16) / CURRENT_SCALE;
            }
        }
        return values;
    }


    /**
     *  Checks a relay number for validity
     *
     *  @param  relay  The relay number
     *  @return  The relay number
     *  @throws  DriverException  if the relay number is out of bounds
     */
    private int checkRelayNumber(int relay) throws  DriverException
    {
        if (relay < 0 || relay >= NUM_RELAYS) {
            throw new DriverException("Invalid relay number: " + relay);
        }
        return relay;
    }


    /**
     *  Gets a relay number from its name
     *
     *  @param  relay  The relay name
     *  @return  The relay number
     *  @throws  DriverException  if the relay name is invalid
     */
    private int getRelayNumber(String relay) throws  DriverException
    {
        Integer number = relayMap.get(relay);
        if (number == null) {
            throw new DriverException("Invalid relay name: " + relay);
        }
        return number;
    }


    /**
     *  Initializes the simulation.
     */
    @Override
    protected void simInitialize()
    {
        super.simInitialize();
        putSimRegMap(REG_MBUS_RESP_RDY, 1);
        int[] data = {100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 112};
        System.arraycopy(data, 0, simCurrents, 0, NUM_RELAYS + 1);
        simWriteSensor(BASE_ADDR_INST, CurrentType.AC, SENSOR_PROT, simCurrents[NUM_RELAYS]);
        simWriteSensor(BASE_ADDR_INST, CurrentType.RMS, SENSOR_PROT, simCurrents[NUM_RELAYS]);
        simWriteSensor(BASE_ADDR_MAX, CurrentType.AC, SENSOR_PROT, simCurrents[NUM_RELAYS]);
        simWriteSensor(BASE_ADDR_MAX, CurrentType.RMS, SENSOR_PROT, simCurrents[NUM_RELAYS]);
        simWriteSensor(BASE_ADDR_MIN, CurrentType.AC, SENSOR_PROT, simCurrents[NUM_RELAYS]);
        simWriteSensor(BASE_ADDR_MIN, CurrentType.RMS, SENSOR_PROT, simCurrents[NUM_RELAYS]);
        
    }


    /**
     *  Writes simulated registers.
     *
     *  @param  addr   The first register address
     *  @param  value  The array of values to write.
     *  @param  count  The number of values to write.
     */
    @Override
    protected void simWriteRegs(int addr, int[] value, int count)
    {
        for (int j = 0; j < count; j++, addr++) {
            int oldValue = getSimRegMap(addr);
            putSimRegMap(addr, value[j]);
            if (addr == REG_MBUS_CMND + 1) {
                putSimRegMap(REG_MBUS_RESP, value[j] >> 7);
            }
            if (addr >= REG_RELAYS && addr < REG_RELAYS + NUM_RELAYS) {
                if (oldValue != value[j]) {
                    int relay = addr - REG_RELAYS;
                    int sensor = simRelaySensors[relay];
                    int current = simReadSensor(BASE_ADDR_INST, CurrentType.AC, sensor);
                    if (oldValue == 0) {
                        current += simCurrents[relay];
                    }
                    else {
                        current -= simCurrents[relay];
                    }
                    simWriteSensor(BASE_ADDR_INST, CurrentType.AC, sensor, current);
                    simWriteSensor(BASE_ADDR_INST, CurrentType.RMS, sensor, current);
                    int maxCurrent = Math.max(simReadSensor(BASE_ADDR_MAX, CurrentType.AC, sensor), current);
                    simWriteSensor(BASE_ADDR_MAX, CurrentType.AC, sensor, maxCurrent);
                    simWriteSensor(BASE_ADDR_MAX, CurrentType.RMS, sensor, maxCurrent);
                    int minCurrent = Math.min(simReadSensor(BASE_ADDR_MIN, CurrentType.AC, sensor), current);
                    simWriteSensor(BASE_ADDR_MIN, CurrentType.AC, sensor, minCurrent);
                    simWriteSensor(BASE_ADDR_MIN, CurrentType.RMS, sensor, minCurrent);
                }
            }
        }
    }


    /**
     *  Writes a range of simulated current values.
     *
     *  Instantaneous, maximum and minimum values are updated
     * 
     *  @param  relay  The first relay number (0 - 11, 12 = protection)
     *  @param  value  The array of values to write.
     *  @throws  DriverException
     */
    public void simWriteCurrent(int relay, double[] value) throws DriverException
    {
        if (isSimulated()) {
            if (relay < 0 || relay > NUM_RELAYS) {
                throw new DriverException("Invalid relay number: " + relay);
            }
            if (relay + value.length > NUM_RELAYS + 1) {
                throw new DriverException("Too many values supplied: " + value.length);
            }
            for (int j = 0; j < value.length; j++, relay++) {
                simCurrents[relay] = (int)Math.round(CURRENT_SCALE * value[j]);
            }
            boolean[] relaysOn = Arrays.copyOf(isRelayOn(), NUM_RELAYS + 1);
            relaysOn[NUM_RELAYS] = true;
            int[] instValue = new int[NUM_SENSORS];
            for (int j = 0; j < NUM_SENSORS; j++) {
                int relays = simSensorRelays[j];
                instValue[j] = relaysOn[relays & 0xff] ? simCurrents[relays & 0xff] : 0;
                relays >>= 8;
                if (relays > 0) {
                    instValue[j] += relaysOn[relays] ? simCurrents[relays] : 0;
                }
            }
            simWriteSensor(BASE_ADDR_INST, CurrentType.AC, instValue);
            simWriteSensor(BASE_ADDR_INST, CurrentType.RMS, instValue);
            int[] maxValue = simReadSensor(BASE_ADDR_MAX, CurrentType.AC);
            for (int j = 0; j < NUM_SENSORS; j++) {
                maxValue[j] = Math.max(maxValue[j], instValue[j]);
            }
            simWriteSensor(BASE_ADDR_MAX, CurrentType.AC, maxValue);
            simWriteSensor(BASE_ADDR_MAX, CurrentType.RMS, maxValue);
            int[] minValue = simReadSensor(BASE_ADDR_MIN, CurrentType.AC);
            for (int j = 0; j < NUM_SENSORS; j++) {
                minValue[j] = Math.min(minValue[j], instValue[j]);
            }
            simWriteSensor(BASE_ADDR_MIN, CurrentType.AC, minValue);
            simWriteSensor(BASE_ADDR_MIN, CurrentType.RMS, minValue);
        }
    }


    /**
     *  Reads a range of simulated sensor values.
     *
     *  @param  addr    The current modbus address
     *  @param  sensor  The first sensor number
     *  @param  count   The number of sensors to read
     *  @return  The array of current values
     */
    private double[] simReadSensor(int addr, int sensor, int count)
    {
        double data[]= new double[count];
        for (int j = 0; j < count; j++) {
            Integer value = simModbus.get(addr + sensor + j);
            data[j] = value == null ? 0.0 : value / CURRENT_SCALE;
        }
        return data;
    }


    /**
     *  Writes a simulated sensor value.
     * 
     *  @param  addr    The base modbus address
     *  @param  type    The sensor type
     *  @param  sensor  The sensor number
     *  @param  value   The value to write
     */
    private void simWriteSensor(int addr, CurrentType type, int sensor, int value)
    {
        simModbus.put(addr + offsetMap.get(type) + sensor, value);
    }


    /**
     *  Writes all simulated sensor values.
     * 
     *  @param  addr    The base modbus address
     *  @param  type    The sensor type
     *  @param  value   The array of values to write
     */
    private void simWriteSensor(int addr, CurrentType type, int[] value)
    {
        for (int j = 0; j < NUM_SENSORS; j++) {
            simModbus.put(addr + offsetMap.get(type) + j, value[j]);
        }
    }


    /**
     *  Reads a simulated sensor value.
     *
     *  @param  addr  The base modbus address
     *  @param  type  The current type
     *  @return  The sensor value
     */
    private int simReadSensor(int addr, CurrentType type, int sensor)
    {
        Integer value = simModbus.get(addr + offsetMap.get(type) + sensor);
        return value == null ? 0 : value;
    }


    /**
     *  Reads all simulated sensor values.
     *
     *  @param  addr  The base modbus address
     *  @param  type  The current type
     *  @return  The array of sensor values
     */
    private int[] simReadSensor(int addr, CurrentType type)
    {
        int[] data= new int[NUM_SENSORS];
        for (int j = 0; j < NUM_SENSORS; j++) {
            Integer value = simModbus.get(addr + offsetMap.get(type) + j);
            data[j] = value == null ? 0 : value;
        }
        return data;
    }


    /**
     *  Resets all simulated minimum and maximum values
     */
    private void simResetExtrema()
    {
        int[] data = simReadSensor(BASE_ADDR_INST, CurrentType.AC);
        simWriteSensor(BASE_ADDR_MIN, CurrentType.AC, data);
        simWriteSensor(BASE_ADDR_MIN, CurrentType.RMS, data);
        simWriteSensor(BASE_ADDR_MAX, CurrentType.AC, data);
        simWriteSensor(BASE_ADDR_MAX, CurrentType.RMS, data);
    }


    /**
     *  Triggers a simulated hold operation
     */
    private void simTriggerHold()
    {
        int[] data = simReadSensor(BASE_ADDR_INST, CurrentType.AC);
        simWriteSensor(BASE_ADDR_HOLD, CurrentType.AC, data);
        simWriteSensor(BASE_ADDR_HOLD, CurrentType.RMS, data);
    }


    /**
     *  Connects to the BFR server.
     * 
     *  @throws  DriverException 
     */
    private void connectServer() throws DriverException
    {
        try {
            serverSock = new Socket(BfrServer.IP, BfrServer.getServerPort(getNode()));
            serverOut = new ObjectOutputStream(serverSock.getOutputStream());
            serverOut.flush();
            serverIn = new ObjectInputStream(serverSock.getInputStream());
        }
        catch (IOException e) {
            serverSock = null;
            throw new DriverException("Error connecting to BFR server", e);
        }
    }


    /**
     *  Closes the connection to the BFR server.
     */
    private void closeServer() {
        if (serverSock != null) {
            try {
                serverSock.close();
                serverSock = null;
            }
            catch (IOException e) {}
        }
    }


    /**
     *  Sends a request to the BFR server.
     * 
     *  @param  addr    The current modbus address
     *  @param  sensor  The first sensor number
     *  @param  count   The number of sensors to read
     *  @return  The array of current values
     *  @throws  DriverException
     */
    private double[] sendServer(int addr, int sensor, int count) throws DriverException
    {
        if (serverSock == null) {
            connectServer();
        }
        try {
            serverOut.writeObject(new BfrServer.Request(addr, sensor, count));
            try {
                BfrServer.Reply reply = (BfrServer.Reply)serverIn.readObject();
                if (reply.ex != null) {
                    throw reply.ex;
                }
                return reply.values;
            }
            catch (IOException | ClassNotFoundException e) {
                closeServer();
                throw new DriverException("Error receiving from BFR server", e);
            }
        }
        catch (IOException e) {
            closeServer();
            throw new DriverException("Error sending to BFR server", e);
        }
    }

}
