package org.lsst.ccs.drivers.auxelex;

import java.io.IOException;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Program to test the SLAC Register Protocol driver
 * 
 *  @author Owen Saxton
 */
public class TestSrp {

    /**
     *  Inner class for detecting console input while reading continuously.
     */
    protected class ConsThread extends Thread {

        private final BlockingQueue<Integer> consQueue = new ArrayBlockingQueue<>(1);
        private boolean[] consDone;
        private Thread mainThread;

        ConsThread() {
            super();
            setDaemon(true);
        }

        @Override
        public void run() {
            while (true) {
                awaitStart();
                awaitTerminal();
                consDone[0] = true;
                if (mainThread != null) {
                    mainThread.interrupt();
                }
            }
        }

        public void start(boolean[] done, boolean wake) {
            if (getState() == Thread.State.NEW) {
                start();
            }
            consDone = done;
            consDone[0] = false;
            mainThread = wake ? Thread.currentThread() : null;
            consQueue.offer(0);
        }

        public void start(boolean[] done) {
            start(done, false);
        }

        private void awaitStart() {
            while (true) {
                try {
                    consQueue.take();
                    return;
                }
                catch (InterruptedException e) {
                }
            }
        }

        private void awaitTerminal() {
            while (true) {
                try {
                    if (System.in.available() > 0) {
                        break;
                    }
                    try {
                        Thread.sleep(50);
                    }
                    catch (InterruptedException e) {
                    }
                }
                catch (IOException e) {
                    break;
                }
            }
            try {
                while (System.in.available() > 0) {
                    System.in.read();
                }
            }
            catch (IOException e) {
            }
        }

    }

    /**
     *  Data fields
     */
    protected static enum OnOff { ON, OFF; };
    protected final Srp srp;
    protected final ConsThread consThread = new ConsThread();


    /**
     *  Constructor.
     */
    public TestSrp()
    {
        srp = new Srp();
    }

    /**
     *  Constructor.
     *
     *  @param  srp  Srp object
     */
    public TestSrp(Srp srp)
    {
        this.srp = srp;
    }

    @Command(description="Set the SRP version")
    public void setSrpVersion(@Argument(description="Version number") int version)
    {
        srp.setSrpVersion(version);
    }

    @Command(description="Set the debug state")
    public void setDebug(@Argument(description="The state (on or off)") OnOff state)
    {
        srp.setDebug(state == OnOff.ON);
    }

    @Command(description="Open connection to device")
    public void connectIp(@Argument(description="IP address") String ipAddr,
                          @Argument(description="Port number") int port) throws DriverException
    {
        srp.open(ipAddr, port);
        announce();
    }

    @Command(description="Open connection to device")
    public void connectIp(@Argument(description="IP address") String ipAddr) throws DriverException
    {
        srp.open(ipAddr);
        announce();
    }

    @Command(description="Open connection to device")
    public void connect(@Argument(description="Node number") int node,
                        @Argument(description="Port number") int port) throws DriverException
    {
        srp.open(node, port);
        announce();
    }

    @Command(description="Open connection to device")
    public void connect(@Argument(description="Node number") int node) throws DriverException
    {
        srp.open(node);
        announce();
    }

    private void announce()
    {
        System.out.println("Board type is " + srp.getBoardType());
    }

    @Command(description="Close connection to device")
    public void disconnect() throws DriverException
    {
        srp.close();
    }

    @Command(description="Show the board type")
    public Srp.BoardType showBoardType() throws DriverException
    {
        return srp.getBoardType();
    }

    @Command(description="Show the build stamp")
    public String showBuildStamp() throws DriverException
    {
        return srp.getBuildStamp();
    }

    @Command(description="Show the FPGA version")
    public String showFpgaVersion() throws DriverException
    {
        return String.format("0x%08x", srp.getFpgaVersion());
    }

    @Command(description="Show the board up time (seconds)")
    public int showUpSecs() throws DriverException
    {
        return srp.getUpTime();
    }

    @Command(description="Show the board up time (ddd hh:mm:ss)")
    public String showUpTime() throws DriverException
    {
        int secs = srp.getUpTime();
        int mins = secs / 60;
        int hrs = mins / 60;
        return String.format("%d %02d:%02d:%02d", hrs / 24, hrs % 60, mins % 60, secs % 60);
    }

    @Command(description="Write the scratch pad register")
    public void writeScratch(@Argument(description="The value to write") int value) throws DriverException
    {
        srp.writeScratchPad(value);
    }

    @Command(description="Read the scratch pad register")
    public int readScratch() throws DriverException
    {
        return srp.readScratchPad();
    }

    @Command(description="Perform a user reset")
    public void userReset() throws DriverException
    {
        srp.userReset();
    }

    @Command(description="Reload the FPGA")
    public void reloadFpga() throws DriverException
    {
        srp.reloadFpga();
    }

    @Command(description="Set the read timeout")
    public void setTimeout(@Argument(description="The timeout (ms)") int timeout)
    {
        srp.setReadTimeout(timeout);
    }

    @Command(description="Show the read timeout")
    public int showTimeout()
    {
        return srp.getReadTimeout();
    }

    @Command(description="Set the read warning time")
    public void setWarnTime(@Argument(description="The time (ms)") int time)
    {
        srp.setReadWarning(time);
    }

    @Command(description="Show the read warning time")
    public int showWarnTime()
    {
        return srp.getReadWarning();
    }

    @Command(description="Show the timeout count")
    public int showNumTimeout()
    {
        return srp.getNumTimeout();
    }

    @Command(description="Show the sequence error count")
    public int showNumSeqErr()
    {
        return srp.getNumSeqErr();
    }

    @Command(description="Read a register")
    public String read(@Argument(description="Register address") int addr) throws DriverException
    {
        return String.format("%08x", srp.readReg(addr));
    }

    @Command(description="Reads registers")
    public String read(@Argument(description="Register address") int addr,
                       @Argument(description="Register count") int count) throws DriverException
    {
        return formatHex(addr, srp.readRegs(addr, count));
    }

    @Command(description="Write a register")
    public void write(@Argument(description="Register address") int addr,
                      @Argument(description="Register values") int... value) throws DriverException
    {
        srp.writeRegs(addr, value);
    }

    @Command(description="Update a register")
    public void update(@Argument(description="Register address") int addr,
                       @Argument(description="Mask of bits to change") int mask,
                       @Argument(description="Register value") int value) throws DriverException
    {
        srp.updateReg(addr, mask, value);
    }

    @Command(description="Continuously read a set of registers")
    public String contRead(@Argument(description="First register address") int addr,
                           @Argument(description="Number of registers") int count) throws DriverException
    {
        System.out.println("Press any key to terminate...");
        boolean[] done = {false};
        consThread.start(done);
        int nRead = 0;
        long startTime = System.currentTimeMillis();
        while (!done[0]) {
            srp.readRegs(addr, count);
            nRead += count;
        }
        return formatRate(startTime, nRead);
    }

    protected static String formatHex(int addr, int[] data)
    {
        StringBuilder text = new StringBuilder();
        for (int j = 0; j < data.length; j++) {
            if ((j & 3) == 0) {
                if (j > 0) {
                    text.append('\n');
                }
                text.append(String.format("%06x:", addr + j));
            }
            text.append(String.format(" %08x", data[j]));
        }
        return text.toString();
    }

    protected static String formatRate(long start, int count)
    {
        long period = System.currentTimeMillis() - start;
        return String.format("Read rate (Hz) = %.4g", count * 1000.0 / (period == 0 ? 1 : period));
    }

}
