package org.lsst.ccs.drivers.auxelex;

import java.net.InetAddress;
import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Routines for controlling a REB power supply
 *
 *  @author  Owen Saxton
 */
@SuppressWarnings("PointlessBitwiseExpression")
public final class RebPS extends Srp {

    /**
     *  Constants and data.
     */
    public static final int
        VERSION_PROTO = 0,
        VERSION_PROD = 1,

        TYPE_UNKNOWN = -1,
        TYPE_SCIENCE = 0,
        TYPE_CORNER  = 1,

        PS_DIGITAL  = 0,
        PS_ANALOG   = 1,
        PS_OD       = 2,
        PS_CLK_HIGH = 3,
        PS_CLK_LOW  = 4,
        PS_HEATER   = 5,
        PS_DPHI     = 5,
        PS_HV_BIAS  = 6,
        NUM_PS      = 7,

        CHAN_VOLT_BEF_LDO = 0,
        CHAN_CURR_BEF_LDO = 1,
        CHAN_VOLT_AFT_LDO = 2,
        CHAN_CURR_AFT_LDO = 3,
        CHAN_VOLT_AFT_SW  = 4,
        CHAN_VOLT2_AFT_LDO = 5,
        CHAN_VOLT_DAC = -1,
        NUM_CHAN_EXT = 3,
        EXT_VALUE_CURR = 0,
        EXT_VALUE_MAX  = 1,
        EXT_VALUE_MIN  = 2,
        EXT_LIMIT_HIGH = 3,
        EXT_LIMIT_LOW  = 4,
        NUM_EXT_VALUE  = 5,

        DAC_MAXIMUM = 4095,

        REB_QUANTUM = 3,  // Used for CR assignment
        MAX_REBS = 2 * REB_QUANTUM;

    private static final int
        DEVC_LTC2945      = 0,
        DEVC_MAX11644     = 1,
        DEVC_MAX5803      = 2,

        REG_FPGA_VERSION  = 0x00,
        REG_SCRATCH       = 0x01,
        REG_SERIAL_NO     = 0x02,
        REG_GEOG_ADDR     = 0x04,
        REG_SPARE0        = 0x05,

        REG_INIT_STATUS   = 0x10,
        STATUS_CR         = 0x80000000,
        STATUS_M_REBS     = 0x3f,
        STATUS_V_INIT_FAIL = 0,
        STATUS_V_INIT_DONE = 8,
        STATUS_V_BIAS_WAIT = 16,
        STATUS_V_CNFG_DONE = 24,
        REG_FAIL_SUMMARY  = 0x11,
        REG_REB_FAIL_BASE = 0x12,

        REG_ENABLE_WRITE    = 0x20,
        REG_ENABLE_POWER_ON = 0x21,
        REG_ENABLE_UNUSED   = 0x22,
        ENABLE_KEY          = 0xdeadbeef,
        REG_ENABLE_STATE    = 0x23,
        ENABLE_V_NOWRITE    = 0,
        ENABLE_V_POWER_ON   = 1,

        REG_DS75LV_WRITE   = 0xe0,
        REG_DS75LV_READ    = 0xe1,

        REG_THRESHOLD      = 0x100,
        REG_REB_BASE       = 0x10000,
        REG_REB_INCR       = 0x10000,
        REG_PS_BASE        = REG_REB_BASE,
        REG_PS_INCR        = 0x1000,
        REG_DEV_INCR       = 0x100,
        REG_REB_DATA_BASE      = REG_REB_BASE + 0x7000,
        REG_REB_RDO_PERIOD     = 0x00,
        REG_REB_RDO_CONFIG     = 0x01,
        REG_REB_RDO_COUNT      = 0x03,
        REG_REB_RDO_DIAG_CNT   = 0x04,
        REG_REB_CHAN_BASE  = REG_REB_DATA_BASE + 0x80,

        SEQ_TIMEOUT = 5000;  // ms

    private static final int UG480_ADDRESS = 0x80080;
    private static final int UG480_TEMPERATURE_OFFSET = 0;
    private static final int UG480_MAX_TEMPERATURE_OFFSET = 0x20;
    private static final int UG480_MIN_TEMPERATURE_OFFSET = 0x24;
    
    private static final List<BoardType> validTypes = new ArrayList<>();
    static {
        validTypes.add(BoardType.REB_PS_PROTO);
        validTypes.add(BoardType.REB_PS_UPDATE);
        validTypes.add(BoardType.REB_PS_PROD_SR);
        validTypes.add(BoardType.REB_PS_PROD_CR);
        validTypes.add(BoardType.SIMULATED);
        validTypes.add(BoardType.UNKNOWN);
    }

    private static final int
        ADDR_REB_PS_PROTO_0 = 20,
        ADDR_REB_PS_PROTO_1 = 21,
        ADDR_REB_PS_UPDATE = 22,
        ADDR_REB_PS_PROD_MIN = 39,
        ADDR_REB_PS_PROD_MAX = 62,
        ADDR_REB_PS_PROD_EXTRA = 76;

    private static final int[][]
        chanDesc =   {{0x01, 0x00, 0x02, 0x10, 0x11},
                      {0x01, 0x00, 0x02, 0x10, 0x11},
                      {0x01, 0x00, 0x02, 0x10, 0x12, 0x11},
                      {0x01, 0x00, 0x02, 0x10, 0x11},
                      {0x01, 0x00, 0x02, 0x10, 0x12, 0x11},
                      {0x01, 0x00, 0x02, 0x10, 0x11, 0x20},
                      {0x01, 0x00, 0x10}},
        devcDesc =   {{DEVC_LTC2945, DEVC_MAX11644},
                      {DEVC_LTC2945, DEVC_MAX11644},
                      {DEVC_LTC2945, DEVC_LTC2945},
                      {DEVC_LTC2945, DEVC_MAX11644},
                      {DEVC_LTC2945, DEVC_LTC2945},
                      {DEVC_LTC2945, DEVC_MAX11644, DEVC_MAX5803},
                      {DEVC_LTC2945, DEVC_MAX5803}};
    private static final int[]
        dacChannel = {-1, -1, -1, -1, -1, 5, 2};
    private static final double[][]
        convConstProto = {{0.0250, 0.000490,  0.00352, 0.00140,   0.00305},
                          {0.0250, 0.000490,  0.00352, 0.00140,   0.00305},
                          {0.0250, 0.0000641, 0.0206,  0.0000641, 0.0206,  0.0250},
                          {0.0250, 0.000125,  0.00654, 0.000617,  0.006},
                          {0.0250, 0.000758,  0.00352, 0.000531,  0.00050, 0.0250},
                          {0.0250, 0.000125,  0.00654, 0.000750,  0.00600, 1.0},
                          {0.0250, 0.000000296, 1.0}};
    private static final double[][]
        convConstProdSR = {{0.0250, 0.000490,  0.00352, 0.001399,  0.00305},
                           {0.0250, 0.000470,  0.00352, 0.002004,  0.00305},
                           {0.0250, 0.0000758, 0.0206,  0.0000758, 0.0206,  0.0250},
                           {0.0250, 0.000158,  0.00655, 0.000826,  0.006},
                           {0.0250, 0.0001923, 0.00352, 0.0001923, 0.00050, 0.0250},
                           {0.0250, 0.000158,  0.00654, 0.000826,  0.00600, 1.0},
                           {0.0250, 0.000000753, 1.0}};
    private static final double[][]
        convConstProdCR = {{0.0250, 0.000490,  0.00352, 0.001399,  0.00305},
                           {0.0250, 0.000470,  0.00352, 0.002004,  0.00305},
                           {0.0250, 0.0000758, 0.0206,  0.0000758, 0.0206,  0.0250},
                           {0.0250, 0.000158,  0.00655, 0.000340,  0.006},
                           {0.0250, 0.000125,  0.00352, 0.000125,  0.00050, 0.0250},
                           {0.0250, 0.000158,  0.00654, 0.000699,  0.00600, 1.0},
                           {0.0250, 0.000000753, 1.0}};
    private static final double[]
        convConstProdCRHtr = {0.0250, 0.0001923,  0.00352, 0.000826,  0.00305};

    private static final List<Long> oldUnits = new ArrayList<>();
    static {
        oldUnits.add(0x6af054000af5c200L);  // LPNHE
        oldUnits.add(0x006af054000af5c2L);  // LPNHE
        oldUnits.add(0x6af0540060f5c200L);  // BNL
        oldUnits.add(0x006af0540060f5c2L);  // BNL
        oldUnits.add(0x6af054005cf5c200L);  // Penn
        oldUnits.add(0x006af054005cf5c2L);  // Penn
    }

    private static final int
        ERR_PS_NONE    = -1,
        ERR_PS_DIGITAL = 0,
        ERR_PS_ANALOG  = 1,
        ERR_PS_OD      = 2,
        ERR_PS_CLK_HI  = 3,
        ERR_PS_CLK_LO  = 4,
        ERR_PS_HEATER  = 5,
        ERR_PS_DPHI    = 6,
        ERR_PS_HV_BIAS = 7,
        NUM_ERR_PS     = 8,
        ERR_TYPE_VOLT_LOW  = 0,
        ERR_TYPE_VI_RANGE  = 1,
        ERR_TYPE_TEMP_HIGH = 2,
        ERR_TYPE_TEMP_INIT = 3,
        ERR_TYPE_PS_INIT   = 4,
        ERR_TYPE_UNDEFINED = 5,
        NUM_ERR_TYPE       = 6,
        NUM_ERR_BITS = 18;
    private static final String[] ErrPsDescs = new String[NUM_ERR_PS];
    static {
        ErrPsDescs[ERR_PS_DIGITAL] = "Digital";
        ErrPsDescs[ERR_PS_ANALOG] = "Analog";
        ErrPsDescs[ERR_PS_OD] = "OD";
        ErrPsDescs[ERR_PS_CLK_HI] = "ClockHigh";
        ErrPsDescs[ERR_PS_CLK_LO] = "ClockLow";
        ErrPsDescs[ERR_PS_HEATER] = "Heater";
        ErrPsDescs[ERR_PS_DPHI] = "Dphi";
        ErrPsDescs[ERR_PS_HV_BIAS] = "HvBias";
    }
    private static final String[] ErrTypeDescs = new String[NUM_ERR_TYPE];
    static {
        ErrTypeDescs[ERR_TYPE_VOLT_LOW] = "voltage too low";
        ErrTypeDescs[ERR_TYPE_VI_RANGE] = "voltage or current out of range";
        ErrTypeDescs[ERR_TYPE_TEMP_HIGH] = "Temperature too high";
        ErrTypeDescs[ERR_TYPE_TEMP_INIT] = "Temperature alarm didn't initialize";
        ErrTypeDescs[ERR_TYPE_PS_INIT] = "A power supply didn't initialize";
        ErrTypeDescs[ERR_TYPE_UNDEFINED] = "Undefined error";
    }
    private static final int[] ErrBitDescsSR = new int[NUM_ERR_BITS];
    static {
        ErrBitDescsSR[0] = ERR_TYPE_VOLT_LOW | (ERR_PS_DIGITAL << 8);
        ErrBitDescsSR[1] = ERR_TYPE_VI_RANGE | (ERR_PS_DIGITAL << 8);
        ErrBitDescsSR[2] = ERR_TYPE_VOLT_LOW | (ERR_PS_ANALOG << 8);
        ErrBitDescsSR[3] = ERR_TYPE_VI_RANGE | (ERR_PS_ANALOG  << 8);
        ErrBitDescsSR[4] = ERR_TYPE_UNDEFINED | (ERR_PS_NONE << 8);
        ErrBitDescsSR[5] = ERR_TYPE_VI_RANGE | (ERR_PS_OD << 8);
        ErrBitDescsSR[6] = ERR_TYPE_VOLT_LOW | (ERR_PS_CLK_HI << 8);
        ErrBitDescsSR[7] = ERR_TYPE_VI_RANGE | (ERR_PS_CLK_HI << 8);
        ErrBitDescsSR[8] = ERR_TYPE_VOLT_LOW | (ERR_PS_CLK_LO << 8);
        ErrBitDescsSR[9] = ERR_TYPE_VI_RANGE | (ERR_PS_CLK_LO << 8);
        ErrBitDescsSR[10] = ERR_TYPE_VOLT_LOW | (ERR_PS_HEATER << 8);
        ErrBitDescsSR[11] = ERR_TYPE_VI_RANGE | (ERR_PS_HEATER << 8);
        ErrBitDescsSR[12] = ERR_TYPE_VI_RANGE | (ERR_PS_HV_BIAS << 8);
        ErrBitDescsSR[13] = ERR_TYPE_UNDEFINED | (ERR_PS_NONE << 8);
        ErrBitDescsSR[14] = ERR_TYPE_UNDEFINED | (ERR_PS_NONE << 8);
        ErrBitDescsSR[15] = ERR_TYPE_TEMP_HIGH | (ERR_PS_NONE << 8);
        ErrBitDescsSR[16] = ERR_TYPE_PS_INIT | (ERR_PS_NONE << 8);
        ErrBitDescsSR[17] = ERR_TYPE_TEMP_INIT | (ERR_PS_NONE << 8);
    }
    private static final int[] ErrBitDescsCR = new int[NUM_ERR_BITS];
    static {
        System.arraycopy(ErrBitDescsSR, 0, ErrBitDescsCR, 0, NUM_ERR_BITS);
        ErrBitDescsCR[10] = ERR_TYPE_VOLT_LOW | (ERR_PS_DPHI << 8);
        ErrBitDescsCR[11] = ERR_TYPE_VI_RANGE | (ERR_PS_DPHI << 8);
        ErrBitDescsCR[13] = ERR_TYPE_VOLT_LOW | (ERR_PS_HEATER << 8);
        ErrBitDescsCR[14] = ERR_TYPE_VI_RANGE | (ERR_PS_HEATER << 8);
    }

    private int numRebs, numTemps;
    private int psVersion;
    private double clockLoAftSwCoeff, clockLoAftSwCoeff2;
    private static double[][] convConst = convConstProto;
    private final Object tempSync = new Object();


    /**
     *  Constructor
     */
    public RebPS(String logName)
    {
        super(logName);
        setValidBoardTypes(validTypes);
        setReadTimeout(10000);
    }


    /**
     *  Opens a connection to a board.
     *
     *  There are three different kinds of board to be handled, distinguished
     *  by the last digit of their IP address:
     *
     *   20 or 21: Prototype with original firmware (3 REBs)
     *   22: Prototype with new firmware (sequencing, new network protocol)
     *   Any other value: Production (with new firmware, 6 REBs)
     *
     *  @param  ipAddr  The IP address, or null or empty for simulation
     *  @param  port    The port number
     *  @throws  DriverException
     */
    @Override
    @SuppressWarnings("UseSpecificCatch")
    public synchronized void open(String ipAddr, int port) throws DriverException
    {
        int addr = 0;
        setSrpVersion(3);
        try {
            addr = InetAddress.getByName(ipAddr).getAddress()[3];
            if (addr == ADDR_REB_PS_PROTO_0 || addr == ADDR_REB_PS_PROTO_1) {
                setSrpVersion(1);
            }
        }
        catch (Exception e) {
            // It'll get handled in super.open
        }
        super.open(ipAddr, port);
        if (boardType == BoardType.UNKNOWN) {
            if (addr == ADDR_REB_PS_PROTO_0 || addr == ADDR_REB_PS_PROTO_1) {
                boardType = BoardType.REB_PS_PROTO;
            }
            else if (addr == ADDR_REB_PS_UPDATE) {
                boardType = BoardType.REB_PS_UPDATE;
            }
            else if (addr >= ADDR_REB_PS_PROD_MIN && addr <= ADDR_REB_PS_PROD_MAX
                       || addr == ADDR_REB_PS_PROD_EXTRA) {
                boardType = BoardType.REB_PS_PROD_SR;
            }
            else {
                close();
                throw new DriverException("Unable to determine board type");
            }
        }
        if (boardType == BoardType.REB_PS_PROD_SR || boardType == BoardType.REB_PS_PROD_CR || boardType == BoardType.SIMULATED) {
            psVersion = VERSION_PROD;
            numRebs = 2 * REB_QUANTUM;
            numTemps = 7;
            convConst = boardType == BoardType.REB_PS_PROD_SR ? convConstProdSR : convConstProdCR;
            clockLoAftSwCoeff = 17.15;
            clockLoAftSwCoeff2 = 3.0;
        }
        else {
            psVersion = boardType == BoardType.REB_PS_UPDATE ? VERSION_PROD : VERSION_PROTO;
            numRebs = REB_QUANTUM;
            numTemps = 1;
            convConst = convConstProto;
            clockLoAftSwCoeff = oldUnits.contains(getSerialNo()) ? 13.1 : 21.1;
            clockLoAftSwCoeff2 = 1.0;
        }
    }


    /**
     *  Gets the PS firmware version
     * 
     *  @return  The PS firmware version (prototype or production)
     */
    public int getVersion()
    {
        return psVersion;
    }


    /**
     *  Gets the number of REBs
     * 
     *  @return  The number of REBs (3: prototype board, 6: production board)
     */
    public int getNumRebs()
    {
        return numRebs;
    }


    /**
     *  Gets the number of temperatures
     * 
     *  @return  The number of temperatures (1: prototype board, 7: production board)
     */
    public int getNumTemperatures()
    {
        return numTemps;
    }


    /**
     *  Gets the PS type
     * 
     *  @return  The type (unknown, science, corner)
     *  @throws  DriverException
     */
    public int getType() throws DriverException
    {
        return psVersion == VERSION_PROTO ? TYPE_UNKNOWN :
               (readReg(REG_INIT_STATUS) & STATUS_CR) == 0 ? TYPE_SCIENCE : TYPE_CORNER;
    }


    /**
     *  Gets the firmware version number.
     *
     *  @return  The version number
     *  @throws  DriverException
     */
    public int getFwVersion() throws DriverException
    {
        return readReg(REG_FPGA_VERSION);
    }


    /**
     *  Gets the serial number.
     *
     *  @return  The serial number
     *  @throws  DriverException
     */
    public long getSerialNo() throws DriverException
    {
        int[] value = readRegs(REG_SERIAL_NO, 2);
        return ((long)value[1] << 32) | (value[0] & 0xffffffffL);
    }


    /**
     *  Gets the geographical address.
     *
     *  @return  The geographical address
     *  @throws  DriverException
     */
    public int getGeogAddr() throws DriverException
    {
        return readReg(REG_GEOG_ADDR);
    }


    /**
     *  Gets the initialization status.
     *
     *  @return  The initialization status
     *  @throws  DriverException
     */
    public int getInitStatus() throws DriverException
    {
        return psVersion == VERSION_PROD ? readReg(REG_INIT_STATUS) : 0;
    }


    /**
     *  Gets the failure summary.
     *
     *  @return  The failure summary
     *  @throws  DriverException
     */
    public int getFailureSummary() throws DriverException
    {
        return psVersion == VERSION_PROD ? readReg(REG_FAIL_SUMMARY) : 0;
    }


    /**
     *  Gets the failure details for a REB.
     *
     *  @param  rebNum  The REB number
     *  @return  The failure detail
     *  @throws  DriverException
     */
    public int getFailureDetail(int rebNum) throws DriverException
    {
        checkRebNumber(rebNum);
        return psVersion == VERSION_PROD ? readReg(REG_REB_FAIL_BASE + rebNum) : 0;
    }


    /**
     *  Gets the failure detail string for a REB.
     *
     *  @param  rebNum  The REB number
     *  @return  The failure details, formatted as a string
     *  @throws  DriverException
     */
    public String getFailureDetailString(int rebNum) throws DriverException
    {
        int value = getFailureDetail(rebNum);
        StringBuilder text = new StringBuilder(String.format("0x%08x", value));
        int initLeng = text.length();
        int[] errBitDescs = getType() == TYPE_SCIENCE ? ErrBitDescsSR : ErrBitDescsCR;
        for (int bNum = 0; bNum < NUM_ERR_BITS; bNum++) {
            if ((value & (1 << bNum)) == 0) continue;
            int errorDesc = errBitDescs[bNum];
            int errorPS = errorDesc >> 8;
            text.append(text.length() == initLeng ? " (" : ", ");
            if (errorPS != ERR_PS_NONE) {
                text.append(ErrPsDescs[errorPS]).append(" ");
            }
            text.append(ErrTypeDescs[errorDesc & 0xff]);
        }
        if (text.length() != initLeng) {
            text.append(")");
        }
        return text.toString();
    }


    /**
     *  Configures the temperature system.
     *
     *  @throws  DriverException
     */
    public void configTemperature() throws DriverException
    {
        synchronized (tempSync) {
            if (psVersion == VERSION_PROD) {
                for (int j = 0; j < numTemps; j++) {
                    writeReg(REG_DS75LV_WRITE, 0x80000000 | (j << 20));
                    try {
                        getTemperatureRes(j);
                    }
                    catch (DriverException e) {
                        // Why not log or throw?
                    }
                }
            }
            else {
                setTemperatureRes(3);
            }
        }
    }


    /**
     *  Sets the temperature resolution for a sensor.
     *
     *  This works only if register writing has been enabled
     *
     *  @param  sensor  The sensor number (0 - 6)
     *  @param  value  The resolution value (0 - 3)
     *  @throws  DriverException
     */
    public void setTemperatureRes(int sensor, int value) throws DriverException
    {
        synchronized (tempSync) {
            checkTempSensor(sensor);
            writeDS75LV(sensor, 1, (value & 0x03) << 5);
            if (psVersion == VERSION_PROD) {
                getTemperatureRes(sensor);
            }
        }
    }


    /**
     *  Sets the temperature resolution for all sensors.
     *
     *  This works only if register writing has been enabled
     *
     *  @param  value  The resolution value (0 - 3)
     *  @throws  DriverException
     */
    public void setTemperatureRes(int value) throws DriverException
    {
        for (int j = 0; j < numTemps; j++) {
            try {
                setTemperatureRes(j, value);
            }
            catch (DriverException e) {
                // Why not log or throw?
            }
        }
    }


    /**
     *  Gets a temperature resolution.
     *
     *  @param  sensor  The sensor number (0 - 6)
     *  @return  The resolution (0 - 3), or -1 if hardware error
     *  @throws  DriverException
     */
    public int getTemperatureRes(int sensor) throws DriverException
    {
        synchronized (tempSync) {
            checkTempSensor(sensor);
            readDS75LV(sensor, 1);
            int value = waitDS75LV();
            return value < 0 ? -1 : (value >> 13) & 0x03;
        }
    }


    /**
     *  Gets all temperature resolutions.
     *
     *  @return  An array of resolutions, in sensor order
     *  @throws  DriverException
     */
    public int[] getTemperatureRes() throws DriverException
    {
        int[] values = new int[numTemps];
        for (int sensor = 0; sensor < numTemps; sensor++) {
            values[sensor] = getTemperatureRes(sensor);
        }
        return values;
    }


    /**
     *  Reads a temperature.
     *
     *  @param  sensor  The sensor number (0 - 6)
     *  @return  The temperature (C), or NaN if hardware error occurred
     *  @throws  DriverException if communications error
     */
    public double readTemperature(int sensor) throws DriverException
    {
        return readTemp(sensor, 0);
    }


    /**
     *  Reads all temperatures.
     *
     *  @return  An array of read temperatures (C) in sensor order
     *  @throws  DriverException
     */
    public double[] readTemperature() throws DriverException
    {
        return readTemp(0);
    }


    /**
     *  Reads all warning temperature limits.
     *
     *  @return  An array of read temperatures (C) in sensor order
     *  @throws  DriverException
     */
    public double[] readTempWarn() throws DriverException
    {
        return readTemp(2);
    }


    /**
     *  Reads all error temperature limits.
     *
     *  @return  An array of read temperatures (C) in sensor order
     *  @throws  DriverException
     */
    public double[] readTempError() throws DriverException
    {
        return readTemp(3);
    }


    /**
     *  Reads a measured or limit temperature.
     *
     *  @param  sensor  The sensor number (0 - 6)
     *  @param  addr  The DS75LV register address (0, 2 or 3)
     *  @return  The temperature (C), or NaN if hardware error occurred
     *  @throws  DriverException if communications error
     */
    private double readTemp(int sensor, int addr) throws DriverException
    {
        synchronized (tempSync) {
            checkTempSensor(sensor);
            readDS75LV(sensor, addr);
            int value = waitDS75LV();
            return value < 0 ? Double.NaN : ((value << 16) >> 16) / 256.0;   // Preserve 16-bit sign
        }
    }


    /**
     *  Reads all measured or limit temperatures.
     *
     *  @param  addr  The DS75LV register address (0, 2 or 3)
     *  @return  An array of read temperatures (C) in sensor order
     *  @throws  DriverException
     */
    private double[] readTemp(int addr) throws DriverException
    {
        double[] values = new double[numTemps];
        for (int sensor = 0; sensor < numTemps; sensor++) {
            values[sensor] = readTemp(sensor, addr);
        }
        return values;
    }


    /**
     *  Writes to a DS75LV temperature sensor.
     *
     *  This works only if register writing has been enabled
     *
     *  @param  sensor  The sensor number (0 - 6)
     *  @param  reg     The register number (0 - 3)
     *  @param  value   The value
     *  @throws  DriverException
     */
    private void writeDS75LV(int sensor, int reg, int value) throws DriverException
    {
        boolean oneByte = reg == 1;
        writeReg(REG_DS75LV_WRITE, (sensor << 20) | (oneByte ? 0x40000 : 0xc0000) | (reg << 16)
                                     | ((oneByte ? value << 8 : value) & 0xffff));
    }


    /**
     *  Initiates a read from a DS75LV temperature sensor.
     *
     *  This works only if register writing has been enabled
     *
     *  @param  sensor  The sensor number (0 - 6)
     *  @param  reg     The register number (0 - 3)
     *  @throws  DriverException
     */
    private void readDS75LV(int sensor, int reg) throws DriverException
    {
        boolean oneByte = reg == 1;
        writeReg(REG_DS75LV_WRITE, (sensor << 20) | (oneByte ? 0 : 0x80000) | (reg << 16));
    }


    /**
     *  Waits for a DS75LV read to complete.
     *
     *  @return  The unsigned 16-bit read value or -1 if hardware error
     *  @throws  DriverException
     */
    @SuppressWarnings({"SleepWhileInLoop", "SleepWhileHoldingLock"})
    private int waitDS75LV() throws DriverException
    {
        synchronized (tempSync) {
            int value = 0;
            for (int count = 0; count < 100; count++) {
                try {
                    Thread.sleep(1);   // Must sleep before first read
                }
                catch (InterruptedException e) {
                }
                value = readReg(REG_DS75LV_READ);
                if ((value & 0xc0000000) != 0) break;
            }
            return (value & 0x80000000) == 0 ? -1 : value & 0xffff;
        }
    }


    /**
     *  Sets the power options for a REB.
     *
     *  @param  rebNum  The REB number
     *  @param  value   The value to set
     *  @throws  DriverException
     */
    @SuppressWarnings("SleepWhileInLoop")
    public void setPower(int rebNum, int value) throws DriverException
    {
        checkRebNumber(rebNum);
        int shift = 8 * (rebNum < REB_QUANTUM ? rebNum + 1 : rebNum - 2);
        int powerReg = rebNum < REB_QUANTUM ? REG_SPARE0 : REG_SCRATCH;
        int prevValue = updateReg(powerReg, 0xff << shift, value << shift) >> shift;
        boolean master = (value & 1) != 0, prevMaster = (prevValue & 1) != 0;
        if (master ^ prevMaster) {
            if (psVersion == VERSION_PROD) {  // Wait for sequencing completion
                int mask = ((1 << STATUS_V_INIT_FAIL) | (1 << STATUS_V_BIAS_WAIT)) << rebNum;
                long endTime = System.currentTimeMillis() + SEQ_TIMEOUT;
                boolean isOn = !master;
                if ( isSimulated() ) {
                    return;
                }
                while ((master ^ isOn) && System.currentTimeMillis() < endTime) {
                    try {
                        Thread.sleep(10);
                    }
                    catch (InterruptedException e) {}
                    isOn = (mask & getInitStatus()) != 0 || (master && (mask & getFailureSummary()) != 0);
                }
                String errMsg = null;
                if (master ^ isOn) {
                    errMsg = "timed out";
                }
                else if (master) {
                    mask = 1 << (STATUS_V_INIT_FAIL + rebNum);
                    if ((mask & getInitStatus()) != 0 || (mask & getFailureSummary()) != 0) {
                        errMsg = "failed: " + getFailureDetailString(rebNum);
                    }
                }
                if (errMsg != null) {
                    throw new DriverException("REB " + rebNum + " power " + (master ? "on" : "off") + " sequencing " + errMsg);
                }
            }
            else if (master) {
                try {
                    Thread.sleep(25);
                }
                catch (InterruptedException e) {
                }
                configure(rebNum);  // MAXIM devices need this every time
            }
        }
    }


    /**
     *  Gets the power options for a REB.
     *
     *  @param  rebNum  The REB number
     *  @return  The bit mask of power options
     *  @throws  DriverException
     */
    public int getPower(int rebNum) throws DriverException
    {
        checkRebNumber(rebNum);
        int shift = 8 * (rebNum < REB_QUANTUM ? rebNum + 1 : rebNum - 2);
        int powerReg = rebNum < REB_QUANTUM ? REG_SPARE0 : REG_SCRATCH;
        return (readReg(powerReg) >> shift) & 0xff;
    }


    /**
     *  Gets the overall powered state of a REB.
     *
     *  @param  rebNum  The REB number
     *  @return  The power state
     *  @throws  DriverException
     */
    public boolean isPowerOn(int rebNum) throws DriverException
    {
        checkRebNumber(rebNum);
        return (getPower(rebNum) & 1) != 0;
    }


    /**
     *  Gets the powered state of a single REB PS element.
     *
     *  @param  rebNum  The REB number
     *  @param  psNum   The power supply number
     *  @return  The power state
     *  @throws  DriverException
     */
    public boolean isPowerOn(int rebNum, int psNum) throws DriverException
    {
        checkRebNumber(rebNum);
        checkPsNumber(psNum);
        return (getPower(rebNum) & (1 << (psNum + 1))) != 0;
    }


    /**
     *  Configures a device.
     *
     *  @param  rebNum  The REB number
     *  @param  psNum   The power supply number
     *  @param  devc    The device number
     *  @throws  DriverException
     */
    public void configure(int rebNum, int psNum, int devc) throws DriverException
    {
        if (psVersion == VERSION_PROD) return;  // Now done automatically
        checkRebNumber(rebNum);
        checkPsNumber(psNum);
        int addr = getDevcBaseAddress(rebNum, psNum, devc);
        switch (getDeviceType(psNum, devc)) {
        case DEVC_LTC2945:
            break;
        case DEVC_MAX11644:
            writeReg(addr, 0x03dadada);
            break;
        case DEVC_MAX5803:
            writeReg(addr + 0x26, 0);
            break;
        }
    }


    /**
     *  Configures all devices of a power supply.
     *
     *  @param  rebNum  The REB number
     *  @param  psNum   The power supply number
     *  @throws  DriverException
     */
    public void configure(int rebNum, int psNum) throws DriverException
    {
        for (int devc = 0; devc < getNumDevices(psNum); devc++) {
            configure(rebNum, psNum, devc);
        }
    }


    /**
     *  Configures all devices of a REB.
     *
     *  @param  rebNum  The REB number
     *  @throws  DriverException
     */
    public void configure(int rebNum) throws DriverException
    {
        for (int psNum = 0; psNum < NUM_PS; psNum++) {
            configure(rebNum, psNum);
        }
    }


    /**
     *  Reads a channel.
     *
     *  @param  rebNum  The REB number
     *  @param  psNum   The power supply number
     *  @param  chan    The channel number
     *  @return  The value of the channel data
     *  @throws  DriverException
     */
    public double readChannel(int rebNum, int psNum, int chan) throws DriverException
    {
        double value = readChanDirect(rebNum, psNum, chan);
        if (psNum == PS_CLK_LOW && (chan == CHAN_VOLT2_AFT_LDO || chan == CHAN_VOLT_AFT_SW)) {
            double mult = (chan == CHAN_VOLT2_AFT_LDO) ? 1 : clockLoAftSwCoeff;
            double scale = (chan == CHAN_VOLT2_AFT_LDO) ? 1 : clockLoAftSwCoeff2;
            value = mult * (scale * value - readChanDirect(rebNum, psNum, CHAN_VOLT_AFT_LDO));
        }
        return value;
    }


    /**
     *  Reads a channel.
     *
     *  @param  rebNum  The REB number
     *  @param  psNum   The power supply number
     *  @param  chan    The channel number
     *  @return  The value of the channel data
     *  @throws  DriverException
     */
    private double readChanDirect(int rebNum, int psNum, int chan) throws DriverException
    {
        checkRebNumber(rebNum);
        checkPsNumber(psNum);
        chan = convChannel(psNum, chan);
        int desc = getChanDesc(psNum, chan);
        int devc = (desc >> 4) & 0x0f, dChan = desc & 0x0f;
        int addr = getDevcBaseAddress(rebNum, psNum, devc);
        int value = 0;
        switch (getDeviceType(psNum, devc)) {
        case DEVC_LTC2945:
            addr += 0x14 + 0x0a * dChan;
            int[] data = readRegs(addr, 2);
            value = ((data[0] & 0xff) << 4) | ((data[1] & 0xff) >> 4);
            break;
        case DEVC_MAX11644:
            value = (readReg(addr) >> (16 * (1 - dChan))) & 0xfff;
            break;
        case DEVC_MAX5803:
            value = (readReg(addr + 0xa0) >> 4) & 0xfff;
            break;
        }
        return value * getConversion(rebNum, psNum, chan);
    }


    /**
     *  Reads all channels of a power supply.
     *
     *  @param  rebNum  The REB number
     *  @param  psNum   The power supply number
     *  @return  The array of power supply values
     *  @throws  DriverException
     */
    public double[] readChannel(int rebNum, int psNum) throws DriverException
    {
        checkRebNumber(rebNum);
        int nChan = getNumChannels(psNum);
        double[] values = new double[nChan];
        for (int chan = 0; chan < nChan; chan++) {
            values[chan] = readChannel(rebNum, psNum, chan);
        }
        return values;
    }


    /**
     *  Reads extended channel data.
     *
     *  @param  rebNum  The REB number
     *  @param  psNum   The power supply number
     *  @param  chan    The channel number
     *  @return  The array of extended channel data
     *  @throws  DriverException
     */
    public double[] readChanExtended(int rebNum, int psNum, int chan) throws DriverException
    {
        checkRebNumber(rebNum);
        checkPsNumber(psNum);
        chan = convChannel(psNum, chan);
        int desc = getChanDesc(psNum, chan);
        int devc = (desc >> 4) & 0x0f, dChan = desc & 0x0f;
        if (getDeviceType(psNum, devc) != DEVC_LTC2945) {
            throw new DriverException("Channel has no extended data available");
        }
        int addr = getDevcBaseAddress(rebNum, psNum, devc) + 0x14 + 0x0a * dChan;
        int[] data = readRegs(addr, 2 * NUM_EXT_VALUE);
        double[] value = new double[NUM_EXT_VALUE];
        double conv = getConversion(rebNum, psNum, chan);
        for (int j = 0; j < data.length; j += 2) {
            value[j / 2] = conv * (((data[j] & 0xff) << 4) | ((data[j + 1] & 0xff) >> 4));
        }
        return value;
    }


    /**
     *  Reads all extended channel data for a power supply.
     *
     *  @param  rebNum  The REB number
     *  @param  psNum   The power supply number
     *  @return  The array of extended channel data
     *  @throws  DriverException
     */
    public double[][] readChanExtended(int rebNum, int psNum) throws DriverException
    {
        checkRebNumber(rebNum);
        checkPsNumber(psNum);
        int nChan = NUM_CHAN_EXT - (psNum == PS_HV_BIAS ? 1 : 0);
        double[][] value = new double[NUM_CHAN_EXT][NUM_EXT_VALUE];
        for (int j = 0; j < nChan; j++) {
            System.arraycopy(readChanExtended(rebNum, psNum, j), 0, value[j], 0, NUM_EXT_VALUE);
        }
        return value;
    }


    /**
     *  Resets channel extrema data.
     *
     *  @param  rebNum  The REB number
     *  @param  psNum   The power supply number
     *  @param  chan    The channel number
     *  @throws  DriverException
     */
    public void resetChanExtrema(int rebNum, int psNum, int chan) throws DriverException
    {
        checkRebNumber(rebNum);
        checkPsNumber(psNum);
        chan = convChannel(psNum, chan);
        int desc = getChanDesc(psNum, chan);
        int devc = (desc >> 4) & 0x0f, dChan = desc & 0x0f;
        if (getDeviceType(psNum, devc) != DEVC_LTC2945) {
            throw new DriverException("Channel has no accessible extended data");
        }
        int addr = getDevcBaseAddress(rebNum, psNum, devc) + 0x14 + 0x0a * dChan + 2 * EXT_VALUE_MAX;
        int[] data = {0, 0, 0xff, 0xf0};
        boolean writeEnabled = isAllWriteEnabled();
        if (!writeEnabled) {
            enableAllWrite(true);
        }
        writeRegs(addr, data);
        if (!writeEnabled) {
            enableAllWrite(false);
        }
    }


    /**
     *  Writes to a DAC.
     *
     *  @param  rebNum  The REB number
     *  @param  psNum   The power supply number
     *  @param  value   The voltage value to write
     *  @throws  DriverException
     */
    public void writeDac(int rebNum, int psNum, double value) throws DriverException
    {
        checkRebNumber(rebNum);
        checkPsNumber(psNum);
        int chan = convChannel(psNum, CHAN_VOLT_DAC);
        int devc = getChanDesc(psNum, chan) >> 4;
        int addr = getDevcBaseAddress(rebNum, psNum, devc);
        int val = (int)(value / getConversion(rebNum, psNum, chan));
        writeReg(addr + 0xa0, (val & 0xfff) << 4);
    }


    /**
     *  Sets whether writing of all registers is enabled.
     *
     *  @param  on  True to enable writing; false to disable it
     *  @throws  DriverException
     */
    public void enableAllWrite(boolean on) throws DriverException
    {
        if (psVersion == VERSION_PROD) {
            writeReg(REG_ENABLE_WRITE, on ? ENABLE_KEY : 0);
        }
    }


    /**
     *  Gets whether writing of all registers is enabled.
     *
     *  @return  Whether writing is enabled
     *  @throws  DriverException
     */
    public boolean isAllWriteEnabled() throws DriverException
    {
        return psVersion == VERSION_PROD ? (readReg(REG_ENABLE_STATE) & (1 << ENABLE_V_NOWRITE)) == 0 : true;
    }


    /**
     *  Sets whether leaving power on after failure is enabled.
     *
     *  @param  on  True to enable power left on; false to disable it
     *  @throws  DriverException
     */
    public void enablePowerLeftOn(boolean on) throws DriverException
    {
        if (psVersion == VERSION_PROD) {
            writeReg(REG_ENABLE_POWER_ON, on ? ENABLE_KEY : 0);
        }
    }


    /**
     *  Gets whether leaving power on after failure is enabled.
     *
     *  @return  Whether leaving power on is enabled
     *  @throws  DriverException
     */
    public boolean isPowerLeftOnEnabled() throws DriverException
    {
        return psVersion == VERSION_PROD ? (readReg(REG_ENABLE_STATE) & (1 << ENABLE_V_POWER_ON)) != 0 : true;
    }


    /**
     *  Tests a channel number for validity.
     *
     *  @param  psNum  The power supply number
     *  @param  chan   The channel number
     *  @return  Whether the channel number is valid
     *  @throws  DriverException
     */
    public static boolean testChannelNumber(int psNum, int chan) throws DriverException
    {
        return chan >= -1 && chan < getNumChannels(psNum);
    }


    /**
     *  Tests a power supply number for validity.
     *
     *  @param  psNum  The power supply number
     *  @return  Whether the power supply number is valid
     */
    public static boolean testPsNumber(int psNum)
    {
        return psNum >= 0 && psNum < NUM_PS;
    }


    /**
     *  Tests a REB number for validity.
     *
     *  @param  rebNum  The REB number
     *  @return  Whether the REB number is valid
     *  @throws  DriverException
     */
    public boolean testRebNumber(int rebNum) throws DriverException
    {
        return rebNum >= 0 && rebNum < numRebs;
    }


    /**
     *  Gets the number of devices in a power supply.
     *
     *  @param  psNum  The power supply number
     *  @return  The number of devices
     *  @throws  DriverException
     */
    private static int getNumDevices(int psNum) throws DriverException
    {
        checkPsNumber(psNum);
        return devcDesc[psNum].length;
    }


    /**
     *  Gets the number of channels in a power supply.
     *
     *  @param  psNum  The power supply number
     *  @return  The number of channels
     *  @throws  DriverException
     */
    private static int getNumChannels(int psNum) throws DriverException
    {
        checkPsNumber(psNum);
        return chanDesc[psNum].length;
    }


    /**
     *  Converts a channel number.
     *
     *  @throws  DriverException
     */
    private static int convChannel(int psNum, int chan) throws DriverException
    {
        if (chan == CHAN_VOLT_DAC) {
            chan = dacChannel[psNum];
        }
        if (chan < 0) {
            throw new DriverException("Power supply has no DAC");
        }
        if (chan >= chanDesc[psNum].length) {
            throw new DriverException("Invalid channel for PS");
        }
        return chan;
    }


    /**
     *  Gets a device type.
     *
     *  @throws  DriverException
     */
    private static int getDeviceType(int psNum, int devc) throws DriverException
    {
        try {
            return devcDesc[psNum][devc];
        }
        catch (IndexOutOfBoundsException e) {
            throw new DriverException("Invalid device index for PS");
        }
    }


     /**
     *  Gets a channel descriptor word.
     *
     *  @throws  DriverException
     */
    private static int getChanDesc(int psNum, int chan) throws DriverException
    {
        return chanDesc[psNum][chan];
    }


    /**
     *  Gets a device base register address.
     *
     *  @throws  DriverException
     */
    private static int getDevcBaseAddress(int rebNum, int psNum, int devc) throws DriverException
    {
        return REG_PS_BASE + REG_REB_INCR * rebNum + REG_PS_INCR * psNum
                 + REG_DEV_INCR * devc;
    }


    /**
     *  Gets a conversion constant.
     *
     *  @throws  DriverException
     */
    private double getConversion(int rebNum, int psNum, int chan) throws DriverException
    {
        if (boardType == BoardType.REB_PS_PROD_CR && psNum == PS_DIGITAL && rebNum % REB_QUANTUM == REB_QUANTUM - 1) {
            return convConstProdCRHtr[chan];
        }
        else {
            return convConst[psNum][chan];
        }
    }


    /**
     *  Checks a REB number for validity.
     *
     *  @throws  DriverException
     */
    private void checkRebNumber(int rebNum) throws DriverException
    {
        if (!testRebNumber(rebNum)) {
            throw new DriverException("Invalid REB number (" + rebNum + ")");
        }
    }


    /**
     *  Checks a power supply number for validity.
     *
     *  @throws  DriverException
     */
    private static void checkPsNumber(int psNum) throws DriverException
    {
        if (!testPsNumber(psNum)) {
            throw new DriverException("Invalid PS number (" + psNum + ")");
        }
    }


    /**
     *  Checks a temperature sensor number for validity.
     *
     *  @throws  DriverException
     */
    private void checkTempSensor(int sensor) throws DriverException
    {
        if (sensor < 0 || sensor >= numTemps) {
            throw new DriverException("Invalid temperature sensor number (" + sensor + ")");
        }
    }


    private static final double[][]
        simValues = {{ 5.0, 0.0,  5.0, 0.0,  5.0},
                     { 7.0, 0.0,  7.0, 0.0,  7.0},
                     {40.0, 0.0, 40.0, 0.0, 40.0, 40.0},
                     {16.0, 0.0, 16.0, 0.0, 16.0},
                     {16.0, 0.0,  0.0, 0.0, 16.0, 16.0},
                     { 8.0, 0.0,  8.0, 0.0,  8.0},
                     { 0.0, 0.0}};


    /**
     *  Initializes the simulation.
     */
    @Override
    protected void simInitialize()
    {
        super.simInitialize();
        putSimRegMap(REG_SPARE0, 0);
        putSimRegMap(REG_DS75LV_READ, (int)(256 * 23.45) | 0x80000000);
    }

    
    /**
     *  Writes simulated registers.
     *
     *  @param  addr   The first register address
     *  @param  value  The array of values to write.
     *  @param  count  The number of values to write.
     */
    @Override
    protected void simWriteRegs(int addr, int[] value, int count)
    {
        for (int j = 0; j < count; j++, addr++) {
            if (addr == REG_SPARE0) {
                simSwitchPower(getSimRegMap(REG_SPARE0), value[j]);
            }
            if (addr < 0x10000 || (addr & 0xfff) != 0x100) {
                putSimRegMap(addr, value[j]);
            }
        }
    }


    /**
     *  Simulates changing the power switch.
     *
     *  @param  oldSwitch  The old switch value
     *  @param  newSwitch  The new switch value
     */
    private void simSwitchPower(int oldSwitch, int newSwitch)
    {
        oldSwitch >>= 8;
        newSwitch >>= 8;
        int changed = oldSwitch ^ newSwitch;
        for (int chan = 0; changed != 0; chan++, changed >>= 1) {
            if ((changed & 1) != 0) {
                int reb = chan / 8;
                int ps = chan % 8;
                if (ps == 0) {
                    simMasterOn(reb, newSwitch >> (8 * reb));
                }
                else {
                    simPsOn(reb, ps - 1, (newSwitch & (1 << chan)) != 0);
                }
            }
        }
    }


    /**
     *  Simulates turning on or off the master switch.
     *
     *  @param  rebNum  The REB number
     *  @param  swtch   The switch value
     */
    private void simMasterOn(int rebNum, int swtch)
    {
        boolean on = (swtch & 1) != 0;
        for (int ps = 0; ps < NUM_PS; ps++) {
            boolean psOn = on && ((1 << (ps + 1)) & swtch) != 0;
            int psChan = ps == PS_HV_BIAS ? CHAN_VOLT_BEF_LDO : CHAN_VOLT_AFT_SW;
            for (int chan = 0; chan < simValues[ps].length; chan++) {
                if (chan != CHAN_CURR_AFT_LDO && chan != CHAN_CURR_BEF_LDO) {
                    simChanOn(rebNum, ps, chan, chan == psChan ? psOn : on);
                }
            }
        }
    }


    /**
     *  Simulates turning on or off one power supply.
     *
     *  @param  rebNum  The REB number
     *  @param  psNum   The power supply number
     *  @param  on      Turn on if true, off if false
     */
    private void simPsOn(int rebNum, int psNum, boolean on)
    {
        int chan = psNum == PS_HV_BIAS ? CHAN_VOLT_BEF_LDO : CHAN_VOLT_AFT_SW;
        simChanOn(rebNum, psNum, chan, on);
    }


    /**
     *  Simulates turning on or off one power supply channel.
     *
     *  @param  rebNum  The REB number
     *  @param  psNum   The power supply number
     *  @param  chan    The channel number
     *  @param  on      Turn on if true, off if false
     */
    private void simChanOn(int rebNum, int psNum, int chan, boolean on)
    {
        double value = on ? simValues[psNum][chan] : 0;
        if (on && psNum == PS_CLK_LOW) {
            if (chan == CHAN_VOLT_AFT_SW) {
                value = value / clockLoAftSwCoeff + simValues[psNum][CHAN_VOLT_AFT_LDO];
            }
            else if (chan == CHAN_VOLT2_AFT_LDO) {
                value += simValues[psNum][CHAN_VOLT_AFT_LDO];
            }
        }
        simSetChannel(rebNum, psNum, chan, value);
    }


    /**
     *  Sets a simulated channel DAC value.
     *
     *  @param  rebNum  The REB number
     *  @param  psNum   The power supply number
     *  @param  chan    The channel number
     *  @param  value   The value to set
     */
    private void simSetChannel(int rebNum, int psNum, int chan, double value)
    {
        try {
            chan = convChannel(psNum, chan);
            int desc = getChanDesc(psNum, chan);
            int devc = (desc >> 4) & 0x0f, dChan = desc & 0x0f;
            int addr = getDevcBaseAddress(rebNum, psNum, devc);
            int dacVal = (int)(value / getConversion(rebNum, psNum, chan) + 0.5);
            switch (getDeviceType(psNum, devc)) {
            case DEVC_LTC2945:
                addr += (dChan == 0) ? 0x14 : (dChan == 1) ? 0x1e : 0x28;
                putSimRegMap(addr, dacVal >> 4);
                putSimRegMap(addr + 1, (dacVal & 0x0f) << 4);
                break;
            case DEVC_MAX11644:
                int oldVal = getSimRegMap(addr);
                int mask = 0xffff << (16 * dChan);
                putSimRegMap(addr, (dacVal << (16 * (1 - dChan))) | (oldVal & mask));
                break;
            case DEVC_MAX5803:
                putSimRegMap(addr + 0xa0, dacVal >> 4);
                break;
            }
        }
        catch (DriverException e) {
            // Can't happen
        }
    }
    
    public double readFPGATemperature() throws DriverException {
        return readFPGATemperature(UG480_ADDRESS + UG480_TEMPERATURE_OFFSET);
    }

    public double readMaxFPGATemperature() throws DriverException {
        return readFPGATemperature(UG480_ADDRESS + UG480_MAX_TEMPERATURE_OFFSET);
    }
    
    public double readMinFPGATemperature() throws DriverException {
        return readFPGATemperature(UG480_ADDRESS + UG480_MIN_TEMPERATURE_OFFSET);
    }
    
    private double readFPGATemperature(int address) throws DriverException {
        int reg = this.readReg(address);
        return (0xfff & (reg>>4)) * (503.975 / 4096.0) - 273.15;
    }
        
}
