package org.lsst.ccs.drivers.auxelex;

import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Routines for controlling the REB PS bulk supplies 
 *
 *  @author  Owen Saxton
 */
public class RebBulkPS extends Srp {

    /**
     *  Constants and data.
     */
    public static final int
        FIRST_BULK = 1,
        NUM_BULKS = 5;

    public static final int
        IOSTAT_REM_ON  = 0x01,
        IOSTAT_ENABLED = 0x02,
        IOSTAT_AC_OK   = 0x04,
        IOSTAT_PWR_OK  = 0x08,
        IOSTAT_OTW     = 0x10;

    protected static final int
        REG_LAMBDA_BASE  = 0x20000,
        LAMBDA_INCREMENT = 0x0400,
        REG_LAMBDA_IO    = 0x21800;
    

    private static final List<BoardType> validTypes = new ArrayList<>();
    static {
        validTypes.add(BoardType.HEATER);
        validTypes.add(BoardType.SIMULATED);
    }

    private final LambdaPS[] lambda = new LambdaPS[NUM_BULKS];
    private final SA56004 temp;


    /**
     *  Constructor.
     */
    public RebBulkPS()
    {
        setValidBoardTypes(validTypes);
        for (int j = 0; j < NUM_BULKS; j++) {
            lambda[j] = new LambdaPS(this, REG_LAMBDA_BASE + (j + FIRST_BULK) * LAMBDA_INCREMENT, LambdaPS.MODEL_48);
        }
        temp = new SA56004(this);
    }


    /**
     *  Turns a PS on or off.
     *
     *  @param  id  The bulk PS ID (0 - 4)
     *  @param  on  Whether to turn on
     *  @throws  DriverException
     */
    public void setPowerOn(int id, boolean on) throws DriverException
    {
        writeReg(REG_LAMBDA_IO + FIRST_BULK + checkBulkId(id), on ? IOSTAT_REM_ON : 0);
    }


    /**
     *  Gets the I/O status of a PS.
     *
     *  @param  id  The bulk PS ID (0 - 4)
     *  @return  The status
     *  @throws  DriverException
     */
    public int getIoStatus(int id) throws DriverException
    {
        return readReg(REG_LAMBDA_IO + FIRST_BULK + checkBulkId(id));
    }


    /**
     *  Reads the board temperature
     *
     *  @return The board temperature (Celsius)
     *  @throws  DriverException
     */
    public double readBoardTemperature() throws DriverException
    {
        return temp.readTemperature();
    }


    /**
     *  Gets the model type of a PS.
     *
     *  @param  id  The bulk PS ID (0 - 4)
     *  @return  The model type
     *  @throws  DriverException
     */
    public int getModel(int id) throws DriverException
    {
        return lambda[checkBulkId(id)].getModel();
    }


    /**
     *  Gets  the serial number of a PS.
     *
     *  @param  id  The bulk PS ID (0 - 4)
     *  @return  The serial number
     *  @throws  DriverException
     */
    public String getSerialNo(int id) throws DriverException
    {
        return lambda[checkBulkId(id)].getSerialNo();
    }


    /**
     *  Gets the firmware version of a PS.
     *
     *  @param  id  The bulk PS ID (0 - 4)
     *  @return  The firmware revision
     *  @throws  DriverException
     */
    public String getFwVersion(int id) throws DriverException
    {
        return lambda[checkBulkId(id)].getFwVersion();
    }


    /**
     *  Gets the product version of a PS.
     *
     *  @param  id  The bulk PS ID (0 - 4)
     *  @return  The product revision
     *  @throws  DriverException
     */
    public String getPrVersion(int id) throws DriverException
    {
        return lambda[checkBulkId(id)].getPrVersion();
    }


    /**
     *  Gets the part number of a PS.
     *
     *  @param  id  The bulk PS ID (0 - 4)
     *  @return  The unit part number
     *  @throws  DriverException
     */
    public String getPartNo(int id) throws DriverException
    {
        return lambda[checkBulkId(id)].getPartNo();
    }


    /**
     *  Gets the manufacturing date of a PS.
     *
     *  @param  id  The bulk PS ID (0 - 4)
     *  @return  The manufacturing date
     *  @throws  DriverException
     */
    public String getManuDate(int id) throws DriverException
    {
        return lambda[checkBulkId(id)].getManuDate();
    }


    /**
     *  Gets the manufacturing location of a PS.
     *
     *  @param  id  The bulk PS ID (0 - 4)
     *  @return  The manufacturing location
     *  @throws  DriverException
     */
    public String getManuLocn(int id) throws DriverException
    {
        return lambda[checkBulkId(id)].getManuLocn();
    }


    /**
     *  Reads the status of a PS.
     *
     *  @param  id  The bulk PS ID (0 - 4)
     *  @return  The contents of the status register
     *  @throws  DriverException
     */
    public int readStatus(int id) throws DriverException
    {
        return lambda[checkBulkId(id)].readStatus();
    }


    /**
     *  Reads the voltage of a PS.
     *
     *  @param  id  The bulk PS ID (0 - 4)
     *  @return  The measured voltage
     *  @throws  DriverException
     */
    public double readVoltage(int id) throws DriverException
    {
        return lambda[checkBulkId(id)].readVoltage();
    }


    /**
     *  Reads the current of a PS.
     *
     *  @param  id  The bulk PS ID (0 - 4)
     *  @return  The measured current
     *  @throws  DriverException
     */
    public double readCurrent(int id) throws DriverException
    {
        return lambda[checkBulkId(id)].readCurrent();
    }


    /**
     *  Reads the baseplate temperature of a PS.
     *
     *  @param  id  The bulk PS ID (0 - 4)
     *  @return  The measured temperature (C)
     *  @throws  DriverException
     */
    public double readTemperature(int id) throws DriverException
    {
        return lambda[checkBulkId(id)].readTemperature();
    }



    /**
     *  Checks a bulk PS ID for validity.
     *
     *  @param  id  The bulk PS ID
     *  @return  The bulk PS ID
     *  @throws  DriverException
     */
    private int checkBulkId(int id) throws DriverException
    {
        if (id < 0 || id >= NUM_BULKS) {
            throw new DriverException("Invalid power supply ID: " + id);
        }
        return id;
    }


    /**
     *  Initializes the simulation.
     */
    @Override
    protected void simInitialize()
    {
        super.simInitialize();
        for (int j = 0; j < NUM_BULKS; j++) {
            lambda[j].simInitialize();
        }
        temp.simInitialize();
    }

    
    /**
     *  Writes simulated registers.
     *
     *  @param  addr   The first register address
     *  @param  value  The array of values to write.
     *  @param  count  The number of values to write.
     */
    @Override
    protected void simWriteRegs(int addr, int[] value, int count)
    {
        for (int j = 0; j < count; j++, addr++) {
            putSimRegMap(addr, value[j]);
            int id = addr - REG_LAMBDA_IO - FIRST_BULK;
            if (id >= 0 && id < NUM_BULKS) {
                lambda[id].simSetOutput(value[j] != 0);
            }
        }
    }

}
