package org.lsst.ccs.drivers.auxelex;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Program to test the REB power supply driver
 * 
 *  @author Owen Saxton
 */
public class TestRebPS extends TestSrp {

    /**
     *  Fields
     */
    private static final Map<Integer, String> versionMap = new HashMap<>();
    static {
        versionMap.put(RebPS.VERSION_PROD, "Production");
        versionMap.put(RebPS.VERSION_PROTO, "Prototype");
    }

    private static final Map<Integer, String> typeMap = new HashMap<>();
    static {
        typeMap.put(RebPS.TYPE_SCIENCE, "Science");
        typeMap.put(RebPS.TYPE_CORNER, "Corner");
        typeMap.put(RebPS.TYPE_UNKNOWN, "Unknown");
    }

    private final RebPS ps;


    /**
     *  Constructor
     */
    public TestRebPS()
    {
        super(new RebPS());
        ps = (RebPS)srp;
    }


    /*
     *  Commands
     */
    @Command(name="showserial", description="Show the serial number")
    public String showSerial() throws DriverException
    {
        return String.format("0x%016x", ps.getSerialNo());
    }


    @Command(name="showfirmware", description="Show the firmware version")
    public String showFirmware() throws DriverException
    {
        return String.format("0x%08x", ps.getFwVersion());
    }


    @Command(name="showgeogaddr", description="Show the geographical address")
    public String showGeogAddr() throws DriverException
    {
        return String.valueOf(ps.getGeogAddr());
    }


    @Command(name="showversion", description="Show the power supply version")
    public String showVersion() throws DriverException
    {
        int value = ps.getVersion();
        return String.format("%s (%s)", versionMap.get(value), value);
    }


    @Command(name="shownumrebs", description="Show the number of REBs")
    public String showNumRebs() throws DriverException
    {
        return String.valueOf(ps.getNumRebs());
    }


    @Command(name="shownumtemps", description="Show the number of temperatures")
    public String showNumTemps() throws DriverException
    {
        return String.valueOf(ps.getNumTemperatures());
    }


    @Command(name="showtype", description="Show the PS type")
    public String showType() throws DriverException
    {
        int value = ps.getType();
        return String.format("%s (%s)", typeMap.get(value), value);
    }


    @Command(name="configure", description="Configure a device")
    public void configure(@Argument(description="REB number")
                          int rebNum,
                          @Argument(description="Power supply number")
                          int psNum,
                          @Argument(description="Device number")
                          int devc) throws DriverException
    {
        ps.configure(rebNum, psNum, devc);
    }


    @Command(name="configure", description="Configure all devices in a PS")
    public void configure(@Argument(description="REB number")
                          int rebNum,
                          @Argument(description="Power supply number")
                          int psNum) throws DriverException
    {
        ps.configure(rebNum, psNum);
    }


    @Command(name="configure", description="Configure all devices in a REB")
    public void configure(@Argument(description="REB number")
                          int rebNum) throws DriverException
    {
        ps.configure(rebNum);
    }


    @Command(name="readpower", description="Show a power value")
    public String readPower(@Argument(description="REB number")
                            int rebNum,
                            @Argument(description="Power supply number")
                            int psNum,
                            @Argument(description="Channel number")
                            int chan) throws DriverException
    {
        return String.format("%.3f", ps.readChannel(rebNum, psNum, chan));
    }


    @Command(name="readpower", description="Show power values")
    public String readPower(@Argument(description="REB number")
                            int rebNum,
                            @Argument(description="Power supply number")
                            int psNum) throws DriverException
    {
        double[] values = ps.readChannel(rebNum, psNum);
        StringBuilder text = new StringBuilder(String.format("PS%s:", psNum));
        for (double value : values) {
            text.append(String.format("  %.2f", value));
        }
        return text.toString();
    }


    @Command(name="readpower", description="Show power values")
    public String readPower(@Argument(description="REB number")
                            int rebNum) throws DriverException
    {
        StringBuilder text = new StringBuilder(readPower(rebNum, 0));
        for (int psNum = 1; psNum < RebPS.NUM_PS; psNum++) {
            text.append("\n");
            text.append(readPower(rebNum, psNum));
        }
        return text.toString();
    }


    @Command(name="readextpower", description="Show extended power values")
    public String readExtPower(@Argument(description="REB number")
                               int rebNum,
                               @Argument(description="Power supply number")
                               int psNum,
                               @Argument(description="Channel number")
                               int chan) throws DriverException
    {
        double[] values = ps.readChanExtended(rebNum, psNum, chan);
        return String.format("Curr = %.2f, Max = %.2f, Min = %.2f, LimH = %.2f, LimL = %.2f",
                             values[RebPS.EXT_VALUE_CURR], values[RebPS.EXT_VALUE_MAX], values[RebPS.EXT_VALUE_MIN],
                             values[RebPS.EXT_LIMIT_HIGH], values[RebPS.EXT_LIMIT_LOW]);
    }


    @Command(name="readextpower", description="Show extended power values")
    public String readExtPower(@Argument(description="REB number")
                               int rebNum,
                               @Argument(description="Power supply number")
                               int psNum) throws DriverException
    {
        double[][] values = ps.readChanExtended(rebNum, psNum);
        StringBuilder text = new StringBuilder();
        for (int j = 0; j < values.length; j++) {
            double[] vals = values[j];
            String line = String.format("Chan %s: Curr = %.2f, Max = %.2f, Min = %.2f, LimH = %.2f, LimL = %.2f",
                                        j, vals[RebPS.EXT_VALUE_CURR], vals[RebPS.EXT_VALUE_MAX],
                                        vals[RebPS.EXT_VALUE_MIN], vals[RebPS.EXT_LIMIT_HIGH], vals[RebPS.EXT_LIMIT_LOW]);
            text.append(line).append(j < values.length - 1 ? "\n" : "");
        }
        return text.toString();
    }


    @Command(name="resetextrema", description="Reset maximum and minimum power values")
    public void resetExtrema(@Argument(description="REB number")
                             int rebNum,
                             @Argument(description="Power supply number")
                             int psNum,
                             @Argument(description="Channel number")
                             int chan) throws DriverException
    {
        ps.resetChanExtrema(rebNum, psNum, chan);
    }


    @Command(name="writedac", description="Write to a DAC")
    public void writeDac(@Argument(description="REB number")
                         int rebNum,
                         @Argument(description="Power supply number")
                         int psNum,
                         @Argument(description="DAC value")
                         double value) throws DriverException
    {
        ps.writeDac(rebNum, psNum, value);
    }


    @Command(name="configtemp", description="Configure the temperature system")
    public void configTemperature() throws DriverException
    {
        ps.configTemperature();
    }


    @Command(name="settempres", description="Set a temperature resolution")
    public void setTemperatureRes(@Argument(description="The sensor number (0 - 6)")
                                  int sensor,
                                  @Argument(description="Resolution value (0 - 3)")
                                  int res) throws DriverException
    {
        ps.setTemperatureRes(sensor, res);
    }


    @Command(name="settempres", description="Set all temperature resolutions")
    public void setTemperatureRes(@Argument(description="Resolution value (0 - 3)")
                                  int res) throws DriverException
    {
        ps.setTemperatureRes(res);
    }


    @Command(name="showtempres", description="Show a temperature resolution")
    public String showTemperatureRes(@Argument(description="The sensor number (0 - 6)")
                                     int sensor) throws DriverException
    {
        return String.valueOf(ps.getTemperatureRes(sensor));
    }


    @Command(name="showtempres", description="Show all temperature resolutions")
    public String showTemperatureRes() throws DriverException
    {
        int values[] = ps.getTemperatureRes();
        StringBuilder text = new StringBuilder();
        for (int value : values) {
            text.append(value).append(" ");
        }
        return text.toString();
    }


    @Command(name="showtemp", description="Show a temperature")
    public String showTemperature(@Argument(description="The sensor number (0 - 6)")
                                  int sensor) throws DriverException
    {
        return String.format("%.2f", ps.readTemperature(sensor));
    }


    @Command(name="showtemp", description="Show all temperatures")
    public String showTemperature() throws DriverException
    {
        return formatValues(ps.readTemperature());
    }


    @Command(name="showtempwarn", description="Show all warning temperature limits")
    public String showTempWarn() throws DriverException
    {
        return formatValues(ps.readTempWarn());
    }


    @Command(name="showtemperror", description="Show all error temperature limits")
    public String showTempError() throws DriverException
    {
        return formatValues(ps.readTempError());
    }


    @Command(name="setpower", description="Sets the power")
    public void setPower(@Argument(description="REB number")
                         int rebNum,
                         @Argument(description="Action value")
                         int action) throws DriverException
    {
        ps.setPower(rebNum, action);
    }


    @Command(name="showpower", description="Show the power")
    public String showPower(@Argument(description="REB number")
                            int rebNum) throws DriverException
    {
        return String.format("0x%02x", ps.getPower(rebNum));
    }


    @Command(name="showinitstatus", description="Show the initialization status")
    public String showInitStatus() throws DriverException
    {
        return String.format("0x%08x", ps.getInitStatus());
    }


    @Command(name="showfailsummary", description="Show the failure summary word")
    public String showFailSummary() throws DriverException
    {
        return String.format("0x%08x", ps.getFailureSummary());
    }


    @Command(name="showfaildetail", description="Show the failure detail")
    public String showFailDetail(@Argument(description="REB number")
                                 int rebNum) throws DriverException
    {
        return String.format("0x%08x", ps.getFailureDetail(rebNum));
    }


    @Command(name="showfailstring", description="Show the failure detail string")
    public String showFailString(@Argument(description="REB number")
                                 int rebNum) throws DriverException
    {
        return ps.getFailureDetailString(rebNum);
    }


    @Command(name="enablewrite", description="Enable writing to all registers")
    public void enableWrite(@Argument(description="Whether to enable (or not)")
                            boolean enable) throws DriverException
    {
        ps.enableAllWrite(enable);
    }


    @Command(name="showenablewrite", description="Show whether writing to all registers is enabled")
    public String showEnableWrite() throws DriverException
    {
        return String.valueOf(ps.isAllWriteEnabled());
    }


    @Command(name="enablepower", description="Enable leaving power on after error")
    public void enablePower(@Argument(description="Whether to enable (or not)")
                            boolean enable) throws DriverException
    {
        ps.enablePowerLeftOn(enable);
    }


    @Command(name="showenablepower", description="Show whether leaving power on is enabled")
    public String showEnablePower() throws DriverException
    {
        return String.valueOf(ps.isPowerLeftOnEnabled());
    }


    @Command(name="showerrors", description="Show the error counters")
    public String showErrors() throws DriverException
    {
        return String.format("Seq: %s   Tmo: %s", ps.getNumSeqErr(), ps.getNumTimeout());
    }


    @Command(name="contpower", description="Continuously read a power value")
    public String contPower(@Argument(description="REB number")
                            int rebNum,
                            @Argument(description="Power supply number")
                            int psNum,
                            @Argument(description="Channel number")
                            int chan) throws DriverException
    {
        System.out.println("Press any key to terminate...");
        boolean[] done = {false};
        consThread.start(done);
        int nRead = 0;
        long startTime = System.currentTimeMillis();
        while (!done[0]) {
            ps.readChannel(rebNum, psNum, chan);
            nRead++;
        }
        return formatRate(startTime, nRead);
    }


    @Command(name="contextpower", description="Continuously read extended power values")
    public String contExtPower(@Argument(description="REB number")
                               int rebNum,
                               @Argument(description="Power supply number")
                               int psNum,
                               @Argument(description="Channel number")
                               int chan) throws DriverException
    {
        System.out.println("Press any key to terminate...");
        boolean[] done = {false};
        consThread.start(done);
        int nRead = 0;
        long startTime = System.currentTimeMillis();
        while (!done[0]) {
            ps.readChanExtended(rebNum, psNum, chan);
            nRead++;
        }
        return formatRate(startTime, nRead);
    }


    @Command(name="conttemp", description="Continuously read a temperature value")
    public String contTemperature(@Argument(description="Channel number")
                                  int chan) throws DriverException
    {
        System.out.println("Press any key to terminate...");
        boolean[] done = {false};
        consThread.start(done);
        int nRead = 0;
        long startTime = System.currentTimeMillis();
        while (!done[0]) {
            ps.readTemperature(chan);
            nRead++;
        }
        return formatRate(startTime, nRead);
    }


    private String formatValues(double[] values) throws DriverException
    {
        StringBuilder text = new StringBuilder();
        for (double value : values) {
            text.append(String.format("%.2f", value)).append(" ");
        }
        return text.toString();
    }

}
