package org.lsst.ccs.drivers.auxelex;

import java.io.IOException;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Program to test the SLAC Register Protocol driver
 * 
 *  @author Owen Saxton
 */
public class TestSrp {

    /**
     *  Inner class for detecting console input while reading continuously.
     */
    protected class ConsThread extends Thread {

        private final BlockingQueue<Integer> consQueue = new ArrayBlockingQueue<>(1);
        private boolean[] consDone;
        private Thread mainThread;

        ConsThread() {
            super();
            setDaemon(true);
        }

        @Override
        public void run() {
            while (true) {
                awaitStart();
                awaitTerminal();
                consDone[0] = true;
                if (mainThread != null) {
                    mainThread.interrupt();
                }
            }
        }

        public void start(boolean[] done, boolean wake) {
            if (getState() == Thread.State.NEW) {
                start();
            }
            consDone = done;
            consDone[0] = false;
            mainThread = wake ? Thread.currentThread() : null;
            consQueue.offer(0);
        }

        public void start(boolean[] done) {
            start(done, false);
        }

        private void awaitStart() {
            while (true) {
                try {
                    consQueue.take();
                    return;
                }
                catch (InterruptedException e) {
                }
            }
        }

        private void awaitTerminal() {
            while (true) {
                try {
                    if (System.in.available() > 0) {
                        break;
                    }
                    try {
                        Thread.sleep(50);
                    }
                    catch (InterruptedException e) {
                    }
                }
                catch (IOException e) {
                    break;
                }
            }
            try {
                while (System.in.available() > 0) {
                    System.in.read();
                }
            }
            catch (IOException e) {
            }
        }

    }

    /**
     *  Data fields
     */
    protected static enum OnOff { ON, OFF; };
    protected final Srp srp;
    protected final ConsThread consThread = new ConsThread();


    /**
     *  Constructor.
     */
    public TestSrp()
    {
        srp = new Srp();
    }


    /**
     *  Constructor.
     *
     *  @param  srp  Srp object
     */
    public TestSrp(Srp srp)
    {
        this.srp = srp;
    }


    /**
     *  Sets the SRP version
     *
     *  @param  version  The version to set
     */
    @Command(description="Set the SRP version")
    public void setSrpVersion(@Argument(description="Version number")
                              int version)
    {
        srp.setSrpVersion(version);
    }


    /**
     *  Sets the debug state.
     *
     *  @param  state  The debug state to set, ON or OFF
     */
    @Command(description="Set the debug state")
    public void setDebug(@Argument(description="The state (on or off)")
                         OnOff state)
    {
        srp.setDebug(state == OnOff.ON);
    }


    /**
     *  Connects to a device.
     * 
     *  @param  ipAddr  The IP address
     *  @param  port    The port number
     *  @throws  DriverException
     */
    @Command(description="Open connection to device")
    public void connectIp(@Argument(description="IP address")
                          String ipAddr,
                          @Argument(description="Port number")
                          int port) throws DriverException
    {
        srp.open(ipAddr, port);
        announce();
    }


    /**
     *  Connects to a device.
     * 
     *  @param  ipAddr  The IP address
     *  @throws  DriverException
     */
    @Command(description="Open connection to device")
    public void connectIp(@Argument(description="IP address")
                          String ipAddr) throws DriverException
    {
        srp.open(ipAddr);
        announce();
    }


    /**
     *  Connects to a device.
     * 
     *  @param  node  The node number
     *  @param  port  The port number
     *  @throws  DriverException
     */
    @Command(description="Open connection to device")
    public void connect(@Argument(description="Node number")
                        int node,
                        @Argument(description="Port number")
                        int port) throws DriverException
    {
        srp.open(node, port);
        announce();
    }


    /**
     *  Connects to a device.
     * 
     *  @param  node  The node number
     *  @throws  DriverException
     */
    @Command(description="Open connection to device")
    public void connect(@Argument(description="Node number")
                        int node) throws DriverException
    {
        srp.open(node);
        announce();
    }


    /**
     *  Announces successful connection.
     */
    private void announce()
    {
        System.out.println("Board type is " + srp.getBoardType());
    }


    /**
     *  Disconnects from the device.
     * 
     *  @throws  DriverException
     */
    @Command(description="Close connection to device")
    public void disconnect() throws DriverException
    {
        srp.close();
    }


    /**
     *  Shows the board type.
     *
     *  @return  The board type
     *  @throws  DriverException
     */
    @Command(description="Show the board type")
    public Srp.BoardType showBoardType() throws DriverException
    {
        return srp.getBoardType();
    }


    /**
     *  Shows the build stamp.
     *
     *  @return  The build stamp
     *  @throws  DriverException
     */
    @Command(description="Show the build stamp")
    public String showBuildStamp() throws DriverException
    {
        return srp.getBuildStamp();
    }


    /**
     *  Reads a register.
     * 
     *  @param  addr  The register address
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(description="Read a register")
    public String read(@Argument(description="Register address")
                       int addr) throws DriverException
    {
        return String.format("%08x", srp.readReg(addr));
    }


    /**
     *  Reads registers.
     * 
     *  @param  addr   The first register address
     *  @param  count  The number of registers to read
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(description="Reads registers")
    public String read(@Argument(description="Register address")
                       int addr,
                       @Argument(description="Register count")
                       int count) throws DriverException
    {
        return formatHex(addr, srp.readRegs(addr, count));
    }


    /**
     *  Writes registers.
     * 
     *  @param  addr   The first register address
     *  @param  value  The values to write
     *  @throws  DriverException
     */
    @Command(description="Write a register")
    public void write(@Argument(description="Register address")
                      int addr,
                      @Argument(description="Register values")
                      int... value) throws DriverException
    {
        srp.writeRegs(addr, value);
    }


    /**
     *  Updates a register.
     * 
     *  @param  addr   The register address
     *  @param  mask   The mask of bits to change
     *  @param  value  The value to write
     *  @throws  DriverException
     */
    @Command(description="Update a register")
    public void update(@Argument(description="Register address")
                       int addr,
                       @Argument(description="Mask of bits to change")
                       int mask,
                       @Argument(description="Register value")
                       int value) throws DriverException
    {
        srp.updateReg(addr, mask, value);
    }


    /**
     *  Continuously reads registers.
     * 
     *  @param  addr   The first register address
     *  @param  count  The number of registers to read
     *  @return  The result string
     *  @throws  DriverException
     */
    @Command(description="Continuously read a set of registers")
    public String contRead(@Argument(description="First register address")
                           int addr,
                           @Argument(description="Number of registers")
                           int count) throws DriverException
    {
        System.out.println("Press any key to terminate...");
        boolean[] done = {false};
        consThread.start(done);
        int nRead = 0;
        long startTime = System.currentTimeMillis();
        while (!done[0]) {
            srp.readRegs(addr, count);
            nRead += count;
        }
        return formatRate(startTime, nRead);
    }


    /**
     *  Formats, in hexadecimal, array of register values.
     *
     *  @param  addr  The address of the first data word
     *  @param  data  Array of data words
     *  @return  The result string
     */
    protected static String formatHex(int addr, int[] data)
    {
        StringBuilder text = new StringBuilder();
        for (int j = 0; j < data.length; j++) {
            if ((j & 3) == 0) {
                if (j > 0) {
                    text.append('\n');
                }
                text.append(String.format("%06x:", addr + j));
            }
            text.append(String.format(" %08x", data[j]));
        }
        return text.toString();
    }


    /**
     *  Formats continuous test rates.
     *
     *  @param  start  The start time
     *  @param  count  The number of operations
     *  @return  The result string
     */
    protected static String formatRate(long start, int count)
    {
        long period = System.currentTimeMillis() - start;
        return String.format("Read rate (Hz) = %.4g", count * 1000.0 / (period == 0 ? 1 : period));
    }

}
