package org.lsst.ccs.drivers.auxelex;

import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Access routines for the SA56004 temperature sensor.
 *
 *  @author  Owen Saxton
 */
public class SA56004 {

    public static final int
        STATUS_LCRIT = 0x01,
        STATUS_RCRIT = 0x02,
        STATUS_OPEN  = 0x04,
        STATUS_RLOW  = 0x08,
        STATUS_RHIGH = 0x10,
        STATUS_LLOW  = 0x20,
        STATUS_LHIGH = 0x40,
        STATUS_BUSY  = 0x80;

    private static final int
        REG_BASE    = 0x40000,
        REG_LTHB    = REG_BASE,
        REG_RTHB    = REG_BASE + 0x01,
        REG_SR      = REG_BASE + 0x02,
        REG_CON_R   = REG_BASE + 0x03,
        REG_CR_R    = REG_BASE + 0x04,
        REG_LHS_R   = REG_BASE + 0x05,
        REG_LLS_R   = REG_BASE + 0x06,
        REG_RHSHB_R = REG_BASE + 0x07,
        REG_RLSHB_R = REG_BASE + 0x08,
        REG_CON_W   = REG_BASE + 0x09,
        REG_CR_W    = REG_BASE + 0x0a,
        REG_LHS_W   = REG_BASE + 0x0b,
        REG_LLS_W   = REG_BASE + 0x0c,
        REG_RHSHB_W = REG_BASE + 0x0d,
        REG_RLSHB_W = REG_BASE + 0x0e,
        REG_CONV    = REG_BASE + 0x0f,
        REG_RTLB    = REG_BASE + 0x10,
        REG_RTOHB   = REG_BASE + 0x11,
        REG_RTOLB   = REG_BASE + 0x12,
        REG_RHSLB   = REG_BASE + 0x13,
        REG_RLSLB   = REG_BASE + 0x14,
        REG_RCS     = REG_BASE + 0x19,
        REG_LCS     = REG_BASE + 0x20,
        REG_TH      = REG_BASE + 0x21,
        REG_LTLB    = REG_BASE + 0x22,
        REG_AM      = REG_BASE + 0xbf,
        REG_RMID    = REG_BASE + 0xfe,
        REG_RDR     = REG_BASE + 0xff;

    private final Srp srp;


    /**
     *  Constructor.
     *
     *  @param  srp  The Srp object to use
     */
    public SA56004(Srp srp)
    {
        this.srp = srp;
    }


    /**
     *  Reads the status register.
     *
     *  @return  The status register contents
     *  @throws  DriverException
     */
    public int readStatus() throws DriverException
    {
        return srp.readReg(REG_SR);
    }


    /**
     *  Reads the temperature.
     *
     *  @return  The temperature (C)
     *  @throws  DriverException
     */
    public double readTemperature() throws DriverException
    {
        return (byte)srp.readReg(REG_LTHB) + srp.readReg(REG_LTLB) / 256.0;
    }


    /**
     *  Sets the conversion rate.
     *
     *  @param  rate  The encoded conversion rate (0 - 9).  Freq. is 2^(rate - 4).
     *  @throws  DriverException
     */
    public void setConvRate(int rate) throws DriverException
    {
        srp.writeReg(REG_CR_W, rate);
    }


    /**
     *  Gets the conversion rate.
     *
     *  @return  The encoded conversion rate
     *  @throws  DriverException
     */
    public int getConvRate() throws DriverException
    {
        return srp.readReg(REG_CR_R);
    }

}
