package org.lsst.ccs.drivers.auxelex;

import java.util.LinkedHashMap;
import java.util.Map;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Program to test the low voltage PDU driver
 * 
 *  @author Owen Saxton
 */
public class TestRebBulkPS extends TestSrp {

   /*
     *  Constants
     */
    static final Map<Integer, String> ioBitsMap = new LinkedHashMap<>();
    static {
        ioBitsMap.put(RebBulkPS.IOSTAT_REM_ON, "rem-on");
        ioBitsMap.put(RebBulkPS.IOSTAT_ENABLED, "enab");
        ioBitsMap.put(RebBulkPS.IOSTAT_AC_OK, "ac-ok");
        ioBitsMap.put(RebBulkPS.IOSTAT_PWR_OK, "pwr-ok");
        ioBitsMap.put(RebBulkPS.IOSTAT_OTW, "otw");
    }

    static final Map<Integer, String> statBitsMap = new LinkedHashMap<>();
    static {
        statBitsMap.put(LambdaPS.STATUS_OFF, "off");
        statBitsMap.put(LambdaPS.STATUS_DSAB, "dsab");
        statBitsMap.put(LambdaPS.STATUS_IOGB, "iogb");
        statBitsMap.put(LambdaPS.STATUS_OTW, "otw");
        statBitsMap.put(LambdaPS.STATUS_OTP, "otp");
        statBitsMap.put(LambdaPS.STATUS_VBAD, "vbad");
        statBitsMap.put(LambdaPS.STATUS_IBAD, "ibad");
    }

     /*
     *  Fields
     */
    private final RebBulkPS bulk;

    /*
     *  Constructor
     */
    public TestRebBulkPS()
    {
        super(new RebBulkPS());
        bulk = (RebBulkPS)srp;
    }

    /*
     *  Commands
     */
    @Command(description="Read the board temperature")
    public double readBoardTemp() throws DriverException
    {
        return bulk.readBoardTemperature();
    }

    @Command(description="Turn a PS on or off")
    public void setPower(@Argument(description="Power supply ID") int id,
                         @Argument(description="Whether to turn on") OnOff on) throws DriverException
    {
        bulk.setPowerOn(id, on == OnOff.ON);
    }

    @Command(description="Show a PS I/O status word")
    public String showIoStatus(@Argument(description="Power supply ID") int id) throws DriverException
    {
        int status = bulk.getIoStatus(id);
        return String.format("0x%02x %s", status, genBitNames(status, ioBitsMap));
    }

    @Command(description="Show a PS model type")
    public int showModel(@Argument(description="Power supply ID") int id) throws DriverException
    {
        return bulk.getModel(id);
    }

    @Command(description="Show a PS serial number")
    public String showSerial(@Argument(description="Power supply ID") int id) throws DriverException
    {
        return bulk.getSerialNo(id);
    }

    @Command(description="Show a PS firmware version")
    public String showFwVersion(@Argument(description="Power supply ID") int id) throws DriverException
    {
        return bulk.getFwVersion(id);
    }

    @Command(description="Show a PS product version")
    public String showPrVersion(@Argument(description="Power supply ID") int id) throws DriverException
    {
        return bulk.getPrVersion(id);
    }

    @Command(description="Show a PS part number")
    public String showPartNo(@Argument(description="Power supply ID") int id) throws DriverException
    {
        return bulk.getPartNo(id);
    }

    @Command(description="Show a PS manufacture date")
    public String showManuDate(@Argument(description="Power supply ID") int id) throws DriverException
    {
        return bulk.getManuDate(id);
    }

    @Command(description="Show a PS manufacture location")
    public String showManuLocn(@Argument(description="Power supply ID") int id) throws DriverException
    {
        return bulk.getManuLocn(id);
    }

    @Command(description="Read a PS status word")
    public String readStatus(@Argument(description="Power supply ID") int id) throws DriverException
    {
        int status = bulk.readStatus(id);
        return String.format("0x%02x %s", status, genBitNames(status, statBitsMap));
    }

    @Command(description="Read a PS voltage")
    public double readVoltage(@Argument(description="Power supply ID") int id) throws DriverException
    {
        return bulk.readVoltage(id);
    }

    @Command(description="Read a PS current")
    public double readCurrent(@Argument(description="Power supply ID") int id) throws DriverException
    {
        return bulk.readCurrent(id);
    }

    @Command(description="Read a PS temperature")
    public double readTemp(@Argument(description="Power supply ID") int id) throws DriverException
    {
        return bulk.readTemperature(id);
    }

    static String genBitNames(int value, Map<Integer, String> bitsMap)
    {
        StringBuilder text = new StringBuilder();
        String prefix = "(";
        for (int mask : bitsMap.keySet()) {
            if ((mask & value) != 0) {
                text.append(prefix).append(bitsMap.get(mask));
                prefix = ", ";
            }
        }
        if (text.length() != 0) {
            text.append(")");
        }
        return text.toString();
    }

}
