package org.lsst.ccs.drivers.auxelex;

import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Routines for controlling the trim heater power supply. 
 *
 *  @author  Owen Saxton
 */
public class HeaterPS extends Srp {

    /**
     *  Constants and data.
     */
    public static final int
        NUM_HEATERS = 12,
        MIN_SWITCH_PERIOD = 100,
        MAX_SWITCH_PERIOD = 500;

    private static final int
        REG_CONTROL_BASE  = 0,
        CONTROL_INCREMENT = 2,
        REG_MONITOR_BASE = 0x10000,
        BULK_CHAN = 0,
        REG_LAMBDA_BASE = RebBulkPS.REG_LAMBDA_BASE + BULK_CHAN * RebBulkPS.LAMBDA_INCREMENT,
        REG_LAMBDA_IO = RebBulkPS.REG_LAMBDA_IO + BULK_CHAN;

    private static final double
        SHUNT_020 = 0.02,
        MIN_SOURCE_VOLTS = 1.0;

    private static final List<BoardType> validTypes = new ArrayList<>();
    static {
        validTypes.add(BoardType.HEATER);
        validTypes.add(BoardType.SIMULATED);
    }

    private final LTC2945 adc;
    private final LambdaPS bulk;
    private final SA56004 temp;
    private final HeaterConv conv = new HeaterConv();
    private int switchPeriod = 500;


    /**
     *  Constructor.
     */
    public HeaterPS()
    {
        setValidBoardTypes(validTypes);
        adc = new LTC2945(this, REG_MONITOR_BASE);
        bulk = new LambdaPS(this, REG_LAMBDA_BASE, LambdaPS.MODEL_28);
        temp = new SA56004(this);
    }


    /**
     *  Opens the connection.
     *
     *  @param  node  The PS node number
     *  @throws  DriverException
     */
    @Override
    public void open(int node) throws DriverException
    {
        super.open(node);
        conv.setNode(node);
    }


    /**
     *  Opens the connection.
     *
     *  @param  host  The PS host name
     *  @throws  DriverException
     */
    @Override
    public void open(String host) throws DriverException
    {
        super.open(host);
        conv.setNode(getIpAddress()[3]);
    }


    /**
     *  Sets the common switching period.
     *
     *  @param  period  The switching period, in 5 ns units
     *  @throws  DriverException
     */
    public void setSwitchPeriod(int period) throws DriverException
    {
        if (period < MIN_SWITCH_PERIOD || period > MAX_SWITCH_PERIOD) {
            throw new DriverException("Switch period must be between 100 and 500");
        }
        switchPeriod = period;
    }


    /**
     *  Gets the common switching period.
     *
     *  @return  The switching period, in 5 ns units
     */
    public int getSwitchPeriod()
    {
        return switchPeriod;
    }


    /**
     *  Sets the output duty cycle for a channel.
     *
     *  @param  chan  The channel number
     *  @param  duty  The duty cycle (0 - 1.0)
     *  @throws  DriverException
     */
    public void setDutyCycle(int chan, double duty) throws DriverException
    {
        int high = Math.max(2 , Math.min(switchPeriod - 2, (int)(duty * switchPeriod)));  // Why is 2 necessary??
        int low = switchPeriod - high;
        int regNum = REG_CONTROL_BASE + CONTROL_INCREMENT * checkChannelNumber(chan);
        writeReg(regNum, (readReg(regNum) & 0xfffc0000) | ((low - 1) << 9) | (high - 1));
    }


    /**
     *  Gets the output duty cycle for a channel.
     *
     *  @param  chan  The channel number
     *  @return  The duty cycle (0 - 1.0)
     *  @throws  DriverException
     */
    public double getDutyCycle(int chan) throws DriverException
    {
        int regNum = REG_CONTROL_BASE + CONTROL_INCREMENT * checkChannelNumber(chan);
        int regVal = readReg(regNum);
        double high = (regVal & 0x1ff) + 1;
        double low = ((regVal >> 9) & 0x1ff) + 1;
        return high / (high + low);
    }


    /**
     *  Sets the output voltage for a channel.
     *
     *  @param  chan   The channel number
     *  @param  volts  The voltage
     *  @throws  DriverException
     */
    public void setVoltage(int chan, double volts) throws DriverException
    {
        checkChannelNumber(chan);
        double source = readMainVoltage();
        if (source < MIN_SOURCE_VOLTS) {
            throw new DriverException("Board is not powered");
        }
        setDutyCycle(chan, conv.convVoltsAndDuty(true, chan, volts / source));
    }


    /**
     *  Gets the set output voltage for a channel.
     *
     *  @param  chan  The channel number
     *  @return  The set voltage
     *  @throws  DriverException
     */
    public double getVoltage(int chan) throws DriverException
    {
        checkChannelNumber(chan);
        return readMainVoltage() * conv.convVoltsAndDuty(false, chan, getDutyCycle(chan));
    }


    /**
     *  Sets the output state for a channel.
     *
     *  @param  chan  The channel number
     *  @param  on    Whether channel output is set on
     *  @throws  DriverException
     */
    public void setOutput(int chan, boolean on) throws DriverException
    {
        int regNum = REG_CONTROL_BASE + CONTROL_INCREMENT * checkChannelNumber(chan);
        writeReg(regNum, (readReg(regNum) & 0xf7ffffff) | ((on ? 1 : 0) << 27));
    }


    /**
     *  Gets the output state for a channel.
     *
     *  @param  chan  The channel number
     *  @return  Whether channel output is set on
     *  @throws  DriverException
     */
    public boolean getOutput(int chan) throws DriverException
    {
        int regNum = REG_CONTROL_BASE + CONTROL_INCREMENT * checkChannelNumber(chan);
        return (readReg(regNum) & 0x08000000) != 0;
    }


    /**
     *  Reads a channel's voltage.
     * 
     *  This returns the input voltage, not the effective output one.
     *
     *  @param  chan  The channel number
     *  @return  The voltage on that channel
     *  @throws  DriverException  if the channel number is invalid
     */
    public double readVoltage(int chan) throws  DriverException
    {
        return adc.readVoltage(checkChannelNumber(chan));
    }


    /**
     *  Reads a channel's current
     *
     *  @param  chan  The channel number
     *  @return  The current on that channel
     *  @throws  DriverException  if the channel number is invalid
     */
    public double readCurrent(int chan) throws  DriverException
    {
        return adc.readCurrent(checkChannelNumber(chan)) / SHUNT_020;
    }


    /**
     *  Reads the board temperature
     *
     *  @return The board temperature (Celsius)
     *  @throws  DriverException
     */
    public double readBoardTemperature() throws DriverException
    {
        return temp.readTemperature();
    }


    /**
     *  Gets the main PS model type.
     *
     *  @return  The model type
     *  @throws  DriverException
     */
    public int getMainModel() throws DriverException
    {
        return bulk.getModel();
    }


    /**
     *  Gets the main PS serial number.
     *
     *  @return  The serial number
     *  @throws  DriverException
     */
    public String getMainSerialNo() throws DriverException
    {
        return bulk.getSerialNo();
    }


    /**
     *  Gets the main PS firmware version.
     *
     *  @return  The firmware revision
     *  @throws  DriverException
     */
    public String getMainFwVersion() throws DriverException
    {
        return bulk.getFwVersion();
    }


    /**
     *  Gets the main PS product version.
     *
     *  @return  The product revision
     *  @throws  DriverException
     */
    public String getMainPrVersion() throws DriverException
    {
        return bulk.getPrVersion();
    }


    /**
     *  Gets the main PS part number.
     *
     *  @return  The unit part number
     *  @throws  DriverException
     */
    public String getMainPartNo() throws DriverException
    {
        return bulk.getPartNo();
    }


    /**
     *  Gets the main PS manufacturing date.
     *
     *  @return  The manufacturing date
     *  @throws  DriverException
     */
    public String getMainManuDate() throws DriverException
    {
        return bulk.getManuDate();
    }


    /**
     *  Gets the main PS manufacturing location.
     *
     *  @return  The manufacturing location
     *  @throws  DriverException
     */
    public String getMainManuLocn() throws DriverException
    {
        return bulk.getManuLocn();
    }


    /**
     *  Reads the main PS status.
     *
     *  @return  The contents of the status register
     *  @throws  DriverException
     */
    public int readMainStatus() throws DriverException
    {
        return bulk.readStatus();
    }


    /**
     *  Reads the main PS voltage.
     *
     *  @return  The measured voltage
     *  @throws  DriverException
     */
    public double readMainVoltage() throws DriverException
    {
        return bulk.readVoltage();
    }


    /**
     *  Reads the main PS current.
     *
     *  @return  The measured current
     *  @throws  DriverException
     */
    public double readMainCurrent() throws DriverException
    {
        return bulk.readCurrent();
    }


    /**
     *  Reads the main PS baseplate temperature.
     *
     *  @return  The measured temperature (C)
     *  @throws  DriverException
     */
    public double readMainTemperature() throws DriverException
    {
        return bulk.readTemperature();
    }


    /**
     *  Turns the main PS on or off.
     *
     *  @param  on  Whether to turn on
     *  @throws  DriverException
     */
    public void setMainPowerOn(boolean on) throws DriverException
    {
        writeReg(REG_LAMBDA_IO, on ? RebBulkPS.IOSTAT_REM_ON : 0);
    }


    /**
     *  Gets the I/O status of the main PS.
     *
     *  @return  The status
     *  @throws  DriverException
     */
    public int getMainIoStatus() throws DriverException
    {
        return readReg(REG_LAMBDA_IO);
    }


    /**
     *  Checks a channel number for validity
     *
     *  @param  chan  The channel number
     *  @return  The channel number
     *  @throws  DriverException  if the channel number is invalid
     */
    private int checkChannelNumber(int chan) throws  DriverException
    {
        if (chan < 0 || chan >= NUM_HEATERS) {
            throw new DriverException("Invalid heater channel number: " + chan);
        }
        return chan;
    }

}
