package org.lsst.ccs.drivers.auxelex;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Common code for testing the PDU drivers
 * 
 *  @author Owen Saxton
 */
public class TestHeaterPS extends TestSrp {

    /*
     *  Fields
     */
    private static final double[]
        calDuties = {0.01, 0.05, 0.1, 0.15, 0.2, 0.25, 0.3, 0.35, 0.4, 0.45,
                     0.5,  0.55, 0.6, 0.65, 0.7, 0.75, 0.8, 0.85, 0.9, 0.95, 0.99};
    protected final HeaterPS htr;

    /*
     *  Constructor
     */
    public TestHeaterPS()
    {
        super(new HeaterPS());
        htr = (HeaterPS)srp;
    }

    /*
     *  Commands
     */
    @Command(description="Read the board temperature")
    public double readBoardTemp() throws DriverException
    {
        return htr.readBoardTemperature();
    }

    @Command(description="Set the common switch period")
    public void setSwitchPeriod(@Argument(description="Switch period") int period) throws DriverException
    {
        htr.setSwitchPeriod(period);
    }

    @Command(description="Show the common switch period")
    public int showSwitchPeriod()
    {
        return htr.getSwitchPeriod();
    }

    @Command(description="Set the duty cycle for a channel")
    public void setDutyCycle(@Argument(description="Channel number") int chan,
                             @Argument(description="Duty cycle") double duty) throws DriverException
    {
        htr.setDutyCycle(chan, duty);
    }

    @Command(description="Show the duty cycle for a channel")
    public double showDutyCycle(@Argument(description="Channel number") int chan) throws DriverException
    {
        return htr.getDutyCycle(chan);
    }

    @Command(description="Set the voltage for a channel")
    public void setVoltage(@Argument(description="Channel number") int chan,
                           @Argument(description="Voltage") double volts) throws DriverException
    {
        htr.setVoltage(chan, volts);
    }

    @Command(description="Show the set voltage for a channel")
    public double showVoltage(@Argument(description="Channel number") int chan) throws DriverException
    {
        return htr.getVoltage(chan);
    }

    @Command(description="Turn on/off a channel")
    public void setOutput(@Argument(description="Channel number") int chan,
                          @Argument(description="State to set") OnOff state) throws DriverException
    {
        htr.setOutput(chan, state == OnOff.ON);
    }

    @Command(description="Show the state of a channel")
    public String showOutput(@Argument(description="Channel number") int chan) throws DriverException
    {
        return htr.getOutput(chan) ? "on" : "off";
    }

    @Command(description="Read a channel's voltage")
    public double readVoltage(@Argument(description="Channel number") int chan) throws DriverException
    {
        return htr.readVoltage(chan);
    }

    @Command(description="Read a channel's current")
    public double readCurrent(@Argument(description="Channel number") int chan) throws DriverException
    {
        return htr.readCurrent(chan);
    }

    @Command(description="Show the complete power state")
    public String showPower() throws DriverException
    {
        StringBuilder text = new StringBuilder("Chan  State   Duty  Volts   mAmps");
        for (int chan = 0; chan < HeaterPS.NUM_HEATERS; chan++) {
            text.append(String.format("\n %2s    %3s   %5.3f  %5.2f  %6.1f",
                        chan, htr.getOutput(chan) ? " on" : "off", htr.getDutyCycle(chan),
                        htr.readVoltage(chan), 1000 * htr.readCurrent(chan)));
        }
        return text.toString();
    }

    @Command(description="Show a calibration table")
    public String showCalibration(@Argument(description="Channel number") int chan) throws DriverException
    {
        StringBuilder text = new StringBuilder("Duty  VFrac");
        for (double duty : calDuties) {
            setDutyCycle(chan, duty);
            try {
                Thread.sleep(1000);
            }
            catch (InterruptedException e) {}
            double vFract = htr.readCurrent(chan) * htr.getHeaterOhms(chan) / htr.getSourceVoltage();
            text.append(String.format("\n%4.2f  %5.3f", duty, vFract));
            System.out.print(".");
        }
        System.out.print("\r");
        for (int j = 0; j < calDuties.length; j++) {
            System.out.print(" ");
        }
        System.out.print("\r");
        return text.toString();
    }

    @Command(description="Show the main model type")
    public int showMainModel() throws DriverException
    {
        return htr.getMainModel();
    }

    @Command(description="Show the main serial number")
    public String showMainSerial() throws DriverException
    {
        return htr.getMainSerialNo();
    }

    @Command(description="Show the main firmware version")
    public String showMainFwVersion() throws DriverException
    {
        return htr.getMainFwVersion();
    }

    @Command(description="Show the main product version")
    public String showMainPrVersion() throws DriverException
    {
        return htr.getMainPrVersion();
    }

    @Command(description="Show the main part number")
    public String showMainPartNo() throws DriverException
    {
        return htr.getMainPartNo();
    }

    @Command(description="Show the main manufacture date")
    public String showMainManuDate() throws DriverException
    {
        return htr.getMainManuDate();
    }

    @Command(description="Show the main manufacture location")
    public String showMainManuLocn() throws DriverException
    {
        return htr.getMainManuLocn();
    }

    @Command(description="Read the main status word")
    public String readMainStatus() throws DriverException
    {
        int status = htr.readMainStatus();
        return String.format("0x%02x %s", status, TestRebBulkPS.genBitNames(status, TestRebBulkPS.statBitsMap));
    }

    @Command(description="Read the main voltage")
    public double readMainVoltage() throws DriverException
    {
        return htr.readMainVoltage();
    }

    @Command(description="Read the main current")
    public double readMainCurrent() throws DriverException
    {
        return htr.readMainCurrent();
    }

    @Command(description="Read the main temperature")
    public double readMainTemp() throws DriverException
    {
        return htr.readMainTemperature();
    }

    @Command(description="Set the main power on or off")
    public void setMainPower(OnOff on) throws DriverException
    {
        htr.setMainPowerOn(on == OnOff.ON);
    }

    @Command(description="Show the main I/O status")
    public String showMainIoStatus() throws DriverException
    {
        int status = htr.getMainIoStatus();
        return String.format("0x%02x %s", status, TestRebBulkPS.genBitNames(status, TestRebBulkPS.ioBitsMap));
    }

}
