package org.lsst.ccs.drivers.auxelex;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Routines for controlling the low-voltage (5V) SLAC PDU 
 *
 *  @author  Owen Saxton
 */
public class PduLV extends PduCommon {

    /**
     *  Constants
     */
    public static final int
        CHAN_SPARE_0 = 0,
        CHAN_SPARE_1 = 1,
        CHAN_SPARE_2 = 2,
        CHAN_SPARE_3 = 3,
        CHAN_SPARE_4 = 4,
        CHAN_SPARE_5 = 5,
        CHAN_OTM_0_A = 8,
        CHAN_OTM_0_B = 9,
        CHAN_OTM_1_A = 10,
        CHAN_OTM_1_B = 11,
        CHAN_OTM_2_A = 12,
        CHAN_OTM_2_B = 13,
        CHAN_OTM_3_A = 14,
        CHAN_OTM_3_B = 15,
        CHAN_OTM_4_A = 16,
        CHAN_OTM_4_B = 17,
        CHAN_OTM_5_A = 18,
        CHAN_OTM_5_B = 19,
        NUM_CHANS = 20;

    /**
     *  Private fields
     */
    private static final Map<String, Integer> CHANNEL_MAP = new HashMap<>();
    static {
        CHANNEL_MAP.put("spare0", CHAN_SPARE_0);
        CHANNEL_MAP.put("spare1", CHAN_SPARE_1);
        CHANNEL_MAP.put("spare2", CHAN_SPARE_2);
        CHANNEL_MAP.put("spare3", CHAN_SPARE_3);
        CHANNEL_MAP.put("spare4", CHAN_SPARE_4);
        CHANNEL_MAP.put("spare5", CHAN_SPARE_5);
        CHANNEL_MAP.put("otm0a", CHAN_OTM_0_A);
        CHANNEL_MAP.put("otm0b", CHAN_OTM_0_B);
        CHANNEL_MAP.put("otm1a", CHAN_OTM_1_A);
        CHANNEL_MAP.put("otm1b", CHAN_OTM_1_B);
        CHANNEL_MAP.put("otm2a", CHAN_OTM_2_A);
        CHANNEL_MAP.put("otm2b", CHAN_OTM_2_B);
        CHANNEL_MAP.put("otm3a", CHAN_OTM_3_A);
        CHANNEL_MAP.put("otm3b", CHAN_OTM_3_B);
        CHANNEL_MAP.put("otm4a", CHAN_OTM_4_A);
        CHANNEL_MAP.put("otm4b", CHAN_OTM_4_B);
        CHANNEL_MAP.put("otm5a", CHAN_OTM_5_A);
        CHANNEL_MAP.put("otm5b", CHAN_OTM_5_B);
    }
    private static final List<BoardType> VALID_TYPES = new ArrayList<>();
    static {
        VALID_TYPES.add(BoardType.PDU_5V);
    }
    private static final int
        VALID_CHAN_MASK = 0x000fff3f;
    private static final double
        SHUNT_SPARE = 0.0128,
        SHUNT_OTM = 0.0512;
    private static final double[] CURR_SCALES = new double[NUM_CHANS];
    static {
        CURR_SCALES[CHAN_SPARE_0] = CURR_SCALE / SHUNT_SPARE;
        CURR_SCALES[CHAN_SPARE_1] = CURR_SCALE / SHUNT_SPARE;
        CURR_SCALES[CHAN_SPARE_2] = CURR_SCALE / SHUNT_SPARE;
        CURR_SCALES[CHAN_SPARE_3] = CURR_SCALE / SHUNT_SPARE;
        CURR_SCALES[CHAN_SPARE_4] = CURR_SCALE / SHUNT_SPARE;
        CURR_SCALES[CHAN_SPARE_5] = CURR_SCALE / SHUNT_SPARE;
        CURR_SCALES[CHAN_OTM_0_A] = CURR_SCALE / SHUNT_OTM;
        CURR_SCALES[CHAN_OTM_0_B] = CURR_SCALE / SHUNT_OTM;
        CURR_SCALES[CHAN_OTM_1_A] = CURR_SCALE / SHUNT_OTM;
        CURR_SCALES[CHAN_OTM_1_B] = CURR_SCALE / SHUNT_OTM;
        CURR_SCALES[CHAN_OTM_2_A] = CURR_SCALE / SHUNT_OTM;
        CURR_SCALES[CHAN_OTM_2_B] = CURR_SCALE / SHUNT_OTM;
        CURR_SCALES[CHAN_OTM_3_A] = CURR_SCALE / SHUNT_OTM;
        CURR_SCALES[CHAN_OTM_3_B] = CURR_SCALE / SHUNT_OTM;
        CURR_SCALES[CHAN_OTM_4_A] = CURR_SCALE / SHUNT_OTM;
        CURR_SCALES[CHAN_OTM_4_B] = CURR_SCALE / SHUNT_OTM;
        CURR_SCALES[CHAN_OTM_5_A] = CURR_SCALE / SHUNT_OTM;
        CURR_SCALES[CHAN_OTM_5_B] = CURR_SCALE / SHUNT_OTM;
    }


    /**
     *  Constructor
     */
    public PduLV()
    {
        setValidBoardTypes(VALID_TYPES);
        setValidChannels(VALID_CHAN_MASK);
        setCurrentScales(CURR_SCALES);
        setChannelMap(CHANNEL_MAP);
    }


    /**
     *  Powers on a channel.
     *
     *  If it's an OTM channel, the complementary one must be powered off.
     *
     *  @param  chan  The channel number
     *  @throws  DriverException  if the channel number is invalid
     */
    @Override
    public void powerOn(int chan) throws  DriverException
    {
        super.powerOn(chan);
        if (chan >= CHAN_OTM_0_A && chan <= CHAN_OTM_5_B) {
            powerOff(chan ^ 1);
        }
    }


    /**
     *  Powers on a named channel.
     *
     *  If it's an OTM channel, the complementary one must be powered off.
     *
     *  @param  chan  The channel name
     *  @throws  DriverException  if the channel name is invalid
     */
    @Override
    public void powerOn(String chan) throws  DriverException
    {
        super.powerOn(chan);
        int iChan = CHANNEL_MAP.get(chan);
        if (iChan >= CHAN_OTM_0_A && iChan <= CHAN_OTM_5_B) {
            powerOff(iChan ^ 1);
        }
    }

}
