package org.lsst.ccs.drivers.auxelex;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Routines for controlling the SLAC Ion Pump Controller (IPC).
 *
 *  As of 5/31/18 the board has a defect which causes the channel switches to be
 *  always closed, regardless of the settings in the control register.  Hence the
 *  only way to turn off the power for a channel is to set the voltage to zero.
 *  This also means that the voltage set needs to be cached and only applied to
 *  its DAC when/if the power is "switched on".
 *
 *  If and when a correctly-functioning board is available, the comments marked
 *  with "BADBOARD" should be obeyed.
 *
 *  @author  Owen Saxton
 */
public class Ipc extends Srp {

    /**
     *  Constants and data.
     */
    public static final int
        CHAN_PUMP0   = 0,
        CHAN_PUMP1   = 1,
        CHAN_PUMP2   = 2,
        CHAN_PUMP3   = 3,
        CHAN_PUMP4   = 4,
        CHAN_PUMP5   = 5,
        CHAN_PUMP6   = 6,
        CHAN_PUMP7   = 7,
        CHAN_PUMP8   = 8,
        NUM_CHANNELS = 9;

    private static final Map<String, Integer> channelMap = new HashMap<>();
    static {
        channelMap.put("pump0", CHAN_PUMP0);
        channelMap.put("pump1", CHAN_PUMP1);
        channelMap.put("pump2", CHAN_PUMP2);
        channelMap.put("pump3", CHAN_PUMP3);
        channelMap.put("pump4", CHAN_PUMP4);
        channelMap.put("pump5", CHAN_PUMP5);
        channelMap.put("pump6", CHAN_PUMP6);
        channelMap.put("pump7", CHAN_PUMP7);
        channelMap.put("pump8", CHAN_PUMP8);
    }
    private static final String[] channelNames = new String[NUM_CHANNELS];
    static {
        for (String name : channelMap.keySet()) {
            channelNames[channelMap.get(name)] = name;
        }
    }

    private static final List<BoardType> VALID_TYPES = new ArrayList<>();
    static {
        VALID_TYPES.add(BoardType.ION_PUMP);
    }

    private static final int
        REG_DEVICE_BASE  = 0x10000,
        REG_CONTROL      = REG_DEVICE_BASE,
        REG_IMODE_STATUS = REG_DEVICE_BASE + 1,
        REG_VMODE_STATUS = REG_DEVICE_BASE + 2,
        REG_PMODE_STATUS = REG_DEVICE_BASE + 3,
        CHAN_INCREMENT   = 0x00400,
        REG_CHAN_BASE    = REG_DEVICE_BASE + CHAN_INCREMENT,
        OFF_DAC_CURRENT  = 0,
        OFF_DAC_VOLTAGE  = 1,
        OFF_DAC_POWER    = 2,
        OFF_ADC_CURRENT  = 0x80,
        OFF_ADC_VOLTAGE  = 0x81,
        OFF_ADC_POWER    = 0x82,
        DAC_FULL_SCALE   = 0x10000,  // 16-bit unipolar
        ADC_FULL_SCALE   = 0x1000000 / 2;  // 24-bit bipolar

    private static final double
        DAC_ISCALE = DAC_FULL_SCALE / 0.016,
        DAC_VSCALE = DAC_FULL_SCALE / 6000.0,
        DAC_PSCALE = DAC_FULL_SCALE / 10.0,
        ADC_ISCALE = 0.016 / ADC_FULL_SCALE,
        ADC_VSCALE = 6000.0 / ADC_FULL_SCALE,
        ADC_PSCALE = 10.0 / ADC_FULL_SCALE;

    private final int[] voltDacs = new int[NUM_CHANNELS];  // BADBOARD Remove


    /**
     *  Constructor
     */
    public Ipc()
    {
        setValidBoardTypes(VALID_TYPES);
    }


    /**
     *  Gets the array of channel names
     *
     *  @return  The array of channel names
     */
    public String[] getChannelNames()
    {
        return channelNames;
    }


    /**
     *  Powers on a channel.
     *
     *  @param  chan  The channel number
     *  @throws  DriverException  if the channel number is invalid
     */
    public void powerOn(int chan) throws  DriverException
    {
        updateReg(REG_CONTROL, 1 << checkChannelNumber(chan), -1);
        setVoltageDAC(chan, voltDacs[chan]);  // BADBOARD Remove
    }


    /**
     *  Powers on a named channel.
     *
     *  @param  chan  The channel name
     *  @throws  DriverException  if the channel name is invalid
     */
    public void powerOn(String chan) throws  DriverException
    {
        //updateReg(REG_CONTROL, 1 << getChannelNumber(chan), -1);  // BADBOARD Uncomment
        int chnum = getChannelNumber(chan);      // BADBOARD Remove
        updateReg(REG_CONTROL, 1 << chnum, -1);  // BADBOARD Remove
        setVoltageDAC(chnum, voltDacs[chnum]);   // BADBOARD Remove
    }


    /**
     *  Powers off a channel.
     *
     *  @param  chan  The channel number
     *  @throws  DriverException  if the channel number is invalid
     */
    public void powerOff(int chan) throws  DriverException
    {
        updateReg(REG_CONTROL, 1 << checkChannelNumber(chan), 0);
        setVoltageDAC(chan, 0);   // BADBOARD Remove
    }


    /**
     *  Powers off a named channel.
     *
     *  @param  chan  The channel name
     *  @throws  DriverException  if the channel name is invalid
     */
    public void powerOff(String chan) throws  DriverException
    {
        //updateReg(REG_CONTROL, 1 << getChannelNumber(chan), 0);  // BADBOARD Uncomment
        int chnum = getChannelNumber(chan);     // BADBOARD Remove 
        updateReg(REG_CONTROL, 1 << chnum, 0);  // BADBOARD Remove 
        setVoltageDAC(chnum, 0);                // BADBOARD Remove 
    }


    /**
     *  Gets the powered state of a channel
     *
     *  @param  chan  The channel number
     *  @return  Whether the channel is powered on
     *  @throws  DriverException  if the channel number is invalid
     */
    public boolean isPowered(int chan) throws  DriverException
    {
        return ((readReg(REG_CONTROL) >> checkChannelNumber(chan)) & 1) != 0;
    }


    /**
     *  Gets the powered state of a named channel
     *
     *  @param  chan  The channel name
     *  @return  Whether the channel is powered on
     *  @throws  DriverException  if the channel name is invalid
     */
    public boolean isPowered(String chan) throws  DriverException
    {
        return ((readReg(REG_CONTROL) >> getChannelNumber(chan)) & 1) != 0;
    }


    /**
     *  Gets the overall powered status
     *
     *  @return  The channel control word
     *  @throws  DriverException 
     */
    public int getPoweredStatus() throws  DriverException
    {
        return readReg(REG_CONTROL);
    }


    /**
     *  Gets whether channel is voltage limited
     *
     *  @param  chan  The channel number
     *  @return  Whether the channel is limited
     *  @throws  DriverException  if the channel number is invalid
     */
    public boolean isVoltageLimited(int chan) throws  DriverException
    {
        return ((readReg(REG_VMODE_STATUS) >> checkChannelNumber(chan)) & 1) != 0;
    }


    /**
     *  Gets whether named channel is voltage limited
     *
     *  @param  chan  The channel name
     *  @return  Whether the channel is limited
     *  @throws  DriverException  if the channel name is invalid
     */
    public boolean isVoltageLimited(String chan) throws  DriverException
    {
        return ((readReg(REG_VMODE_STATUS) >> getChannelNumber(chan)) & 1) != 0;
    }


    /**
     *  Gets the voltage limit status word
     *
     *  @return  The limit status word
     *  @throws  DriverException
     */
    public int getVlimitStatus() throws  DriverException
    {
        return readReg(REG_VMODE_STATUS);
    }


    /**
     *  Gets whether channel is current limited
     *
     *  @param  chan  The channel number
     *  @return  Whether the channel is limited
     *  @throws  DriverException  if the channel number is invalid
     */
    public boolean isCurrentLimited(int chan) throws  DriverException
    {
        return ((readReg(REG_IMODE_STATUS) >> checkChannelNumber(chan)) & 1) != 0;
    }


    /**
     *  Gets whether named channel is current limited
     *
     *  @param  chan  The channel name
     *  @return  Whether the channel is limited
     *  @throws  DriverException  if the channel name is invalid
     */
    public boolean isCurrentLimited(String chan) throws  DriverException
    {
        return ((readReg(REG_IMODE_STATUS) >> getChannelNumber(chan)) & 1) != 0;
    }


    /**
     *  Gets the current limit status word
     *
     *  @return  The limit status word
     *  @throws  DriverException
     */
    public int getIlimitStatus() throws  DriverException
    {
        return readReg(REG_IMODE_STATUS);
    }


    /**
     *  Gets whether channel is power limited
     *
     *  @param  chan  The channel number
     *  @return  Whether the channel is limited
     *  @throws  DriverException  if the channel number is invalid
     */
    public boolean isPowerLimited(int chan) throws  DriverException
    {
        return ((readReg(REG_PMODE_STATUS) >> checkChannelNumber(chan)) & 1) != 0;
    }


    /**
     *  Gets whether named channel is power limited
     *
     *  @param  chan  The channel name
     *  @return  Whether the channel is limited
     *  @throws  DriverException  if the channel name is invalid
     */
    public boolean isPowerLimited(String chan) throws  DriverException
    {
        return ((readReg(REG_PMODE_STATUS) >> getChannelNumber(chan)) & 1) != 0;
    }


    /**
     *  Gets the power limit status word
     *
     *  @return  The limit status word
     *  @throws  DriverException
     */
    public int getPlimitStatus() throws  DriverException
    {
        return readReg(REG_PMODE_STATUS);
    }


    /**
     *  Sets the voltage for a channel.
     *
     *  @param  chan   The channel number
     *  @param  value  The value to set
     *  @throws  DriverException  if the channel number is invalid
     */
    public void setVoltage(int chan, double value) throws  DriverException
    {
        //writeReg(REG_CHAN_BASE + CHAN_INCREMENT * checkChannelNumber(chan) + OFF_DAC_VOLTAGE,  // BADBOARD Uncomment
        //         limitDacValue(value * DAC_VSCALE));                                           // BADBOARD Uncomment
        voltDacs[checkChannelNumber(chan)] = limitDacValue(value * DAC_VSCALE);  // BADBOARD Remove
        if (isPowered(chan)) {                                                   // BADBOARD Remove
            setVoltageDAC(chan, voltDacs[chan]);                                 // BADBOARD Remove
        }                                                                        // BADBOARD Remove
    }


    /**
     *  Sets the voltage for a named channel.
     *
     *  @param  chan   The channel name
     *  @param  value  The value to set
     *  @throws  DriverException  if the channel name is invalid
     */
    public void setVoltage(String chan, double value) throws  DriverException
    {
        //writeReg(REG_CHAN_BASE + CHAN_INCREMENT * getChannelNumber(chan) + OFF_DAC_VOLTAGE,  // BADBOARD Uncomment
        //         limitDacValue(value * DAC_VSCALE));                                         // BADBOARD Uncomment
        int chnum = getChannelNumber(chan);                   // BADBOARD Remove
        voltDacs[chnum] = limitDacValue(value * DAC_VSCALE);  // BADBOARD Remove
        if (isPowered(chnum)) {                               // BADBOARD Remove
            setVoltageDAC(chnum, voltDacs[chnum]);            // BADBOARD Remove
        }                                                     // BADBOARD Remove
    }


    // BADBOARD Remove this whole method
    /**
     *  Sets the voltage DAC for a channel
     *
     *  @param  chan   The channel number
     *  @param  value  The value to set
     *  @throws  DriverException  if the channel number is invalid
     */
    private void setVoltageDAC(int chan, int value) throws  DriverException
    {
        writeReg(REG_CHAN_BASE + CHAN_INCREMENT * chan + OFF_DAC_VOLTAGE, value);
    }


    /**
     *  Sets the current for a channel
     *
     *  @param  chan   The channel number
     *  @param  value  The value to set
     *  @throws  DriverException  if the channel number is invalid
     */
    public void setCurrent(int chan, double value) throws  DriverException
    {
        writeReg(REG_CHAN_BASE + CHAN_INCREMENT * checkChannelNumber(chan) + OFF_DAC_CURRENT,
                 limitDacValue(value * DAC_ISCALE));
    }


    /**
     *  Sets the current for a named channel
     *
     *  @param  chan   The channel name
     *  @param  value  The value to set
     *  @throws  DriverException  if the channel name is invalid
     */
    public void setCurrent(String chan, double value) throws  DriverException
    {
        writeReg(REG_CHAN_BASE + CHAN_INCREMENT * getChannelNumber(chan) + OFF_DAC_CURRENT,
                 limitDacValue(value * DAC_ISCALE));
    }


    /**
     *  Sets the power for a channel
     *
     *  @param  chan   The channel number
     *  @param  value  The value to set
     *  @throws  DriverException  if the channel number is invalid
     */
    public void setPower(int chan, double value) throws  DriverException
    {
        writeReg(REG_CHAN_BASE + CHAN_INCREMENT * checkChannelNumber(chan) + OFF_DAC_POWER,
                 limitDacValue(value * DAC_PSCALE));
    }


    /**
     *  Sets the power for a named channel
     *
     *  @param  chan   The channel name
     *  @param  value  The value to set
     *  @throws  DriverException  if the channel name is invalid
     */
    public void setPower(String chan, double value) throws  DriverException
    {
        writeReg(REG_CHAN_BASE + CHAN_INCREMENT * getChannelNumber(chan) + OFF_DAC_POWER,
                 limitDacValue(value * DAC_PSCALE));
    }


    /**
     *  Reads the voltage for a channel
     *
     *  @param  chan  The channel number
     *  @return  The voltage on that channel
     *  @throws  DriverException  if the channel number is invalid
     */
    public double readVoltage(int chan) throws  DriverException
    {
        return readReg(REG_CHAN_BASE + CHAN_INCREMENT * checkChannelNumber(chan) + OFF_ADC_VOLTAGE) * ADC_VSCALE;
    }


    /**
     *  Reads the voltage for a named channel
     *
     *  @param  chan  The channel name
     *  @return  The voltage on that channel
     *  @throws  DriverException  if the channel name is invalid
     */
    public double readVoltage(String chan) throws  DriverException
    {
        return readReg(REG_CHAN_BASE + CHAN_INCREMENT * getChannelNumber(chan) + OFF_ADC_VOLTAGE) * ADC_VSCALE;
    }


    /**
     *  Reads the current for a channel
     *
     *  @param  chan  The channel number
     *  @return  The current on that channel
     *  @throws  DriverException  if the channel number is invalid
     */
    public double readCurrent(int chan) throws  DriverException
    {
        return readReg(REG_CHAN_BASE + CHAN_INCREMENT * checkChannelNumber(chan) + OFF_ADC_CURRENT) * ADC_ISCALE;
    }


    /**
     *  Reads the current for a named channel
     *
     *  @param  chan  The channel name
     *  @return  The current on that channel
     *  @throws  DriverException  if the channel name is invalid
     */
    public double readCurrent(String chan) throws  DriverException
    {
        return readReg(REG_CHAN_BASE + CHAN_INCREMENT * getChannelNumber(chan) + OFF_ADC_CURRENT) * ADC_ISCALE;
    }


    /**
     *  Reads the power for a channel
     *
     *  @param  chan  The channel number
     *  @return  The power on that channel
     *  @throws  DriverException  if the channel number is invalid
     */
    public double readPower(int chan) throws  DriverException
    {
        return readReg(REG_CHAN_BASE + CHAN_INCREMENT * checkChannelNumber(chan) + OFF_ADC_POWER) * ADC_PSCALE;
    }


    /**
     *  Reads the power for a named channel
     *
     *  @param  chan  The channel name
     *  @return  The power on that channel
     *  @throws  DriverException  if the channel name is invalid
     */
    public double readPower(String chan) throws  DriverException
    {
        return readReg(REG_CHAN_BASE + CHAN_INCREMENT * getChannelNumber(chan) + OFF_ADC_POWER) * ADC_PSCALE;
    }


    /**
     *  Checks a channel number for validity
     *
     *  @param  chan  The channel number
     *  @return  The channel number
     *  @throws  DriverException  if the channel number is invalid
     */
    private int checkChannelNumber(int chan) throws  DriverException
    {
        if (chan < 0 || chan >= NUM_CHANNELS) {
            throw new DriverException("Invalid channel number: " + chan);
        }
        return chan;
    }


    /**
     *  Gets a channel number from its name
     *
     *  @param  chan  The channel name
     *  @return  The channel number
     *  @throws  DriverException  if the channel name is invalid
     */
    private int getChannelNumber(String chan) throws  DriverException
    {
        Integer cnum = channelMap.get(chan);
        if (cnum == null) {
            throw new DriverException("Invalid channel name: " + chan);
        }
        return cnum;
    }


    /**
     *  Limits a DAC value
     *
     *  @param  value  The desired value
     *  @return  The limited value
     */
    private static int limitDacValue(double value)
    {
        return Math.max(Math.min((int)(value + 0.5), DAC_FULL_SCALE - 1), 0);
    }

}
