package org.lsst.ccs.drivers.auxelex;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.utilities.conv.Convert;

/**
 *  Routines for controlling the high-voltage (24V & 48V) SLAC PDUs 
 *
 *  @author  Owen Saxton
 */
public class PduHV extends PduCommon {

    /**
     *  Constants and data.
     */
    public static final int
        CHAN_ION_PUMPS     = 0,
        CHAN_UNUSED_24C    = 1,
        CHAN_FES_CHANGER_C = 2,
        CHAN_BPU_MAQ20     = 3,
        CHAN_SHUTTER_1     = 4,
        CHAN_SHUTTER_2     = 5,
        CHAN_FES_CAROUSEL  = 6,
        CHAN_FES_LOADER_C  = 7,
        CHAN_BODY_PURGE    = 8,
        CHAN_GAUGES        = 9,
        CHAN_HCU_PWR_CRY   = 10,
        CHAN_HCU_FES_SHU   = 11,
        NUM_CHANS_24C      = 12,

        CHAN_FES_LOADER_D  = 0,
        CHAN_FES_CHANGER_D = 1,
        CHAN_CRYO_TURBO    = 2,
        CHAN_HEX_TURBO     = 3,
        CHAN_FES_CLAMPS    = 4,
        CHAN_FES_BRAKES    = 5,
        CHAN_SHUTTER_BRAKE = 6,
        NUM_CHANS_24D      = 7,

        CHAN_FES_CAROUSEL_D = 0,
        CHAN_UNUSED_48      = 1,
        CHAN_SHUTTER_1_D    = 2,
        CHAN_SHUTTER_2_D    = 3,
        CHAN_PURGE_FAN      = 4,
        CHAN_FES_HEATER     = 5,
        NUM_CHANS_48        = 6;

    public static final int
        MAIN_MODEL_UNKNOWN = -1,
        MAIN_MODEL_12 = 0,
        MAIN_MODEL_28 = 1,
        MAIN_MODEL_48 = 2;

    public static final int
        MAIN_STATUS_OFF  = 0x01,
        MAIN_STATUS_DSAB = 0x02,
        MAIN_STATUS_IOGB = 0x04,
        MAIN_STATUS_OTW  = 0x08,
        MAIN_STATUS_OTP  = 0x10,
        MAIN_STATUS_VBAD = 0x20,
        MAIN_STATUS_IBAD = 0x40,
        MAIN_STATUS_MASK = 0x7f;

    private static final Map<String, Integer> CHANNEL_MAP_24C = new HashMap<>();
    static {
        CHANNEL_MAP_24C.put("bodyPurge", CHAN_BODY_PURGE);
        CHANNEL_MAP_24C.put("fesChanger", CHAN_FES_CHANGER_C);
        CHANNEL_MAP_24C.put("fesCarousel", CHAN_FES_CAROUSEL);
        CHANNEL_MAP_24C.put("fesLoader", CHAN_FES_LOADER_C);
        CHANNEL_MAP_24C.put("shutter1", CHAN_SHUTTER_1);
        CHANNEL_MAP_24C.put("shutter2", CHAN_SHUTTER_2);
        CHANNEL_MAP_24C.put("bpuMaq20", CHAN_BPU_MAQ20);
        CHANNEL_MAP_24C.put("ionPumps", CHAN_ION_PUMPS);
        CHANNEL_MAP_24C.put("gauges", CHAN_GAUGES);
        CHANNEL_MAP_24C.put("hcuPwrCry", CHAN_HCU_PWR_CRY);
        CHANNEL_MAP_24C.put("hcuFesShu", CHAN_HCU_FES_SHU);
        CHANNEL_MAP_24C.put("unused", CHAN_UNUSED_24C);
    }

    private static final Map<String, Integer> CHANNEL_MAP_24D = new HashMap<>();
    static {
        CHANNEL_MAP_24D.put("fesLoader", CHAN_FES_LOADER_D);
        CHANNEL_MAP_24D.put("fesChanger", CHAN_FES_CHANGER_D);
        CHANNEL_MAP_24D.put("cryoTurbo", CHAN_CRYO_TURBO);
        CHANNEL_MAP_24D.put("hexTurbo", CHAN_HEX_TURBO);
        CHANNEL_MAP_24D.put("fesClamps", CHAN_FES_CLAMPS);
        CHANNEL_MAP_24D.put("fesBrakes", CHAN_FES_BRAKES);
        CHANNEL_MAP_24D.put("shuBrakes", CHAN_SHUTTER_BRAKE);
    }

    private static final Map<String, Integer> CHANNEL_MAP_48 = new HashMap<>();
    static {
        CHANNEL_MAP_48.put("fesCarousel", CHAN_FES_CAROUSEL_D);
        CHANNEL_MAP_48.put("fesHeater", CHAN_FES_HEATER);
        CHANNEL_MAP_48.put("shutter1", CHAN_SHUTTER_1_D);
        CHANNEL_MAP_48.put("shutter2", CHAN_SHUTTER_2_D);
        CHANNEL_MAP_48.put("purgeFan", CHAN_PURGE_FAN);
        CHANNEL_MAP_48.put("unused", CHAN_UNUSED_48);
    }

    private static final List<BoardType> VALID_TYPES = new ArrayList<>();
    static {
        VALID_TYPES.add(BoardType.PDU_24V_CLEAN);
        VALID_TYPES.add(BoardType.PDU_24V_DIRTY);
        VALID_TYPES.add(BoardType.PDU_48V);
    }

    private static final int
        VALID_CHAN_MASK_24C = 0x0ffd,
        VALID_CHAN_MASK_24D = 0x7f,
        VALID_CHAN_MASK_48 = 0x03d;

    private static final double
        SHUNT_040 = 0.0040;

    private static final int
        MAX_CHANNELS = 12,
        REG_LAMBDA_BASE = REG_ADC_BASE + MAX_CHANNELS * ADC_INCREMENT,
        REG_SERIAL_NO  = REG_LAMBDA_BASE,
        REG_FW_VERSION = REG_LAMBDA_BASE + 5,
        REG_PR_VERSION = REG_LAMBDA_BASE + 6,
        REG_VOLTAGE    = REG_LAMBDA_BASE + 7,
        REG_CURRENT    = REG_LAMBDA_BASE + 8,
        REG_TEMP       = REG_LAMBDA_BASE + 9,
        REG_STATUS     = REG_LAMBDA_BASE + 10,
        REG_PART_NO    = REG_LAMBDA_BASE + 11,
        REG_MANU_DATE  = REG_LAMBDA_BASE + 14,
        REG_MANU_LOCN  = REG_LAMBDA_BASE + 16,
        LENG_SERIAL_NO  = 20,
        LENG_FW_VERSION = 4,
        LENG_PR_VERSION = 4,
        LENG_PART_NO    = 12,
        LENG_MANU_DATE  = 8,
        LENG_MANU_LOCN  = 3,
        ADC_BAD_MASK    = 0xfffffc00;
    private static final Map<String, Integer> MAIN_MODEL_MAP = new HashMap<>();
    static {
        MAIN_MODEL_MAP.put("12", MAIN_MODEL_12);
        MAIN_MODEL_MAP.put("28", MAIN_MODEL_28);
        MAIN_MODEL_MAP.put("48", MAIN_MODEL_48);
    }
    private static final double[]
        V_SCALES = {0.01668, 0.03973, 0.06772},
        I_SCALES = {0.0815, 0.0488, 0.0282};
    private static final double
        T_SCALE = 2.048;

    private int mainModel;
    private double vScale, iScale;


    /**
     *  Constructor
     */
    public PduHV()
    {
        setValidBoardTypes(VALID_TYPES);
    }


    /**
     *  Opens a connection to a board.
     *
     *  @param  host  The host name or IP address, or null or empty for simulation
     *  @param  port  The port number
     *  @throws  DriverException
     */
    @Override
    public synchronized void open(String host, int port) throws DriverException
    {
        super.open(host, port);
        BoardType type = getBoardType();
        int numChans;
        if (type == BoardType.PDU_48V) {
            setValidChannels(VALID_CHAN_MASK_48);
            //setCurrentScales(CURR_SCALES_48);
            setChannelMap(CHANNEL_MAP_48);
            numChans = NUM_CHANS_48;
        }
        else if (type == BoardType.PDU_24V_DIRTY) {
            setValidChannels(VALID_CHAN_MASK_24D);
            //setCurrentScales(CURR_SCALES_24D);
            setChannelMap(CHANNEL_MAP_24D);
            numChans = NUM_CHANS_24D;
        }
        else {
            setValidChannels(VALID_CHAN_MASK_24C);
            //setCurrentScales(CURR_SCALES_24C);
            setChannelMap(CHANNEL_MAP_24C);
            numChans = NUM_CHANS_24C;
        }
        double[] currScales = new double[numChans];
        for (int j = 0; j < numChans; j++) {
            currScales[j] = CURR_SCALE / SHUNT_040;
        }
        setCurrentScales(currScales);
        mainModel = MAIN_MODEL_UNKNOWN;
        try {
            String mName = getMainPartNo();
            if (mName.startsWith("CPFE1000FI")) {
                Integer mod = MAIN_MODEL_MAP.get(mName.substring(10));
                mainModel = mod == null ? mainModel : mod;
            }
            if (mainModel == MAIN_MODEL_UNKNOWN) {
                throw new DriverException("Unrecognized Lambda PS model: " + mName);
            }
            vScale = V_SCALES[mainModel];
            iScale = I_SCALES[mainModel];
        }
        catch (DriverException e) {
            close();
            throw e;
        }
    }


    /**
     *  Gets the main PS model.
     *
     *  @return  The model type
     */
    public int getMainModel()
    {
        return mainModel;
    }


    /**
     *  Gets the main PS serial number.
     *
     *  @return  The serial number
     *  @throws  DriverException
     */
    public String getMainSerialNo() throws DriverException
    {
        return readString(REG_SERIAL_NO, LENG_SERIAL_NO);
    }


    /**
     *  Gets the main PS firmware version.
     *
     *  @return  The firmware revision
     *  @throws  DriverException
     */
    public String getMainFwVersion() throws DriverException
    {
        return readString(REG_FW_VERSION, LENG_FW_VERSION);
    }


    /**
     *  Gets the main PS product version.
     *
     *  @return  The product revision
     *  @throws  DriverException
     */
    public String getMainPrVersion() throws DriverException
    {
        return readString(REG_PR_VERSION, LENG_PR_VERSION);
    }


    /**
     *  Gets the main PS part number.
     *
     *  @return  The unit part number
     *  @throws  DriverException
     */
    public String getMainPartNo() throws DriverException
    {
        return readString(REG_PART_NO, LENG_PART_NO);
    }


    /**
     *  Gets the main PS manufacturing date.
     *
     *  @return  The manufacturing date
     *  @throws  DriverException
     */
    public String getMainManuDate() throws DriverException
    {
        return readString(REG_MANU_DATE, LENG_MANU_DATE);
    }


    /**
     *  Gets the main PS manufacturing location.
     *
     *  @return  The manufacturing location
     *  @throws  DriverException
     */
    public String getMainManuLocn() throws DriverException
    {
        return readString(REG_MANU_LOCN, LENG_MANU_LOCN);
    }


    /**
     *  Reads the main PS status.
     *
     *  @return  The contents of the status register
     *  @throws  DriverException
     */
    public int readMainStatus() throws DriverException
    {
        return readReg(REG_STATUS) & MAIN_STATUS_MASK;
    }


    /**
     *  Reads the main PS voltage.
     *
     *  @return  The measured voltage
     *  @throws  DriverException
     */
    public double readMainVoltage() throws DriverException
    {
        return fixValue(readReg(REG_VOLTAGE)) * vScale;
    }


    /**
     *  Reads the main PS current.
     *
     *  @return  The measured current
     *  @throws  DriverException
     */
    public double readMainCurrent() throws DriverException
    {
        return fixValue(readReg(REG_CURRENT)) * iScale;
    }


    /**
     *  Reads the main PS baseplate temperature.
     *
     *  @return  The measured temperature (C)
     *  @throws  DriverException
     */
    public double readMainTemperature() throws DriverException
    {
        return (fixValue(readReg(REG_TEMP)) - 610) / T_SCALE + 25.0;
    }


    /**
     *  Fixes a value.
     *
     *  If the raw value is 2^10 or greater, NaN is returned.
     *
     *  @param  raw    The raw value
     *  @return  The calculated value
     */
    private static double fixValue(int raw)
    {
        return (raw & ADC_BAD_MASK) == 0 ? raw : Double.NaN;
    }


    /**
     *  Reads a string from the Lambda PS
     *
     *  @param  addr  The register address
     *  @param  leng  The length of the string
     *  @return  The string
     *  @throws  DriverException
     */
    private String readString(int addr, int leng) throws  DriverException
    {
        int[] words = readRegs(addr, (leng + 3) / 4);
        byte[] bytes = new byte[4 * words.length];
        for (int j = 0; j < words.length; j++) {
            Convert.intToBytesBE(words[j], bytes, 4 * j);
        }
        return new String(bytes, 0, leng);
    }

}
