package org.lsst.ccs.drivers.auxelex;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Common code for testing the PDU drivers
 * 
 *  @author Owen Saxton
 */
public class TestPduCommon extends TestSrp {

    /**
     *  Fields
     */
    protected final PduCommon pdu;

    /**
     *  Constructor
     *
     *  @param  pdu  The PDU object to use
     */
    public TestPduCommon(PduCommon pdu)
    {
        super(pdu);
        this.pdu = pdu;
    }

    /*
     *  Commands
     */
    @Command(description="Turn on the main power")
    public void mainOn() throws DriverException
    {
        pdu.setMainOn();
    }

    @Command(description="Turn off the main power")
    public void mainOff() throws DriverException
    {
        pdu.setMainOff();
    }

    @Command(description="Get the main power state")
    public String showMain() throws DriverException
    {
        return pdu.isMainOn() ? "on" : "off";
    }

    @Command(description="Show the main power status")
    public String showMainStatus() throws DriverException
    {
        return String.format("0x%02x", pdu.getMainStatus());
    }

    @Command(description="Turn on a channel")
    public void channelOn(@Argument(description="Channel number")
                          int chan) throws DriverException
    {
        pdu.setChannelOn(chan);
    }

    @Command(description="Turn on a named channel")
    public void channelOnN(@Argument(description="Channel name")
                           String chan) throws DriverException
    {
        pdu.setChannelOn(chan);
    }

    @Command(description="Turn off a channel")
    public void channelOff(@Argument(description="Channel number")
                           int chan) throws DriverException
    {
        pdu.setChannelOff(chan);
    }

    @Command(description="Turn off a named channel")
    public void channelOffN(@Argument(description="Channel name")
                            String chan) throws DriverException
    {
        pdu.setChannelOff(chan);
    }

    @Command(description="Show the state of a channel")
    public String showChannel(@Argument(description="Channel number")
                              int chan) throws DriverException
    {
        return pdu.isChannelOn(chan) ? "on" : "off";
    }

    @Command(description="Show the state of a named channel")
    public String showChannelN(@Argument(description="Channel name")
                               String chan) throws DriverException
    {
        return pdu.isChannelOn(chan) ? "on" : "off";
    }

    @Command(description="Show the valid channel mask")
    public String showValidChannels() throws DriverException
    {
        return String.format("0x%06x", pdu.getValidChannels());
    }

    @Command(description="Show the channel switch status")
    public String showSwitchStatus() throws DriverException
    {
        return String.format("0x%06x", pdu.getSwitchStatus());
    }

    @Command(description="Show the channel alert status")
    public String showAlertStatus() throws DriverException
    {
        return String.format("0x%06x", pdu.getAlertStatus());
    }

    @Command(description="Show the link fault word")
    public int showLinkFault() throws DriverException
    {
        return pdu.getLinkFault();
    }

    @Command(description="Read a channel's voltage")
    public double readVoltage(@Argument(description="Channel number")
                              int chan) throws DriverException
    {
        return pdu.readVoltage(chan);
    }

    @Command(description="Read a channel's current")
    public double readCurrent(@Argument(description="Channel number")
                              int chan) throws DriverException
    {
        return pdu.readCurrent(chan);
    }

    @Command(description="Show the channel names")
    public String showChannelNames() throws DriverException
    {
        StringBuilder text = new StringBuilder();
        String[] names = pdu.getChannelNames();
        String prefix = "";
        for (String name : names) {
            text.append(prefix).append(name == null ? "-" : name);
            prefix = " ";
        }
        return text.toString();
    }

    @Command(description="Show the complete power state")
    public String showPower() throws DriverException
    {
        StringBuilder text = new StringBuilder("Chan   Name     State  Alert  Volts   mAmps");
        String[] names = pdu.getChannelNames();
        int mask = pdu.getValidChannels();
        int alerts = pdu.getAlertStatus();
        for (int chan = 0; mask != 0; chan++, mask >>= 1, alerts >>= 1) {
            if ((mask & 1) == 0) continue;
            text.append(String.format("\n %2s  %-10s  %3s    %3s   %5.2f  %6.1f",
                        chan, names[chan], pdu.isChannelOn(chan) ? " on" : "off",
                        (alerts & 1) != 0 ? " on" : "off",
                        pdu.readVoltage(chan), 1000 * pdu.readCurrent(chan)));
        }
        return text.toString();
    }

}
