package org.lsst.ccs.drivers.auxelex;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.utilities.conv.Convert;

/**
 *  Routines for controlling the high-voltage (24V & 48V) SLAC PDUs 
 *
 *  @author  Owen Saxton
 */
public class PduHV extends PduCommon {

    /**
     *  Constants and data.
     */
    public static final int
        CHAN_BODY_PURGE = 0,
        CHAN_FES        = 1,
        CHAN_SHUTTER_C  = 2,
        CHAN_CRYOSTAT   = 3,
        CHAN_HEX        = 4,
        CHAN_ION_PUMPS  = 5,
        CHAN_GAUGES     = 6,
        CHAN_NETWORK0   = 7,
        CHAN_NETWORK1   = 8,
        CHAN_NETWORK2   = 9,
        CHAN_CMS600     = 10,

        CHAN_CRYO_TURBO = 0,
        CHAN_HEX_TURBO  = 1,
        CHAN_FES1       = 2,
        CHAN_FES2       = 3,
        CHAN_SHUTTER_D  = 4,

        CHAN_PURGE_FAN  = 0,
        CHAN_SHUTTER1   = 1,
        CHAN_SHUTTER2   = 2;

    public static final int
        MAIN_MODEL_UNKNOWN = -1,
        MAIN_MODEL_12 = 0,
        MAIN_MODEL_28 = 1,
        MAIN_MODEL_48 = 2;

    public static final int
        MAIN_STATUS_OFF  = 0x01,
        MAIN_STATUS_DSAB = 0x02,
        MAIN_STATUS_IOGB = 0x04,
        MAIN_STATUS_OTW  = 0x08,
        MAIN_STATUS_OTP  = 0x10,
        MAIN_STATUS_VBAD = 0x20,
        MAIN_STATUS_IBAD = 0x40,
        MAIN_STATUS_MASK = 0x7f;

    private static final Map<String, Integer> CHANNEL_MAP_24C = new HashMap<>();
    static {
        CHANNEL_MAP_24C.put("bodyPurge", CHAN_BODY_PURGE);
        CHANNEL_MAP_24C.put("fes", CHAN_FES);
        CHANNEL_MAP_24C.put("shutter", CHAN_SHUTTER_C);
        CHANNEL_MAP_24C.put("cryostat", CHAN_CRYOSTAT);
        CHANNEL_MAP_24C.put("hex", CHAN_HEX);
        CHANNEL_MAP_24C.put("ionPumps", CHAN_ION_PUMPS);
        CHANNEL_MAP_24C.put("gauges", CHAN_GAUGES);
        CHANNEL_MAP_24C.put("network0", CHAN_NETWORK0);
        CHANNEL_MAP_24C.put("network1", CHAN_NETWORK1);
        CHANNEL_MAP_24C.put("network2", CHAN_NETWORK2);
        CHANNEL_MAP_24C.put("cms600", CHAN_CMS600);
    }
    private static final Map<String, Integer> CHANNEL_MAP_24D = new HashMap<>();
    static {
        CHANNEL_MAP_24D.put("cryoTurbo", CHAN_CRYO_TURBO);
        CHANNEL_MAP_24D.put("hexTurbo", CHAN_HEX_TURBO);
        CHANNEL_MAP_24D.put("fes1", CHAN_FES1);
        CHANNEL_MAP_24D.put("fes2", CHAN_FES2);
        CHANNEL_MAP_24D.put("shutter", CHAN_SHUTTER_D);
    }
    private static final Map<String, Integer> CHANNEL_MAP_48 = new HashMap<>();
    static {
        CHANNEL_MAP_48.put("purgeFan", CHAN_PURGE_FAN);
        CHANNEL_MAP_48.put("shutter1", CHAN_SHUTTER1);
        CHANNEL_MAP_48.put("shutter2", CHAN_SHUTTER2);
    }
    private static final List<BoardType> VALID_TYPES = new ArrayList<>();
    static {
        VALID_TYPES.add(BoardType.PDU_24V_CLEAN);
        VALID_TYPES.add(BoardType.PDU_24V_DIRTY);
        VALID_TYPES.add(BoardType.PDU_48V);
    }
    private static final int
        VALID_CHAN_MASK_48 = 0x07,
        VALID_CHAN_MASK_24D = 0x1f,
        VALID_CHAN_MASK_24C = 0x07ff;
    private static final double
        SHUNT_102 = 0.01024,
        SHUNT_128 = 0.0128,
        SHUNT_204 = 0.02048,
        SHUNT_512 = 0.0512;
    private static final double[] CURR_SCALES_48 =
       {CURR_SCALE / SHUNT_128, CURR_SCALE / SHUNT_128, CURR_SCALE / SHUNT_128};
    private static final double[] CURR_SCALES_24D =
       {CURR_SCALE / SHUNT_102, CURR_SCALE / SHUNT_204,
        CURR_SCALE / SHUNT_102, CURR_SCALE / SHUNT_102, CURR_SCALE / SHUNT_512};
    private static final double[] CURR_SCALES_24C =
       {CURR_SCALE / SHUNT_204, CURR_SCALE / SHUNT_204, CURR_SCALE / SHUNT_204,
        CURR_SCALE / SHUNT_512, CURR_SCALE / SHUNT_512, CURR_SCALE / SHUNT_128,
        CURR_SCALE / SHUNT_128, CURR_SCALE / SHUNT_512, CURR_SCALE / SHUNT_512,
        CURR_SCALE / SHUNT_512, CURR_SCALE / SHUNT_512};
    private static final int
        MAX_CHANNELS = 12,
        REG_LAMBDA_BASE = REG_ADC_BASE + MAX_CHANNELS * ADC_INCREMENT,
        REG_SERIAL_NO  = REG_LAMBDA_BASE,
        REG_FW_VERSION = REG_LAMBDA_BASE + 5,
        REG_PR_VERSION = REG_LAMBDA_BASE + 6,
        REG_VOLTAGE    = REG_LAMBDA_BASE + 7,
        REG_CURRENT    = REG_LAMBDA_BASE + 8,
        REG_TEMP       = REG_LAMBDA_BASE + 9,
        REG_STATUS     = REG_LAMBDA_BASE + 10,
        REG_PART_NO    = REG_LAMBDA_BASE + 11,
        REG_MANU_DATE  = REG_LAMBDA_BASE + 14,
        REG_MANU_LOCN  = REG_LAMBDA_BASE + 16,
        LENG_SERIAL_NO  = 20,
        LENG_FW_VERSION = 4,
        LENG_PR_VERSION = 4,
        LENG_PART_NO    = 12,
        LENG_MANU_DATE  = 8,
        LENG_MANU_LOCN  = 3,
        ADC_BAD_MASK    = 0xfffffc00;
    private static final Map<String, Integer> MAIN_MODEL_MAP = new HashMap<>();
    static {
        MAIN_MODEL_MAP.put("12", MAIN_MODEL_12);
        MAIN_MODEL_MAP.put("28", MAIN_MODEL_28);
        MAIN_MODEL_MAP.put("48", MAIN_MODEL_48);
    }
    private static final double[]
        V_SCALES = {0.01668, 0.03973, 0.06772},
        I_SCALES = {0.0815, 0.0488, 0.0282};
    private static final double
        T_SCALE = 2.048;

    private int mainModel;
    private double vScale, iScale;


    /**
     *  Constructor
     */
    public PduHV()
    {
        setValidBoardTypes(VALID_TYPES);
        setProbeAddress(REG_DEVICE_BASE);
    }


    /**
     *  Opens a connection to a board.
     *
     *  @param  host  The host name or IP address, or null or empty for simulation
     *  @param  port  The port number
     *  @throws  DriverException
     */
    @Override
    public synchronized void open(String host, int port) throws DriverException
    {
        super.open(host, port);
        BoardType type = getBoardType();
        if (type == BoardType.PDU_48V) {
            setValidChannels(VALID_CHAN_MASK_48);
            setCurrentScales(CURR_SCALES_48);
            setChannelMap(CHANNEL_MAP_48);
        }
        else if (type == BoardType.PDU_24V_DIRTY) {
            setValidChannels(VALID_CHAN_MASK_24D);
            setCurrentScales(CURR_SCALES_24D);
            setChannelMap(CHANNEL_MAP_24D);
        }
        else {
            setValidChannels(VALID_CHAN_MASK_24C);
            setCurrentScales(CURR_SCALES_24C);
            setChannelMap(CHANNEL_MAP_24C);
        }
        mainModel = MAIN_MODEL_UNKNOWN;
        try {
            String mName = getMainPartNo();
            if (mName.startsWith("CPFE1000FI")) {
                Integer mod = MAIN_MODEL_MAP.get(mName.substring(10));
                mainModel = mod == null ? mainModel : mod;
            }
            if (mainModel == MAIN_MODEL_UNKNOWN) {
                throw new DriverException("Unrecognized Lambda PS model: " + mName);
            }
            vScale = V_SCALES[mainModel];
            iScale = I_SCALES[mainModel];
        }
        catch (DriverException e) {
            close();
            throw e;
        }
    }


    /**
     *  Gets the main PS model.
     *
     *  @return  The model type
     */
    public int getMainModel()
    {
        return mainModel;
    }


    /**
     *  Gets the main PS serial number.
     *
     *  @return  The serial number
     *  @throws  DriverException
     */
    public String getMainSerialNo() throws DriverException
    {
        return readString(REG_SERIAL_NO, LENG_SERIAL_NO);
    }


    /**
     *  Gets the main PS firmware version.
     *
     *  @return  The firmware revision
     *  @throws  DriverException
     */
    public String getMainFwVersion() throws DriverException
    {
        return readString(REG_FW_VERSION, LENG_FW_VERSION);
    }


    /**
     *  Gets the main PS product version.
     *
     *  @return  The product revision
     *  @throws  DriverException
     */
    public String getMainPrVersion() throws DriverException
    {
        return readString(REG_PR_VERSION, LENG_PR_VERSION);
    }


    /**
     *  Gets the main PS part number.
     *
     *  @return  The unit part number
     *  @throws  DriverException
     */
    public String getMainPartNo() throws DriverException
    {
        return readString(REG_PART_NO, LENG_PART_NO);
    }


    /**
     *  Gets the main PS manufacturing date.
     *
     *  @return  The manufacturing date
     *  @throws  DriverException
     */
    public String getMainManuDate() throws DriverException
    {
        return readString(REG_MANU_DATE, LENG_MANU_DATE);
    }


    /**
     *  Gets the main PS manufacturing location.
     *
     *  @return  The manufacturing location
     *  @throws  DriverException
     */
    public String getMainManuLocn() throws DriverException
    {
        return readString(REG_MANU_LOCN, LENG_MANU_LOCN);
    }


    /**
     *  Reads the main PS status.
     *
     *  @return  The contents of the status register
     *  @throws  DriverException
     */
    public int readMainStatus() throws DriverException
    {
        return readReg(REG_STATUS) & MAIN_STATUS_MASK;
    }


    /**
     *  Reads the main PS voltage.
     *
     *  @return  The measured voltage
     *  @throws  DriverException
     */
    public double readMainVoltage() throws DriverException
    {
        return fixValue(readReg(REG_VOLTAGE)) * vScale;
    }


    /**
     *  Reads the main PS current.
     *
     *  @return  The measured current
     *  @throws  DriverException
     */
    public double readMainCurrent() throws DriverException
    {
        return fixValue(readReg(REG_CURRENT)) * iScale;
    }


    /**
     *  Reads the main PS baseplate temperature.
     *
     *  @return  The measured temperature (C)
     *  @throws  DriverException
     */
    public double readMainTemperature() throws DriverException
    {
        return (fixValue(readReg(REG_TEMP)) - 610) / T_SCALE + 25.0;
    }


    /**
     *  Fixes a value.
     *
     *  If the raw value is 2^10 or greater, NaN is returned.
     *
     *  @param  raw    The raw value
     *  @return  The calculated value
     */
    private static double fixValue(int raw)
    {
        return (raw & ADC_BAD_MASK) == 0 ? raw : Double.NaN;
    }


    /**
     *  Reads a string from the Lambda PS
     *
     *  @param  addr  The register address
     *  @param  leng  The length of the string
     *  @return  The string
     *  @throws  DriverException
     */
    private String readString(int addr, int leng) throws  DriverException
    {
        int[] words = readRegs(addr, (leng + 3) / 4);
        byte[] bytes = new byte[4 * words.length];
        for (int j = 0; j < words.length; j++) {
            Convert.intToBytesBE(words[j], bytes, 4 * j);
        }
        return new String(bytes, 0, leng);
    }

}
