package org.lsst.ccs.drivers.auxelex;

import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Common routines for controlling the SLAC PDUs 
 *
 *  @author  Owen Saxton
 */
public class PduCommon extends Srp {

    /**
     *  Constants and data.
     */
    protected static final int
        REG_DEVICE_BASE = 0x10000,
        REG_SWITCHES    = REG_DEVICE_BASE,
        REG_MAIN_SWITCH = REG_DEVICE_BASE + 1,
        REG_MAIN_STATUS = REG_DEVICE_BASE + 2,
        REG_ALERTS      = REG_DEVICE_BASE + 3,
        REG_LINK_FAULT  = REG_DEVICE_BASE + 4,
        ADC_INCREMENT   = 0x00400,
        REG_ADC_BASE    = REG_DEVICE_BASE + ADC_INCREMENT,
        ADC_CONTROL     = 0,
        ADC_ALERT       = 1,
        ADC_STATUS      = 2,
        ADC_FAULT       = 3,
        ADC_FAULT_COR   = 4,
        ADC_POWER_BASE  = 5,
        ADC_SENSE_BASE  = 10,
        ADC_V_IN_BASE   = 15,
        ADC_AD_IN_BASE  = 20,
        ADC_CURR_MEAS   = 0,
        ADC_MAX_MEAS    = 1,
        ADC_MIN_MEAS    = 2,
        ADC_MAX_THRESH  = 3,
        ADC_MIN_THRESH  = 4;

    protected static final double
        VOLT_SCALE = 102.4 / 4096,
        CURR_SCALE = 0.1024 / 4096;

    private int validChanMask;
    private double[] currentScales;
    private Map<String, Integer> channelMap;
    private String[] channelNames;


    /**
     *  Sets the mask of valid channels
     *
     *  @param  chans  The valid channel mask
     */
    protected void setValidChannels(int chans)
    {
        validChanMask = chans;
    }


    /**
     *  Gets the mask of valid channels
     *
     *  @return  The valid channel mask
     */
    public int getValidChannels()
    {
        return validChanMask;
    }


    /**
     *  Sets the array of current scale factors
     *
     *  @param  scales  The array of scales, indexed by channel number
     */
    protected void setCurrentScales(double[] scales)
    {
        currentScales = scales;
    }


    /**
     *  Sets the map of channel names
     *
     *  @param  chans  The map of channel names to numbers
     */
    protected void setChannelMap(Map chans)
    {
        channelMap = chans;
        int maxChan = -1;
        for (int chan : channelMap.values()) {
            maxChan = Math.max(maxChan, chan);
        }
        channelNames = new String[maxChan + 1];
        for (String name : channelMap.keySet()) {
            channelNames[channelMap.get(name)] = name;
        }
    }


    /**
     *  Gets the array of channel names
     *
     *  @return  The array of channel names
     */
    public String[] getChannelNames()
    {
        return channelNames;
    }


    /**
     *  Turns on the main power
     *
     *  @throws  DriverException
     */
    public void setMainOn() throws  DriverException
    {
        writeReg(REG_MAIN_SWITCH, 1);
    }


    /**
     *  Turns off the main power
     *
     *  @throws  DriverException
     */
    public void setMainOff() throws  DriverException
    {
        writeReg(REG_MAIN_SWITCH, 0);
    }


    /**
     *  Gets the state of the main power
     *
     *  @return  Whether the main power is on
     *  @throws  DriverException
     */
    public boolean isMainOn() throws  DriverException
    {
        return readReg((REG_MAIN_SWITCH) & 1) != 0;
    }


    /**
     *  Gets the status of the main power
     *
     *  @return  The status word
     *  @throws  DriverException
     */
    public int getMainStatus() throws  DriverException
    {
        return readReg(REG_MAIN_STATUS);
    }


    /**
     *  Turns on a channel
     *
     *  @param  chan  The channel number
     *  @throws  DriverException  if the channel number is invalid
     */
    public synchronized void setChannelOn(int chan) throws  DriverException
    {
        checkChannelNumber(chan);
        updateReg(REG_SWITCHES, 1 << chan, 1 << chan);
    }


    /**
     *  Turns on a named channel
     *
     *  @param  chan  The channel name
     *  @throws  DriverException  if the channel name is invalid
     */
    public synchronized void setChannelOn(String chan) throws  DriverException
    {
        int mask = 1 << getChannelNumber(chan);
        updateReg(REG_SWITCHES, mask, mask);
    }


    /**
     *  Turns off a channel
     *
     *  @param  chan  The channel number
     *  @throws  DriverException  if the channel number is invalid
     */
    public synchronized void setChannelOff(int chan) throws  DriverException
    {
        checkChannelNumber(chan);
        updateReg(REG_SWITCHES, 1 << chan, 0);
    }


    /**
     *  Turns off a named channel
     *
     *  @param  chan  The channel name
     *  @throws  DriverException  if the channel name is invalid
     */
    public synchronized void setChannelOff(String chan) throws  DriverException
    {
        updateReg(REG_SWITCHES, 1 << getChannelNumber(chan), 0);
    }


    /**
     *  Gets the state of a channel's power
     *
     *  @param  chan  The channel number
     *  @return  Whether the channel is powered on
     *  @throws  DriverException  if the channel number is invalid
     */
    public boolean isChannelOn(int chan) throws  DriverException
    {
        checkChannelNumber(chan);
        return ((readReg(REG_SWITCHES) >> chan) & 1) != 0;
    }


    /**
     *  Gets the state of a named channel's power
     *
     *  @param  chan  The channel name
     *  @return  Whether the channel is powered on
     *  @throws  DriverException  if the channel name is invalid
     */
    public boolean isChannelOn(String chan) throws  DriverException
    {
        return ((readReg(REG_SWITCHES) >> getChannelNumber(chan)) & 1) != 0;
    }


    /**
     *  Gets the channel power status
     *
     *  @return  The channel switch word
     *  @throws  DriverException 
     */
    public int getSwitchStatus() throws  DriverException
    {
        return readReg(REG_SWITCHES);
    }


    /**
     *  Gets the status of the alerts
     *
     *  @return  The alert status word
     *  @throws  DriverException
     */
    public int getAlertStatus() throws  DriverException
    {
        return readReg(REG_ALERTS);
    }


    /**
     *  Gets the I2C link fault indicator
     *
     *  @return  The fault word
     *  @throws  DriverException
     */
    public int getLinkFault() throws  DriverException
    {
        return readReg(REG_LINK_FAULT);
    }


    /**
     *  Reads a channel's voltage
     *
     *  @param  chan  The channel number
     *  @return  The voltage on that channel
     *  @throws  DriverException  if the channel number is invalid
     */
    public double readVoltage(int chan) throws  DriverException
    {
        checkChannelNumber(chan);
        return readReg(REG_ADC_BASE + ADC_INCREMENT * chan + ADC_V_IN_BASE + ADC_CURR_MEAS) * VOLT_SCALE;
    }


    /**
     *  Reads a named channel's voltage
     *
     *  @param  chan  The channel name
     *  @return  The voltage on that channel
     *  @throws  DriverException  if the channel name is invalid
     */
    public double readVoltage(String chan) throws  DriverException
    {
        return readReg(REG_ADC_BASE + ADC_INCREMENT * getChannelNumber(chan) + ADC_V_IN_BASE + ADC_CURR_MEAS)
                 * VOLT_SCALE;
    }


    /**
     *  Reads a channel's current
     *
     *  @param  chan  The channel number
     *  @return  The current on that channel
     *  @throws  DriverException  if the channel number is invalid
     */
    public double readCurrent(int chan) throws  DriverException
    {
        checkChannelNumber(chan);
        return readReg(REG_ADC_BASE + ADC_INCREMENT * chan + ADC_SENSE_BASE + ADC_CURR_MEAS) * currentScales[chan];
    }


    /**
     *  Reads a named channel's current
     *
     *  @param  chan  The channel name
     *  @return  The current on that channel
     *  @throws  DriverException  if the channel name is invalid
     */
    public double readCurrent(String chan) throws  DriverException
    {
        int cnum = getChannelNumber(chan);
        return readReg(REG_ADC_BASE + ADC_INCREMENT * cnum + ADC_SENSE_BASE + ADC_CURR_MEAS) * currentScales[cnum];
    }


    /**
     *  Checks a channel number for validity
     *
     *  @param  chan  The channel number
     *  @throws  DriverException  if the channel number is invalid
     */
    private void checkChannelNumber(int chan) throws  DriverException
    {
        if (chan < 0 || ((1 << chan) & validChanMask) == 0) {
            throw new DriverException("Invalid channel number (" + chan + ")");
        }
    }


    /**
     *  Gets a channel number from its name
     *
     *  @param  chan  The channel name
     *  @return  The channel number
     *  @throws  DriverException  if the channel name is invalid
     */
    private int getChannelNumber(String chan) throws  DriverException
    {
        Integer cnum = channelMap.get(chan);
        if (cnum == null) {
            throw new DriverException("Invalid channel name (" + chan + ")");
        }
        return cnum;
    }

}
