package org.lsst.ccs.drivers.auxelex;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Routines for controlling the power breaker-filter-relay board (BFR)
 *
 *  @author  Owen Saxton
 */
public class Bfr extends Srp {

    /**
     *  Constants and data.
     */
    public static final int
        RELAY_HTR_0 = 0,
        RELAY_HTR_1 = 1,
        RELAY_PWS_1 = 2,
        RELAY_PWS_3 = 3,
        RELAY_PWS_4 = 4,
        RELAY_PWS_16 = 5,
        RELAY_PWS_17 = 6,
        RELAY_PWS_18 = 7,
        RELAY_PWS_19 = 8,
        RELAY_PWS_25 = 9,
        RELAY_PWS_20 = 10,
        RELAY_PWS_21 = 11,
        NUM_RELAYS = 12;

    private static final Map<String, Integer> relayMap = new HashMap<>();
    static {
        relayMap.put("htr0", RELAY_HTR_0);
        relayMap.put("htr1", RELAY_HTR_1);
        relayMap.put("pws1", RELAY_PWS_1);
        relayMap.put("pws3", RELAY_PWS_3);
        relayMap.put("pws4", RELAY_PWS_4);
        relayMap.put("pws16", RELAY_PWS_16);
        relayMap.put("pws17", RELAY_PWS_17);
        relayMap.put("pws18", RELAY_PWS_18);
        relayMap.put("pws19", RELAY_PWS_19);
        relayMap.put("pws25", RELAY_PWS_25);
        relayMap.put("pws20", RELAY_PWS_20);
        relayMap.put("pws21", RELAY_PWS_21);
    }
    private static final int
        REG_RELAYS = 0x00;
    private static final List<BoardType> validTypes = new ArrayList<>();
    static {
        validTypes.add(BoardType.BFR);
    }
    static String[] relayNames = new String[NUM_RELAYS];
    static {
        for (String name : relayMap.keySet()) {
            relayNames[relayMap.get(name)] = name;
        }
    }


    /**
     *  Constructor
     */
    public Bfr()
    {
        setValidBoardTypes(validTypes);
        setProbeAddress(REG_RELAYS);
    }


    /**
     *  Turns on a relay by number
     *
     *  @param  relay  The relay number (0 - 11)
     *  @throws  DriverException if the relay number is out of bounds
     */
    public void setRelayOn(int relay) throws  DriverException
    {
        checkRelayNumber(relay);
        writeReg(REG_RELAYS + relay, 1);
    }


    /**
     *  Turns on a relay by name
     *
     *  @param  relay  The relay name
     *  @throws  DriverException if the relay name is invalid
     */
    public void setRelayOn(String relay) throws  DriverException
    {
        writeReg(REG_RELAYS + getRelayNumber(relay), 1);
    }


    /**
     *  Turns on several relays by number
     *
     *  @param  relays  An arg list or array of relay numbers (0 - 11)
     *  @throws  DriverException if any relay number is out of bounds
     */
    public void setRelayOn(int... relays) throws  DriverException
    {
        for (int relay : relays) {
            checkRelayNumber(relay);
        }
        for (int relay : relays) {
            writeReg(REG_RELAYS + relay, 1);
        }
    }


    /**
     *  Turns on several relays by name
     *
     *  @param  relays  An arg list or array of relay names
     *  @throws  DriverException if any relay name is invalid
     */
    public void setRelayOn(String... relays) throws  DriverException
    {
        List<Integer> numbers = new ArrayList<>();
        for (String relay : relays) {
            numbers.add(getRelayNumber(relay));
        }
        for (int number : numbers) {
            writeReg(REG_RELAYS + number, 1);
        }
    }


    /**
     *  Turns on all relays
     *
     *  @throws  DriverException  if any relay number is out of bounds
     */
    public void setRelayOn() throws  DriverException
    {
        for (int relay = 0; relay < NUM_RELAYS; relay++) {
            writeReg(REG_RELAYS + relay, 1);
        }
    }


    /**
     *  Turns off a relay by number
     *
     *  @param  relay  The relay number (0 - 11)
     *  @throws  DriverException  if the relay number is out of bounds
     */
    public void setRelayOff(int relay) throws  DriverException
    {
        checkRelayNumber(relay);
        writeReg(REG_RELAYS + relay, 0);
    }


    /**
     *  Turns off a relay by name
     *
     *  @param  relay  The relay name
     *  @throws  DriverException if the relay name is invalid
     */
    public void setRelayOff(String relay) throws  DriverException
    {
        writeReg(REG_RELAYS + getRelayNumber(relay), 0);
    }


    /**
     *  Turns off several relays by number
     *
     *  @param  relays  An arg list or array of relay numbers (0 - 11)
     *  @throws  DriverException  if any relay number is out of bounds
     */
    public void setRelayOff(int... relays) throws  DriverException
    {
        for (int relay : relays) {
            checkRelayNumber(relay);
        }
        for (int relay : relays) {
            writeReg(REG_RELAYS + relay, 0);
        }
    }


    /**
     *  Turns off several relays by name
     *
     *  @param  relays  An arg list or array of relay names
     *  @throws  DriverException if any relay name is invalid
     */
    public void setRelayOff(String... relays) throws  DriverException
    {
        List<Integer> numbers = new ArrayList<>();
        for (String relay : relays) {
            numbers.add(getRelayNumber(relay));
        }
        for (int number : numbers) {
            writeReg(REG_RELAYS + number, 0);
        }
    }


    /**
     *  Turns off all relays
     *
     *  @throws  DriverException
     */
    public void setRelayOff() throws  DriverException
    {
        for (int relay = 0; relay < NUM_RELAYS; relay++) {
            writeReg(REG_RELAYS + relay, 0);
        }
    }


    /**
     *  Gets the state of a relay by number
     *
     *  @param  relay  The relay number (0 - 11)
     *  @return  Whether the relay is on
     *  @throws  DriverException if the relay number is out of bounds
     */
    public boolean isRelayOn(int relay) throws  DriverException
    {
        checkRelayNumber(relay);
        return readReg(REG_RELAYS + relay) != 0;
    }


    /**
     *  Gets the state of a relay by name
     *
     *  @param  relay  The relay name
     *  @return  Whether the relay is on
     *  @throws  DriverException if the relay name is invalid
     */
    public boolean isRelayOn(String relay) throws  DriverException
    {
        return readReg(REG_RELAYS + getRelayNumber(relay)) != 0;
    }


    /**
     *  Gets the state of all relays
     *
     *  @return  12-element boolean array of relay states
     *  @throws  DriverException
     */
    public boolean[] isRelayOn() throws  DriverException
    {
        int[] regs = readRegs(REG_RELAYS, NUM_RELAYS);
        boolean[] state = new boolean[NUM_RELAYS];
        for (int relay = 0; relay < NUM_RELAYS; relay++) {
            state[relay] = regs[relay] != 0;
        }
        return state;
    }


    /**
     *  Gets the names of the relays
     *
     *  @return  12-element string array of relay names, in number order
     *  @throws  DriverException
     */
    public static String[] getRelayNames() throws  DriverException
    {
        return relayNames;
    }


    /**
     *  Checks a relay number for validity
     *
     *  @param  relay  The relay number
     *  @throws  DriverException  if the relay number is out of bounds
     */
    private void checkRelayNumber(int relay) throws  DriverException
    {
        if (relay < 0 || relay >= NUM_RELAYS) {
            throw new DriverException("Invalid relay number (" + relay + ")");
        }
    }


    /**
     *  Gets a relay number from its name
     *
     *  @param  relay  The relay name
     *  @return  The relay number
     *  @throws  DriverException  if the relay name is invalid
     */
    private int getRelayNumber(String relay) throws  DriverException
    {
        Integer number = relayMap.get(relay);
        if (number == null) {
            throw new DriverException("Invalid relay name (" + relay + ")");
        }
        return number;
    }

}
