package org.lsst.ccs.drivers.ascii;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 *  Program to test the Ascii command device driver.
 * 
 *  @author Owen Saxton
 */
public class TestAscii {

    protected final Ascii dev;


    /**
     *  Constructor.
     */
    public TestAscii()
    {
        dev = new Ascii();
    }


    /**
     *  Constructor.
     *
     *  @param  dev  The device to use
     */
    public TestAscii(Ascii dev)
    {
        this.dev = dev;
    }


    /**
     *  Sets device options.
     *
     *  @param  option  The options enum
     */
    @Command(description="Set options")
    public void setOptions(@Argument(description="Device options") Ascii.Option option)
    {
        dev.setOptions(option);
    }


    /**
     *  Sets default baud rate
     *
     *  @param  baud  The default baud rate
     */
    @Command(description="Set default baud rate")
    public void setDefaultBaud(@Argument(description="Default baud rsate") int baud)
    {
        dev.setDefaultBaud(baud);
    }


    /**
     *  Sets default network port
     *
     *  @param  port  The default network port
     */
    @Command(description="Set default network port")
    public void setDefaultPort(@Argument(description="Default parameter") int port)
    {
        dev.setDefaultPort(port);
    }


    /**
     *  Opens connection to a device.
     *
     *  @param  type   The connection type
     *  @param  ident  The device identification
     *  @param  param  The device parameter
     *  @throws  DriverException
     */
    @Command(description="Open connection to general device")
    public void open(@Argument(description="Connection type: net or serial") Ascii.ConnType type,
                     @Argument(description="Device identifier") String ident,
                     @Argument(description="Device parameter") int param) throws DriverException
    {
        dev.open(type, ident, param);
    }


    /**
     *  Opens connection to a device with default parameters.
     *
     *  @param  type   The connection type
     *  @param  ident  The device identification
     *  @throws  DriverException
     */
    @Command(description="Open connection to general device")
    public void open(@Argument(description="Connection type: net or serial") Ascii.ConnType type,
                     @Argument(description="Device identifier") String ident) throws DriverException
    {
        dev.open(type, ident);
    }


    /**
     *  Opens connection to a network device.
     *
     *  @param  host  The host name
     *  @param  port  The port number
     *  @throws  DriverException
     */
    @Command(description="Open connection to network device")
    public void openNet(@Argument(description="Host name") String host,
                        @Argument(description="Port number") int port) throws DriverException
    {
        dev.openNet(host, port);
    }


    /**
     *  Opens connection to a network device with default port.
     *
     *  @param  host  The host name
     *  @throws  DriverException
     */
    @Command(description="Open connection to network device")
    public void openNet(@Argument(description="Host name") String host) throws DriverException
    {
        dev.openNet(host);
    }


    /**
     *  Opens connection to a serial device.
     * 
     *  @param  devc    The device name
     *  @param  baud    The baud rate
     *  @param  dbits   The number of data bits (eight or seven)
     *  @param  sbits   The number of stop bits (one or two)
     *  @param  parity  The parity (none, odd, even, mark or space)
     *  @param  flow    The flow control (none, xon, rts or dtr)
     *  @throws  DriverException
     */
    @Command(description="Open connection to serial device")
    public void openSerial(@Argument(description="Device name") String devc,
                           @Argument(description="Baud rate") int baud,
                           @Argument(description="The number of data bits") Ascii.DataBits dbits,
                           @Argument(description="The number of stop bits") Ascii.StopBits sbits,
                           @Argument(description="The parity") Ascii.Parity parity,
                           @Argument(description="The flow control") Ascii.FlowCtrl flow) throws DriverException
    {
        dev.openSerial(devc, baud, Ascii.makeDataCharacteristics(dbits, sbits, parity, flow));
    }


    /**
     *  Opens connection to a serial device.
     *
     *  @param  devc  The device name
     *  @param  baud  The baud rate
     *  @throws  DriverException
     */
    @Command(description="Open connection to serial device")
    public void openSerial(@Argument(description="Device name") String devc,
                           @Argument(description="Baud rate") int baud) throws DriverException
    {
        dev.openSerial(devc, baud);
    }


    /**
     *  Opens connection to a serial device with default baud rate.
     *
     *  @param  devc  The device name
     *  @throws  DriverException
     */
    @Command(description="Open connection to serial device")
    public void openSerial(@Argument(description="Device name") String devc) throws DriverException
    {
        dev.openSerial(devc);
    }


    /**
     *  Closes connection to a device.
     * 
     *  @throws  DriverException
     */
    @Command(description="Close connection to device")
    public void close() throws DriverException
    {
        dev.close();
    }


    /**
     *  Writes a command.
     *
     *  @param  command  The command word to send
     *  @param  args     The arguments
     *  @throws  DriverException
     */
    @Command(description="Write command to device")
    public void write(@Argument(description="Command word") String command,
                      @Argument(description="Arguments") String... args) throws DriverException
    {
        StringBuilder cmnd = new StringBuilder(command);
        for (String arg : args) {
            cmnd.append(" ").append(arg);
        }
        dev.write(makeCommandString(command, args));
    }


    /**
     *  Reads a response.
     *
     *  @return  The received response
     *  @throws  DriverException
     */
    @Command(description="Read response from device")
    public String read() throws DriverException
    {
        return dev.read();
    }


    /**
     *  Writes a command and reads the response.
     *
     *  @param  command  The command word to send
     *  @param  args     The arguments
     *  @return  The response string
     *  @throws  DriverException
     */
    @Command(description="Write command and read response")
    public String read(@Argument(description="Command word") String command,
                       @Argument(description="Arguments") String... args) throws DriverException
    {
        return dev.read(makeCommandString(command, args));
    }


    /**
     *  Sets the response timeout.
     *
     *  @param  time  The timeout (msec); 0 means no timeout
     *  @throws  DriverException
     */
    @Command(description="Set read response timeout")
    public void setTimeout(@Argument(description="Timeout value (msec)") int time) throws DriverException
    {
        dev.setTimeout(time);
    }


    /**
     *  Shows the response timeout.
     *
     *  @return  The timeout (msec)
     */
    @Command(description="Show read response timeout (msec)")
    public int showTimeout()
    {
        return dev.getTimeout();
    }


    /**
     *  Sets the network connection timeout.
     *
     *  @param  time  The timeout (msec); 0 means no timeout
     */
    @Command(description="Set network connection timeout")
    public void setConnTimeout(@Argument(description="Timeout value (msec)") int time)
    {
        dev.setConnTimeout(time);
    }


    /**
     *  Shows the network connection timeout.
     *
     *  @return  The timeout (msec)
     */
    @Command(description="Show network connection timeout (msec)")
    public int showConnTimeout()
    {
        return dev.getConnTimeout();
    }


    /**
     *  Sets the command terminator.
     *
     *  @param  term  The terminator to set
     */
    @Command(description="Set command terminator")
    public void setCommandTerm(@Argument(description="Terminator enum") Ascii.Terminator term)
    {
        dev.setCommandTerm(term);
    }


    /**
     *  Sets the response terminator.
     *
     *  @param  term  The terminator to set
     */
    @Command(description="Set response terminator")
    public void setResponseTerm(@Argument(description="Terminator enum") Ascii.Terminator term)
    {
        dev.setResponseTerm(term);
    }


    /**
     *  Sets both the command and response terminators.
     *
     *  @param  term  The terminator to set
     */
    @Command(description="Set both terminators")
    public void setTerminator(@Argument(description="Terminator enum") Ascii.Terminator term)
    {
        dev.setTerminator(term);
    }


    /**
     *  Makes a command string.
     *
     *  @param  command  The command word
     *  @param  args     The command arguments
     *  @return  The generated command string
     */
    protected static String makeCommandString(String command, String... args)
    {
        StringBuilder cmnd = new StringBuilder(command);
        for (String arg : args) {
            cmnd.append(' ').append(arg);
        }
        return cmnd.toString();
    }

}
