package org.lsst.ccs.drivers.ascii;

import java.util.concurrent.ConcurrentHashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;

/**
 *  General access routines for a multi-drop device using Ascii commands.
 *
 *  @author Owen Saxton
 */
public class Multidrop {

    /**
     *  Inner class for storing open connection data.
     */
    static class OpenDesc {

        Ascii asc;
        Set<Integer> addresses = new HashSet();

    }

    /**
     *  Public enumerations.
     */
    /** Connection type */
    public enum ConnType {

        /** FTDI USB adapter */
        FTDI,

        /** Serial port */
        SERIAL
    }

    /** Operation options */
    public enum Option {

    }

    /**
     *  Private & package constants & fields.
     */
    private static final Map<String, OpenDesc> openMap = new ConcurrentHashMap<>();
    private int defaultBaud = 0, defaultAddr = 0;
    private OpenDesc desc;
    private int timeout = 1000;
    private Ascii.Terminator cmndTerm = Ascii.Terminator.CRLF, respTerm = Ascii.Terminator.CRLF;
    protected int address;


    /**
     *  Sets the options.
     *
     *  @param  option  Various options
     */
    public void setOptions(Option option) {
    }


    /**
     *  Sets the default baud rate.
     *
     *  @param  baud  Default baud rate
     */
    public void setDefaultBaud(int baud) {
        defaultBaud = baud;
    }


    /**
     *  Sets the default multi-drop address.
     *
     *  @param  addr  Default baud rate
     */
    public void setDefaultAddress(int addr) {
        defaultAddr = addr;
    }


    /**
     *  Opens a connection to the device.
     *
     *  This is the open method that should be overridden by extending
     *  classes. since it is called by all other open methods.
     *
     *  @param  type      The enumerated type of connection to make
     *  @param  ident     The device identifier:
     *                      serial number for FTDI device;
     *                      device name for serial
     *  @param  baudRate  The baud rate
     *  @param  dataChar  The encoded data characteristics:
     *                      0 sets 8-bit, no parity, 1 stop bit, no flow control
     *  @param  addr      The multi-drop address
     *  @throws  DriverException
     */
    public synchronized void open(ConnType type, String ident, int baudRate, int dataChar, int addr)
        throws DriverException
    {
        if (isOpen()) {
            throw new DriverException("Connection already open");
        }
        String key = type.name() + ":" + ident;
        OpenDesc newDesc = openMap.get(key);
        if (newDesc == null) {
            newDesc = new OpenDesc();
            newDesc.asc = new Ascii();
            newDesc.asc.setOptions(Ascii.Option.NO_NET);
            newDesc.asc.setMultidrop();
            openMap.put(key, newDesc);
        }
        synchronized (newDesc) {
            if (newDesc.addresses.size() <= 0) {
                Ascii.ConnType cType = type == ConnType.FTDI ? Ascii.ConnType.FTDI : Ascii.ConnType.SERIAL;
                newDesc.asc.open(cType, ident, baudRate == 0 ? defaultBaud : baudRate, dataChar);
                newDesc.asc.setTimeout(timeout);
                newDesc.asc.setCommandTerm(cmndTerm);
                newDesc.asc.setResponseTerm(respTerm);
            }
            address = addr < 0 ? defaultAddr : addr;
            if (newDesc.addresses.contains(address)) {
                throw new DriverException("Device already in use");
            }
            newDesc.addresses.add(address);
        }
        desc = newDesc;
    }


    /**
     *  Opens a connection to the device with default data characteristics.
     *
     *  @param  type      The enumerated type of connection to make
     *  @param  ident     The device identifier:
     *                      serial number for FTDI device;
     *                      device name for serial
     *  @param  baudRate  The baud rate
     *  @param  addr      The multi-drop address
     *  @throws  DriverException
     */
    public void open(ConnType type, String ident, int baudRate, int addr) throws DriverException
    {
        open(type, ident, baudRate, 0, addr);
    }


    /**
     *  Opens a connection to the device with default parameters.
     *
     *  @param  type   The enumerated type of connection to make
     *  @param  ident  The device identifier:
     *                   serial number for FTDI device;
     *                   device name for serial
     *  @param  addr   The multi-drop address
     *  @throws  DriverException
     */
    public void open(ConnType type, String ident, int addr) throws DriverException
    {
        open(type, ident, 0, 0, addr);
    }


    /**
     *  Opens a connection to a serial device.
     *
     *  @param  devcName  The device name
     *  @param  baudRate  The baud rate
     *  @param  dataChar  The encoded data characteristics:
     *                      0 sets 8-bit, no parity, 1 stop bit, no flow ctrl
     *  @param  addr      The multi-drop address
     *  @throws  DriverException
     */
    public void openSerial(String devcName, int baudRate, int dataChar, int addr) throws DriverException
    {
        open(ConnType.SERIAL, devcName, baudRate, dataChar, addr);
    }


    /**
     *  Opens a connection to a serial device with default data characteristic.
     *
     *  @param  devcName  The device name
     *  @param  baudRate  The baud rate
     *  @param  addr      The multi-drop address
     *  @throws  DriverException
     */
    public void openSerial(String devcName, int baudRate, int addr) throws DriverException
    {
        openSerial(devcName, baudRate, 0, addr);
    }


    /**
     *  Opens a connection to a serial device with default parameters.
     *
     *  @param  devcName  The device name
     *  @param  addr      The multi-drop address
     *  @throws  DriverException
     */
    public void openSerial(String devcName, int addr) throws DriverException
    {
        openSerial(devcName, 0, addr);
    }


    /**
     *  Opens a connection to an FTDI device.
     *
     *  @param  serialNo  The USB serial number
     *  @param  baudRate  The baud rate
     *  @param  dataChar  The encoded data characteristics:
     *                     0 sets 8-bit, no parity, 1 stop bit, no flow ctrl
     *  @param  addr      The multi-drop address
     *  @throws  DriverException
     */
    public void openFtdi(String serialNo, int baudRate, int dataChar, int addr) throws DriverException
    {
        open(ConnType.FTDI, serialNo, baudRate, dataChar, addr);
    }


    /**
     *  Opens a connection to an FTDI device with default data characteristics.
     *
     *  @param  serialNo  The USB serial number
     *  @param  baudRate  The baud rate
     *  @param  addr      The multi-drop address
     *  @throws  DriverException
     */
    public void openFtdi(String serialNo, int baudRate, int addr) throws DriverException
    {
        openFtdi(serialNo, baudRate, 0, addr);
    }


    /**
     *  Opens a connection to an FTDI device with default parameters.
     *
     *  @param  serialNo  The USB serial number
     *  @param  addr      The multi-drop address
     *  @throws  DriverException
     */
    public void openFtdi(String serialNo, int addr) throws DriverException
    {
        openFtdi(serialNo, 0, addr);
    }


    /**
     *  Closes the device connection.
     *
     *  @throws  DriverException
     */
    public synchronized void close() throws DriverException
    {
        checkOpen();
        try {
            synchronized (desc) {
                desc.addresses.remove(address);
                if (desc.addresses.isEmpty()) {
                    desc.asc.close();
                }
            }
        }
        finally {
            desc = null;
        }
    }


    /**
     *  Closes the device connection silently.
     *
     *  @return  Whether or not the close was successful
     */
    public boolean closeSilent()
    {
        try {
            close();
            return true;
        }
        catch (DriverException e) {
            return false;
        }
    }


    /**
     *  Tests whether the device connection is open.
     *
     *  @return  Whether or not the connection is open
     */
    public boolean isOpen()
    {
        return desc != null;
    }


    /**
     *  Writes a command.
     *
     *  @param  command  The command to write, excluding terminator
     *  @throws  DriverException
     */
    public synchronized void write(String command) throws DriverException
    {
        checkOpen();
        desc.asc.write(command);
    }


    /**
     *  Reads a terminated response.
     *
     *  @return  The command response string
     *  @throws  DriverException
     *  @throws  DriverTimeoutException
     */
    public synchronized String read() throws DriverException
    {
        checkOpen();
        return desc.asc.read();
    }


    /**
     *  Reads a response after writing a command.
     *
     *  @param  command  The command to write, excluding terminator
     *  @return  The command response string
     *  @throws  DriverException
     *  @throws  DriverTimeoutException
     */
    public synchronized String read(String command) throws DriverException
    {
        checkOpen();
        return desc.asc.read(command);
    }


    /**
     *  Writes a command as bytes.
     *
     *  @param  command  The command to write, including any terminator
     *  @throws  DriverException
     */
    public void writeBytes(byte[] command) throws DriverException
    {
        checkOpen();
        desc.asc.writeBytes(command);
    }


    /**
     *  Writes a command as bytes.
     * 
     *  @param  command  The command to write, including any terminator
     *  @param  offset   The offset to the first byte to write
     *  @param  leng     The number of bytes to write
     *  @throws  DriverException
     */
    public void writeBytes(byte[] command, int offset, int leng) throws DriverException
    {
        checkOpen();
        desc.asc.writeBytes(command, offset, leng);
    }


    /**
     *  Reads available response data as bytes.
     *
     *  @param  buff    The buffer to receive the response data
     *  @param  offset  The offset to the first available byte in the buffer
     *  @return  The number of bytes read
     *  @throws  DriverException
     *  @throws  DriverTimeoutException
     */
    public int readBytes(byte[] buff, int offset) throws DriverException
    {
        checkOpen();
        return desc.asc.readBytes(buff, offset);
    }


    /**
     *  Reads available response data as bytes.
     *
     *  @param  buff    The buffer to receive the response data
     *  @param  offset  The offset to the first available byte in the buffer
     *  @param  leng    The maximum number of bytes to read
     *  @return  The number of bytes read
     *  @throws  DriverException
     *  @throws  DriverTimeoutException
     */
    public int readBytes(byte[] buff, int offset, int leng) throws DriverException
    {
        checkOpen();
        return desc.asc.readBytes(buff, offset, leng);
    }


    /**
     *  Flushes any unread data.
     *
     *  @throws  DriverException
     */
    public synchronized void flush() throws DriverException
    {
        checkOpen();
        desc.asc.flush();
    }


    /**
     *  Sets the response terminator.
     *
     *  @param  term  The expected terminator for a response (CR, LF or CRLF)
     */
    public void setResponseTerm(Ascii.Terminator term)
    {
        respTerm = term;
        if (isOpen()) {
            desc.asc.setResponseTerm(term);
        }
    }


    /**
     *  Sets the command terminator.
     *
     *  @param  term  The terminator to be appended to each command (CR, LF or CRLF)
     */
    public void setCommandTerm(Ascii.Terminator term)
    {
        cmndTerm = term;
        if (isOpen()) {
            desc.asc.setCommandTerm(term);
        }
    }


    /**
     *  Sets both the command & response terminators.
     *
     *  @param  term  The terminator to be used
     */
    public void setTerminator(Ascii.Terminator term)
    {
        setCommandTerm(term);
        setResponseTerm(term);
    }


    /**
     *  Sets the read timeout.
     *
     *  @param  time  The read timeout (sec).  0 means no timeout.
     *  @throws  DriverException
     */
    public void setTimeout(double time) throws DriverException
    {
        setTimeout((int)(1000 * time));
    }


    /**
     *  Sets the read timeout.
     *
     *  @param  time  The read timeout (msec).  0 means no timeout.
     *  @throws  DriverException
     */
    public synchronized void setTimeout(int time) throws DriverException
    {
        timeout = time;
        if (isOpen()) {
            desc.asc.setTimeout(timeout);
        }
    }


    /**
     *  Gets the synchronization object.
     * 
     *  Returns an object that can be used to synchronize multiple accesses
     *  to the underlying hardware.
     * 
     *  @return  The synchronization object
     */
    public Object getSyncObject()
    {
        return desc.asc;
    }


    /**
     *  Checks whether a connection is open.
     *
     *  @throws  DriverException
     */
    private void checkOpen() throws DriverException
    {
        if (!isOpen()) {
            throw new DriverException("Connection not open");
        }
    }

}
