package org.lsst.ccs.drivers.ascii;

import java.io.PrintStream;
import java.util.HashMap;
import java.util.Map;
import jline.console.ConsoleReader;
import org.kohsuke.args4j.Argument;
import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;
import org.kohsuke.args4j.Option;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;

/**
 *  Program to enable direct talking to a commanded device.
 *
 *  @author Owen Saxton
 */
public class TalkAscii {

    private enum Terminator {NONE, INT, CR, LF, CRLF}
    private static final Map<Terminator, Ascii.Terminator> termMap = new HashMap<>();
    static {
        termMap.put(Terminator.INT, Ascii.Terminator.CRLF);
        termMap.put(Terminator.CR, Ascii.Terminator.CR);
        termMap.put(Terminator.LF, Ascii.Terminator.LF);
        termMap.put(Terminator.CRLF, Ascii.Terminator.CRLF);
    }
    private final static PrintStream out = System.out;
    private final Thread readW = new Thread(new Reader());
    private final Ascii dev = new Ascii();
    private Ascii.Terminator respTerm;
    private int nRead = 0, nByte = 0, nTimeout = 0;
    private boolean open;


    /**
     *  Inner class for holding command line options.
     */
    private static class Options {

        @Option(name="-c", metaVar="<connType>",
                usage="Connection type: net (default), ftdi or serial")
        private Ascii.ConnType type = Ascii.ConnType.NET;

        @Option(name="-p", metaVar="<parity>",
                usage="Parity: none (default), odd, even, mark or space")
        private Ascii.Parity parity = Ascii.Parity.NONE;

        @Option(name="-d", metaVar="<dbits>",
                usage="No. data bits: eight (default) or seven")
        private Ascii.DataBits dataBits = Ascii.DataBits.EIGHT;

        @Option(name="-s", metaVar="<sbits>",
                usage="No. stop bits: one (default) or two")
        private Ascii.StopBits stopBits = Ascii.StopBits.ONE;

        @Option(name="-f", metaVar="<flow>",
                usage="Flow control: none (default), rts, dtr or xon")
        private Ascii.FlowCtrl flowCtrl = Ascii.FlowCtrl.NONE;

        @Option(name="-r", metaVar="<timeout>",
                usage="Read timeout (sec) (default = 0.1)")
        private double timeout = 0.1;

        @Option(name="-t", metaVar="<term>",
                usage="Cmnd/resp termination: int (default), cr, lf, crlf or none")
        private Terminator term = Terminator.INT;

        @Option(name="-n", usage="Displays read counters when command is empty")
        private boolean showCounts;

        @Option(name="-help", usage="Displays this help")
        private boolean showHelp;

        @Argument(index=0, required=true, metaVar="<ident>",
                  usage="Device identifier (node, serial no. or device name)")
        private String ident;

        @Argument(index=1, required=true, metaVar="<parm>",
                  usage="Device parameter (port number or baud rate)")
        private int parm;

    }


    /**
     *  Inner class to implement device reading thread.
     */
    private class Reader implements Runnable {

        @Override
        public void run()
        {
            while (open) {
                byte[] buff = new byte[1024];
                try {
                    if (respTerm != null) {
                        String reply = dev.read();
                        out.println(reply);
                        nByte += reply.length();
                    }
                    else {
                        int leng = dev.readBytes(buff, 0);
                        out.print(new String(buff, 0, leng));
                        nByte += leng;
                    }
                    nRead++;
                }
                catch (DriverTimeoutException e) {
                    nTimeout++;
                }
                catch (DriverException e) {
                    if (open) {
                        out.println(e);
                        System.exit(0);
                    }
                }
            }
        }

    }


    /**
     *  Main program.
     *
     *  @param  args  Command-line arguments
     */
    public static void main(String[] args)
    {
        Options optns = new Options();
        CmdLineParser parser = new CmdLineParser(optns);
        try {
            parser.parseArgument(args);
            if (optns.showHelp) {
                showHelp(parser);
            }
            else {
                (new TalkAscii()).run(optns);
            }
        }
        catch (CmdLineException e) {
            if (!optns.showHelp) {
                out.println(e.getMessage());
            }
            showHelp(parser);
        }
        catch (Exception e) {
            out.println(e);
        }
        System.exit(0);
    }


    /**
     *  Sends commands to the device.
     *
     *  Loops reading and processing each new typed command line.
     */
    private void run(Options optns) throws Exception
    {
        respTerm = termMap.get(optns.term);
        if (respTerm != null) {
            dev.setResponseTerm(respTerm);
        }
        String termStr = respTerm != null ? optns.term == Terminator.INT ? "\r" : respTerm.getValue() : "";
        dev.setTimeout(optns.timeout);
        dev.open(optns.type, optns.ident, optns.parm,
                 Ascii.makeDataCharacteristics(optns.dataBits, optns.stopBits, optns.parity, optns.flowCtrl));
        open = true;
        readW.setDaemon(true);
        readW.start();
        ConsoleReader readC = new ConsoleReader();
        out.println("Connected. Type CTRL-D to exit" + (optns.showCounts ? ", CR for read stats." : "."));

        while (true) {
            String line = readC.readLine("");
            if (line == null) break;
            if (line.isEmpty() && optns.showCounts) {
                out.format("#reads: %s; #bytes: %s; #timeouts: %s %n", nRead, nByte, nTimeout);
            }
            else {
                dev.writeBytes((line + termStr).getBytes());
            }
        }

        open = false;
        dev.close();
    }


    /**
     *  Displays help.
     */
    private static void showHelp(CmdLineParser parser)
    {
        out.println("Usage: TalkAscii [options...] <ident> <parm>");
        out.println("Parameters & options:");
        parser.printUsage(out);
    }

}
