package org.lsst.ccs.drivers.ascii;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.drivers.ftdi.Ftdi;

/**
 *  Communications I/O routines for serial devices connected via FTDI.
 *
 *  @author Owen Saxton
 */

public class AsciiIOFtdi implements AsciiIO {

    /**
     *  Private fields.
     */
    private final static Map<Integer, Integer> dbitsMap = new HashMap<>();
    static {
        dbitsMap.put(Ascii.DBITS_SEVEN, Ftdi.DATABITS_7);
        dbitsMap.put(Ascii.DBITS_EIGHT, Ftdi.DATABITS_8);
    }

    private final static Map<Integer, Integer> sbitsMap = new HashMap<>();
    static {
        sbitsMap.put(Ascii.SBITS_ONE, Ftdi.STOPBITS_1);
        sbitsMap.put(Ascii.SBITS_TWO, Ftdi.STOPBITS_2);
    }

    private final static Map<Integer, Integer> parityMap = new HashMap<>();
    static {
        parityMap.put(Ascii.PARITY_NONE,  Ftdi.PARITY_NONE);
        parityMap.put(Ascii.PARITY_ODD,   Ftdi.PARITY_ODD);
        parityMap.put(Ascii.PARITY_EVEN,  Ftdi.PARITY_EVEN);
        parityMap.put(Ascii.PARITY_MARK,  Ftdi.PARITY_MARK);
        parityMap.put(Ascii.PARITY_SPACE, Ftdi.PARITY_SPACE);
    }

    Ftdi ftdi = new Ftdi();
    private int timeout = 0;


    /**
     *  Opens a connection.
     *
     *  @param  serial  The serial number string of the interface, optionally
     *                  prepended with the node name of a remote FTDI server,
     *                  and separated from it by a ":".
     *  @param  baud    The baud rate
     *  @param  dchars  The data characteristics
     *  @throws  DriverException
     */
    @Override
    public void open(String serial, int baud, int dchars) throws DriverException
    {
        Integer dataBits = dbitsMap.get((dchars >> 24) & 0xff);
        if (dataBits == null) {
            throw new DriverException("Invalid number of data bits");
        }
        Integer stopBits = sbitsMap.get((dchars >> 16) & 0xff);
        if (stopBits == null) {
            throw new DriverException("Invalid number of stop bits");
        }
        Integer parity = parityMap.get((dchars >> 8) & 0xff);
        if (parity == null) {
            throw new DriverException("Invalid parity");
        }
        int fc = dchars & 0xff;
        int flowCtrl =
          ((fc & Ascii.FLOW_RTS_CTS) != 0 ? Ftdi.FLOW_CONTROL_RTS_CTS : 0)
           | ((fc & Ascii.FLOW_DTR_DSR) != 0 ? Ftdi.FLOW_CONTROL_DTR_DSR : 0)
           | ((fc & Ascii.FLOW_XON_XOFF) != 0 ? Ftdi.FLOW_CONTROL_XON_XOFF : 0);
        String[] fields = serial.split(":", 2);
        if (fields.length == 1) {
            ftdi.open(0, serial);
        }
        else {
            ftdi.open(fields[0], 0, fields[1]);
        }
        ftdi.setBaudrate(baud);
        ftdi.setDataCharacteristics(dataBits, stopBits, parity);
        ftdi.setFlowControl(flowCtrl);
        ftdi.enableEvents(Ftdi.EVENT_RXCHAR);
    }


    /**
     *  Closes the connection.
     *
     *  @throws  DriverException
     */
    @Override
    public void close() throws DriverException
    {
        ftdi.close();
    }


    /**
     *  Writes command data.
     *
     *  @param  buff    The buffer containing the command to write
     *  @param  offset  The offset to the command
     *  @param  leng    The length of the command
     *  @throws  DriverException
     */
    @Override
    public void write(byte[] buff, int offset, int leng)
        throws DriverException
    {
        ftdi.write(buff, offset, leng);
    }


    /**
     *  Reads available response data.
     *
     *  @param  buff    The buffer to receive the response data
     *  @param  offset  The offset to the first available byte in the buffer
     *  @param  mleng   The maximum number of bytes to read
     *  @return  The number of bytes read
     *  @throws  DriverException
     *  @throws  DriverTimeoutException
     */
    @Override
    public int read(byte[] buff, int offset, int mleng) throws DriverException
    {
        if (offset + mleng > buff.length) {
            mleng = buff.length - offset;
        }
        int leng = ftdi.getQueueStatus();
        if (leng == 0) {
            if (ftdi.awaitEvent(timeout) == 0) {
                throw new DriverTimeoutException("Read timed out");
            }
            leng = ftdi.getQueueStatus();
        }
        if (leng > mleng) {
            leng = mleng;
        }
        if (leng > 0) {
            leng = ftdi.read(buff, offset, leng);
        }

        return leng;
    }


    /**
     *  Flushes any available response data.
     *
     *  @throws  DriverException
     */
    @Override
    public void flush() throws DriverException
    {
        int count = ftdi.getQueueStatus();
        if (count > 0) {
            byte[] buff = new byte[count];
            ftdi.read(buff, 0, count);
        }
    }

    
    /**
     *  Sets the receive timeout.
     *
     *  @param  timeout  The receive timeout (ms).  0 means no timeout.
     *  @throws  DriverException
     */
    @Override
    public void setTimeout(int timeout) throws DriverException
    {
        this.timeout = timeout;
    }

}
