package org.lsst.ccs.drivers.ascii;

import java.net.SocketException;
import org.lsst.ccs.drivers.commons.DriverConstants;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;

/**
 *  General access routines for a device using Ascii commands.
 *
 *  @author Owen Saxton
 */
public class Ascii implements DriverConstants {

    /**
     *  Public enumerations.
     */
    /** Constructor options */
    public enum Option {

        /** Network connection not allowed */
        NO_NET,

        /** Serial connection not allowed */
        NO_SERIAL,

        /** Keep network connection alive */
        KEEP_ALIVE,

        /** Keep network connection alive and serial connection not allowed */
        KEEP_ALIVE_NO_SERIAL;
    }

    /** Data characteristics - parity */
    public enum Parity {

        /** No parity */
        NONE(PARITY_NONE),

        /** Odd parity */
        ODD(PARITY_ODD),

        /** Even parity */
        EVEN(PARITY_EVEN),

        /** Mark parity */
        MARK(PARITY_MARK),

        /** Space parity */
        SPACE(PARITY_SPACE);

        int value;

        Parity(int value) {
            this.value = value;
        }

        private int getValue() {
            return value;
        }
    }

    /** Data characteristics - number of data bits */
    public enum DataBits {

        /** Seven data bits */
        SEVEN(DBITS_SEVEN),

        /** Eight data bits */
        EIGHT(DBITS_EIGHT);

        int value;

        DataBits(int value) {
            this.value = value;
        }

        private int getValue() {
            return value;
        }
    }

    /** Data characteristics - number of stop bits */
    public enum StopBits {

        /** One stop bit */
        ONE(SBITS_ONE),

        /** Two stop bits */
        TWO(SBITS_TWO);

        int value;

        StopBits(int value) {
            this.value = value;
        }

        private int getValue() {
            return value;
        }
    }

    /** Data characteristics - flow control */
    public enum FlowCtrl {

        /** No flow control */
        NONE(FLOW_NONE),

        /** RTS/CTS flow control */
        RTS(FLOW_RTS_CTS),

        /** DTR/DSR flow control (ignored for serial port) */
        DTR(FLOW_DTR_DSR),

        /** XON/XOFF flow control */
        XON(FLOW_XON_XOFF),

        /** RTS/CTS and DTR/DSR flow control */
        RTS_DTR(FLOW_RTS_CTS | FLOW_DTR_DSR),

        /** RTS/CTS and XON/XOFF flow control */
        RTS_XON(FLOW_RTS_CTS | FLOW_XON_XOFF),

        /** DTR/DSR and XON/XOFF flow control */
        DTR_XON(FLOW_DTR_DSR | FLOW_XON_XOFF),

        /** RTS/CTS, DTR/DSR and XON/XOFF flow control */
        RTS_DTR_XON(FLOW_RTS_CTS | FLOW_DTR_DSR | FLOW_XON_XOFF);

        int value;

        FlowCtrl(int value) {
            this.value = value;
        }

        private int getValue() {
            return value;
        }
    }

    /** Command or response terminators */
    public enum Terminator {

        /** Carriage return */
        CR("\r"),

        /** Line feed */
        LF("\n"),

        /** Carriage return/Line feed */
        CRLF("\r\n");

        String value;

        Terminator(String value) {
            this.value = value;
        }

        public String getValue() {
            return value;
        }
    }

    /**
     *  Private & package constants & fields.
     */
    final static int
        PARITY_NONE   = 0,
        PARITY_ODD    = 1,
        PARITY_EVEN   = 2,
        PARITY_MARK   = 3,
        PARITY_SPACE  = 4,
        DBITS_EIGHT   = 0,
        DBITS_SEVEN   = 1,
        SBITS_ONE     = 0,
        SBITS_TWO     = 1,
        FLOW_NONE     = 0,
        FLOW_RTS_CTS  = 0x01,
        FLOW_DTR_DSR  = 0x02,
        FLOW_XON_XOFF = 0x04;

    private boolean noNet = false, noSerial = false, keepAlive = false;
    private int defaultParm1 = 0;

    private final byte[] buff = new byte[4096];
    private ConnType connType;
    private String connIdent;
    private int connParm1, connParm2;
    private AsciiIO io;
    private String cmndTerm = "\r\n";
    private byte[] respTerm = {};
    private int timeout = 1000;
    private int buffIn, buffOut;


    /**
     *  Constructor.
     */
    public Ascii() {
    }


    /**
     *  Constructor (deprecated).
     *
     *  @param  option  Various options
     */
    @Deprecated
    public Ascii(Option option) {
        setOpts(option);
    }


    /**
     *  Sets the options.
     *
     *  @param  option  Various options
     */
    public void setOptions(Option option) {
        setOpts(option);
    }


    /**
     *  Set the options (private).
     *
     *  @param  option  Various options
     */
    private void setOpts(Option option) {
        noNet = (option == Option.NO_NET);
        noSerial = (option == Option.NO_SERIAL || option == Option.KEEP_ALIVE_NO_SERIAL);
        keepAlive = (option == Option.KEEP_ALIVE || option == Option.KEEP_ALIVE_NO_SERIAL);
    }


    /**
     *  Set the default parameter.
     *
     *  @param  parm1  Default parm1 value
     */
    public void setDefaultParm(int parm1) {
        defaultParm1 = parm1;
    }


    /**
     *  Generates the serial data characteristics parameter.
     *
     *  @param  dataBits  The enumerated number of data bits
     *  @param  stopBits  The enumerated number of stop bits
     *  @param  parity    The enumerated parity
     *  @param  flowCtrl  The enumerated flow control
     *  @return  The encoded data characteristics
     */
    public static int makeDataCharacteristics(DataBits dataBits, StopBits stopBits,
                                              Parity parity, FlowCtrl flowCtrl)
    {
        return (dataBits.getValue() << 24) | (stopBits.getValue() << 16)
                 | (parity.getValue() << 8) | flowCtrl.getValue();
    }


    /**
     *  Opens a connection to the device.
     *
     *  This is the open method that should be overridden by extending
     *  classes. since it is called by all other open methods.
     *
     *  @param  type   The enumerated type of connection to make
     *  @param  ident  The device identifier:
     *                   host name or IP address for network;
     *                   serial number for FTDI device;
     *                   device name for serial
     *  @param  parm1  The first device parameter, or 0 for default:
     *                   port number for network;
     *                   baud rate for FTDI or serial
     *  @param  parm2  The second device parameter:
     *                   unused for network;
     *                   encoded data characteristics for FTDI or serial:
     *                     0 sets 8-bit, no parity, 1 stop bit, no flow control
     *  @throws  DriverException
     */
    public synchronized void open(ConnType type, String ident, int parm1, int parm2) throws DriverException
    {
        if (isOpen()) {
            throw new DriverException("Device already connected");
        }

        AsciiIO newIo;

        switch (type) {

        case NET:
            if (noNet) {
                throw new DriverException("Network connection not allowed");
            }
            newIo = new AsciiIONet();
            break;

        case FTDI:
            if (noSerial) {
                throw new DriverException("FTDI connection not allowed");
            }
            newIo = new AsciiIOFtdi();
            break;

        case SERIAL:
            if (noSerial) {
                throw new DriverException("Serial connection not allowed");
            }
            newIo = new AsciiIOSerial();
            break;

        default:
            throw new DriverException("Connection type not handled: " + type);
        }

        int p1 = parm1 == 0 ? defaultParm1 : parm1;
        newIo.open(ident, p1, parm2);
        io = newIo;
        io.setTimeout(timeout);
        connType = type;
        connIdent = ident;
        connParm1 = p1;
        connParm2 = parm2;
    }


    /**
     *  Opens a connection to the device with default data characteristics.
     *
     *  @param  type   The enumerated type of connection to make
     *  @param  ident  The device identifier:
     *                   host name or IP address for network;
     *                   serial number for FTDI device;
     *                   device name for serial
     *  @param  parm   The device parameter:
     *                   port number for network;
     *                   baud rate for FTDI or serial
     *  @throws  DriverException
     */
    public void open(ConnType type, String ident, int parm) throws DriverException
    {
        open(type, ident, parm, 0);
    }


    /**
     *  Opens a connection to the device with default parameters.
     *
     *  @param  type   The enumerated type of connection to make
     *  @param  ident  The device identifier:
     *                   host name or IP address for network;
     *                   serial number for FTDI device;
     *                   device name for serial
     *  @throws  DriverException
     */
    public void open(ConnType type, String ident) throws DriverException
    {
        open(type, ident, 0);
    }


    /**
     *  Opens a connection to a network device.
     *
     *  @param  host  The host name or IP address
     *  @param  port  The port number
     *  @throws  DriverException
     */
    public void openNet(String host, int port) throws DriverException
    {
        open(ConnType.NET, host, port);
    }


    /**
     *  Opens a connection to a network device with default port.
     *
     *  @param  host  The host name or IP address
     *  @throws  DriverException
     */
    public void openNet(String host) throws DriverException
    {
        openNet(host, 0);
    }


    /**
     *  Opens a connection to a serial device.
     *
     *  @param  devcName  The device name
     *  @param  baudRate  The baud rate
     *  @param  dataChar  The encoded data characteristics:
     *                     0 sets 8-bit, no parity, 1 stop bit, no flow ctrl
     *  @throws  DriverException
     */
    public void openSerial(String devcName, int baudRate, int dataChar) throws DriverException
    {
        open(ConnType.SERIAL, devcName, baudRate, dataChar);
    }


    /**
     *  Opens a connection to a serial device with default data characteristic.
     *
     *  @param  devcName  The device name
     *  @param  baudRate  The baud rate
     *  @throws  DriverException
     */
    public void openSerial(String devcName, int baudRate) throws DriverException
    {
        openSerial(devcName, baudRate, 0);
    }


    /**
     *  Opens a connection to a serial device with default parameters.
     *
     *  @param  devcName  The device name
     *  @throws  DriverException
     */
    public void openSerial(String devcName) throws DriverException
    {
        openSerial(devcName, 0);
    }


    /**
     *  Opens a connection to an FTDI device.
     *
     *  @param  serialNo  The USB serial number
     *  @param  baudRate  The baud rate
     *  @param  dataChar  The encoded data characteristics:
     *                     0 sets 8-bit, no parity, 1 stop bit, no flow ctrl
     *  @throws  DriverException
     */
    public void openFtdi(String serialNo, int baudRate, int dataChar) throws DriverException
    {
        open(ConnType.FTDI, serialNo, baudRate, dataChar);
    }


    /**
     *  Opens a connection to an FTDI device with default data characteristics.
     *
     *  @param  serialNo  The USB serial number
     *  @param  baudRate  The baud rate
     *  @throws  DriverException
     */
    public void openFtdi(String serialNo, int baudRate) throws DriverException
    {
        openFtdi(serialNo, baudRate, 0);
    }


    /**
     *  Opens a connection to an FTDI device with default parameters.
     *
     *  @param  serialNo  The USB serial number
     *  @throws  DriverException
     */
    public void openFtdi(String serialNo) throws DriverException
    {
        openFtdi(serialNo, 0);
    }


    /**
     *  Closes the device connection.
     *
     *  @throws  DriverException
     */
    public synchronized void close() throws DriverException
    {
        checkOpen();
        try {
            io.close();
        }
        finally {
            io = null;
            buffIn = buffOut = 0;
        }
    }


    /**
     *  Closes the device connection silently.
     *
     *  @return  Whether or not the close caused an error
     */
    public boolean closeSilent()
    {
        try {
            close();
            return true;
        }
        catch (DriverException e) {
            return false;
        }
    }


    /**
     *  Tests whether the device connection is open.
     *
     *  @return  Whether or not the connection is open
     */
    public boolean isOpen()
    {
        return io != null;
    }


    /**
     *  Writes a command.
     *
     *  @param  command  The command to write, excluding terminator
     *  @throws  DriverException
     */
    public synchronized void write(String command) throws DriverException
    {
        checkOpen();
        writeBytes((command + cmndTerm).getBytes());
    }


    /**
     *  Reads a terminated response.
     *
     *  The current version accepts only a single-byte or a two-byte terminator.
     *  The previous version accepted either a CR or LF as a terminator.  But
     *  this must remain supported for some time since a given program may 
     *  contain some code that depends on the old behavior and some that depends
     *  on the new.  The default behavior will be the old, indicated by a zero-
     *  byte terminator.  The new behavior will come into effect when the new
     *  setReceiveTerm method is called.
     *
     *  @return  The command response string
     *  @throws  DriverException
     *  @throws  DriverTimeoutException
     */
    public synchronized String read() throws DriverException
    {
        checkOpen();
        StringBuilder sBuff = null;
        int curr = buffOut, term = -1;
        while (true) {
            while (curr < buffIn) {
                byte buffByte = buff[curr++];
                if (respTerm.length == 0) {
                    if (buffByte != 0x0a && buffByte != 0x0d) continue;
                    if (curr == buffOut + 1) {
                        buffOut = curr;
                        continue;
                    }
                    term = curr - 1;
                }
                else {
                    if (term < 0) {
                        if (buffByte != respTerm[0]) continue;
                        term = curr - 1;
                        if (respTerm.length != 1) continue;
                    }
                    else {
                        if (buffByte != respTerm[1]) {
                            term = -1;
                            continue;
                        }
                    }
                }
                String resp = new String(buff, buffOut, term - buffOut);
                buffOut = term + respTerm.length;
                if (sBuff == null) {
                    return resp;
                }
                else {
                    return sBuff.append(resp).toString();
                }
            }
            if (buffOut > 0) {
                System.arraycopy(buff, buffOut, buff, 0, buffIn - buffOut);
                buffIn -= buffOut;
                term -= buffOut;
                buffOut = 0;
            }
            if (buffIn >= buff.length) {
                if (sBuff == null) {
                    sBuff = new StringBuilder(new String(buff, 0, buffIn - 1));
                }
                else {
                    sBuff.append(new String(buff, 0, buffIn - 1));
                }
                buff[0] = buff[buffIn - 1];
                term -= buffIn - 1;
                buffIn = 1;
            }
            curr = buffIn;
            try {
                buffIn += readBytes(buff, buffIn);
            }
            catch (DriverTimeoutException re) {
                throw re;
            }
            catch (DriverException re) {
                closeSilent();
                throw re;
            }
        }
    }


    /**
     *  Reads a response after writing a command.
     *
     *  @param  command  The command to write, excluding terminator
     *  @return  The command response string
     *  @throws  DriverException
     *  @throws  DriverTimeoutException
     */
    public synchronized String read(String command) throws DriverException
    {
        flush();
        write(command);
        return read();
    }


    /**
     *  Writes a command as bytes.
     *
     *  @param  command  The command to write, including any terminator
     *  @throws  DriverException
     */
    public void writeBytes(byte[] command) throws DriverException
    {
        checkOpen();
        writeBytes(command, 0, command.length);
    }


    /**
     *  Writes a command as bytes.
     *
     *  @param  command  The command to write, including any terminator
     *  @param  offset   The offset to the first byte to write
     *  @param  leng     The number of bytes to write
     *  @throws  DriverException
     */
    public void writeBytes(byte[] command, int offset, int leng) throws DriverException
    {
        checkOpen();
        try {
            io.write(command, offset, leng);
        }
        catch (DriverException e) {
            tryReconnect(e);
            io.write(command, offset, leng);
        }
    }


    /**
     *  Reads available response data as bytes.
     *
     *  @param  buff    The buffer to receive the response data
     *  @param  offset  The offset to the first available byte in the buffer
     *  @return  The number of bytes read
     *  @throws  DriverException
     *  @throws  DriverTimeoutException
     */
    public int readBytes(byte[] buff, int offset) throws DriverException
    {
        checkOpen();
        return readBytes(buff, offset, buff.length - offset);
    }


    /**
     *  Reads available response data as bytes.
     *
     *  @param  buff    The buffer to receive the response data
     *  @param  offset  The offset to the first available byte in the buffer
     *  @param  leng    The maximum number of bytes to read
     *  @return  The number of bytes read
     *  @throws  DriverException
     *  @throws  DriverTimeoutException
     */
    public int readBytes(byte[] buff, int offset, int leng) throws DriverException
    {
        checkOpen();
        try {
            return io.read(buff, offset, leng);
        }
        catch (DriverException e) {
            tryReconnect(e);
            return io.read(buff, offset, leng);
        }
    }


    /**
     *  Flushes any unread data.
     *
     *  @throws  DriverException
     */
    public synchronized void flush() throws DriverException
    {
        checkOpen();
        buffIn = buffOut;
        io.flush();
    }


    /**
     *  Sets the response terminator.
     *
     *  @param  term  The expected terminator for a response (CR, LF or CRLF)
     */
    public void setResponseTerm(Terminator term)
    {
        respTerm = term.getValue().getBytes();
    }


    /**
     *  Sets the command terminator.
     *
     *  @param  term  The terminator to be appended to each command (CR, LF or CRLF)
     */
    public void setCommandTerm(Terminator term)
    {
        cmndTerm = term.getValue();
    }


    /**
     *  Sets both the command & response terminators.
     *
     *  @param  term  The terminator to be used
     */
    public void setTerminator(Terminator term)
    {
        setCommandTerm(term);
        setResponseTerm(term);
    }


    /**
     *  Sets the read timeout.
     *
     *  @param  time  The read timeout (sec).  0 means no timeout.
     *  @throws  DriverException
     */
    public void setTimeout(double time) throws DriverException
    {
        setTimeout((int)(1000 * time));
    }


    /**
     *  Sets the read timeout.
     *
     *  @param  time  The read timeout (msec).  0 means no timeout.
     *  @throws  DriverException
     */
    public synchronized void setTimeout(int time) throws DriverException
    {
        timeout = time;
        if (isOpen()) {
            io.setTimeout(timeout);
        }
    }


    /**
     *  Checks that the connection is open.
     *
     *  @throws  DriverException
     */
    private void checkOpen() throws DriverException
    {
        if (!isOpen()) {
            throw new DriverException("Device not connected");
        }
    }


    /**
     *  Tries to reconnect if network socket exception.
     *
     *  @param  e  The exception to check
     *  @throws  DriverException
     */
    private void tryReconnect(DriverException e) throws DriverException
    {
        if (keepAlive && e.getCause() instanceof SocketException) {
            closeSilent();
            try {
                open(connType, connIdent, connParm1, connParm2);
            }
            catch (DriverException eo) {
                throw e;
            }
        }
        else {
            throw e;
        }
    }

}
