package org.lsst.ccs.drivers.ascii;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.util.ArrayList;
import java.util.List;
import org.apache.commons.net.telnet.TelnetClient;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;

/**
 *  Driver for managing an interactive Telnet or Ascii session
 *
 *  @author Owen Saxton
 */
public class Session {

    /**
     *  Public constants.
     */
    public enum ConnType {

        TELNET(null),
        FTDI(Ascii.ConnType.FTDI),
        SERIAL(Ascii.ConnType.SERIAL);

        private final Ascii.ConnType connType;

        ConnType(Ascii.ConnType connType) {
            this.connType = connType;
        }

        Ascii.ConnType getValue() {
            return connType;
        }

    }

    public static final int OPTN_KEEP_ALIVE = 0x01;

    /**
     *  Private data.
     */
    private static final int
        TELNET_PORT = 23,
        SOCKET_TIMEOUT = 20,
        POLL_PERIOD = 10,
        INITIAL_TIMEOUT = 2000,
        FINAL_TIMEOUT = 200,
        LOG_TIMEOUT_INCR = 100;

    private final int options;
    private final String sessnPrompt, unamePrompt, passwdPrompt;
    private final Ascii.Terminator terminator;
    private final Ascii asc = new Ascii(Ascii.Option.NO_NET);
    private final TelnetClient tc = new TelnetClient();
    private ConnType connType;
    private String ident, username, password;
    private int baudRate;
    private boolean isTelnet;
    private int initTimeout = INITIAL_TIMEOUT, finalTimeout = FINAL_TIMEOUT;
    private int logTimeout, runTimeout;
    private InputStream in;
    private OutputStream out;
    private String prompt = "";
    private boolean usePrompt = false;


    /**
     *  Constructor
     *
     *  @param  options       Options word
     *  @param  sessnPrompt   The expected session prompt
     *  @param  unamePrompt   The expected username prompt
     *  @param  passwdPrompt  The expected password prompt
     *  @param  terminator    The Ascii command/response terminator
     */
    public Session(int options, String sessnPrompt, String unamePrompt, String passwdPrompt, Ascii.Terminator terminator)
    {
        this.options = options;
        this.sessnPrompt = sessnPrompt;
        this.unamePrompt = unamePrompt;
        this.passwdPrompt = passwdPrompt;
        this.terminator = terminator;
    }


    /**
     *  Opens a connection and logs in.
     *
     *  @param  connType    The enumerated connection type: TELNET, FTDI or SERIAL
     *  @param  ident       The host name (TELNET), USB ID (FTDI) or port name (SERIAL)
     *  @param  baudRate    The baud rate for FTDI and Serial
     *  @param  username    The user name
     *  @param  password    The password
     *  @param  logTimeout  The character timeout when logging in (msec)
     *  @param  runTimeout  The character timeout when running (msec) 
     *  @throws  DriverException
     */
    public void open(ConnType connType, String ident, int baudRate, String username, String password,
                     int logTimeout, int runTimeout) throws DriverException
    {
        open(connType, ident, baudRate, runTimeout);
        this.username = username;
        this.password = password;
        this.logTimeout = logTimeout;
        usePrompt = false;
        try {
            finalTimeout = runTimeout;
            if (isTelnet) {
                receive();
            }
            else {
                receive("");
                if (prompt.isEmpty()) {  // Try again
                    receive("");
                }
                if (prompt.contains(sessnPrompt)) return;  // Already logged in
            }
            if (!prompt.contains(unamePrompt)) {
                throw new DriverException("Unrecognized login prompt: " + prompt);
            }

            receive(username);
            if (!prompt.contains(passwdPrompt)) {
                throw new DriverException("Unrecognized password prompt: " + prompt);
            }

            finalTimeout = Math.min(logTimeout, LOG_TIMEOUT_INCR);
            int ntry = logTimeout / finalTimeout + 1;
            send(password);
            for (int j = 0; j < ntry; j++) {
                if (receive().length != 0) break;
                if (j == ntry - 1) {
                    throw new DriverException("Login failed");
                }
            }
            finalTimeout = runTimeout;
            usePrompt = true;
        }
        catch (DriverException e) {
            close();
            throw e;
        }
    }


    /**
     *  Opens a connection.
     *
     *  @param  connType    The enumerated connection type: FTDI or SERIAL
     *  @param  ident       The USB ID (FTDI) or port name (SERIAL)
     *  @param  baudRate    The baud rate for FTDI and Serial
     *  @param  runTimeout  The character timeout when running (msec) 
     *  @throws  DriverException
     */
    public void open(ConnType connType, String ident, int baudRate, int runTimeout) throws DriverException
    {
        this.connType = connType;
        this.ident = ident;
        this.baudRate = baudRate;
        this.runTimeout = runTimeout;
        this.username = null;
        this.password = null;
        Ascii.ConnType cType = connType.getValue();
        if (cType != null) {
            isTelnet = false;
            asc.open(cType, ident, baudRate);
            asc.setTerminator(terminator);
            asc.setTimeout(initTimeout);
        }
        else {
            isTelnet = true;
            if (in != null) {
                throw new DriverException("Connection already open");
            }
            try {
                tc.connect(ident, TELNET_PORT);
                tc.setSoTimeout(SOCKET_TIMEOUT);
                in = tc.getInputStream();
                out = tc.getOutputStream();
            }
            catch (IOException e) {
                throw new DriverException(e);
            }
        }
        finalTimeout = runTimeout;
        usePrompt = true;
    }


    /**
     *  Closes the connection
     *
     *  @throws  DriverException
     */
    public void close() throws DriverException
    {
        checkOpen();
        if (!isTelnet) {
            try {
                send("quit");
            }
            catch (DriverException e) {
            }
        }
        try {
            if (isTelnet) {
                tc.disconnect();
            }
            else {
                asc.close();
            }
        }
        catch (IOException e) {
            throw new DriverException(e);
        }
        finally {
            prompt = "";
            in = null;
            out = null;
            isTelnet = false;
        }
    }


    /**
     *  Sends a command
     *
     *  @param  command  The command to send
     *  @throws  DriverException
     */
    public synchronized void send(String command) throws DriverException
    {
        checkOpen();
        if (isTelnet) {
            try {
                out.write((command + "\r").getBytes());
                out.flush();
            }
            catch (IOException e) {
                throw new DriverException(e);
            }
        }
        else {
            asc.write(command);
        }
    }


    /**
     *  Receives a response
     *
     *  @return  An array of strings each of which is a line of the response.
     *           The first element is the echoed command.
     *  @throws  DriverException
     */
    public synchronized String[] receive() throws DriverException
    {
        checkOpen();
        List<String> resp = new ArrayList<>();
        int timeout = initTimeout;
        byte[] buff = new byte[1024];
        String line = "";
        boolean gotData = false, gotCr = false, gotTimeout = false;
        while (true) {
            int leng = 0;
            try {
                leng = read(buff, 0, buff.length, timeout);
                gotData = true;
                //System.out.println("Read length = " + leng);
            }
            catch (DriverTimeoutException e) {
                if (!gotData) {
                    throw e;
                }
                gotTimeout = true;
            }
            int start = 0;
            for (int posn = 0; posn < leng; posn++) {
                if (buff[posn] == '\r') {
                    if (!gotCr) {
                        line += new String(buff, start, posn - start);
                        //System.out.println("Line = " + line);
                        resp.add(line);
                        line = "";
                    }
                    start = posn + 1;
                    gotCr = true;
                }
                else {
                    if (buff[posn] == '\n' && gotCr) {
                        start = posn + 1;
                    }
                    gotCr = false;
                }
            }
            line += new String(buff, start, leng - start);
            if (usePrompt) {
                if (line.startsWith(sessnPrompt)) break;
            }
            else {
                timeout = finalTimeout;
            }
            if (gotTimeout) break;
        }
        prompt = line;
        //System.out.println("Prompt = " + line);

        return resp.toArray(new String[0]);
    }


    /**
     *  Sends a command and receives the response
     *
     *  @param  command  The command to send
     *  @return  An array of strings each of which is a line of the response.
     *           The first element is the echoed command.
     *  @throws  DriverException
     */
    public synchronized String[] receive(String command) throws DriverException
    {
        try {
            send(command);
            return receive();
        }
        catch (DriverException e) {
            if ((options & OPTN_KEEP_ALIVE) != 0 && (e.getCause() instanceof SocketException)) {
                try {
                    close();
                }
                catch (DriverException ec) {
                }
                try {
                    if (username == null) {
                        open(connType, ident, baudRate, runTimeout);
                    }
                    else {
                        open(connType, ident, baudRate, username, password, logTimeout, runTimeout);
                    }
                }
                catch (DriverException eo) {
                    throw e;
                }
                send(command);
                return receive();
            }
            else {
                throw e;
            }
        }
    }


    /**
     *  Sets the response timeout.
     *
     *  @param  timeout  The timeout (ms)
     *  @throws DriverException
     */
    public void setTimeout(int timeout) throws DriverException
    {
        asc.setTimeout(initTimeout = timeout);
    }


    /**
     *  Sets the response inter-character timeout.
     *
     *  @param  timeout  The timeout (ms)
     *  @throws DriverException
     */
    public void setCharTimeout(int timeout) throws DriverException
    {
        asc.setTimeout(finalTimeout = timeout);
    }


    /**
     *  Gets the prompt
     *
     *  @return  The saved prompt
     */
    public String getPrompt()
    {
        return prompt;
    }


    /**
     *  Checks whether the connection is open
     *
     *  @throws  DriverException if not
     */
    private void checkOpen() throws DriverException
    {
        if (isTelnet && in == null) {
            throw new DriverException("Connection not open");
        }
    }


    /**
     *  Checks whether a reply size is valid
     *
     *  @param  reply  The array of reply lines
     *  @param  count  The expected line count
     *  @throws  DriverException if unequal
     */
    public static void checkReplyLength(String[] reply, int count) throws DriverException
    {
        if (reply.length == count) return;
        if (reply.length == 0) {
            throw new DriverException("Missing response");
        }
        else {
            throw new DriverException("Incorrect response length: " + reply.length + "; first line = " + reply[0]); 
        }
    }


    /**
     *  Generates an invalid response exception
     *
     *  @param  resp  The response string
     *  @throws  DriverException always
     */
    public static void responseError(String resp) throws DriverException
    {
        throw new DriverException("Incorrect response: " + resp);
    }


    /**
     *  Reads available data.
     * 
     *  A polling loop is used because the socket timeout mechanism was
     *  found to be unreliable.
     *
     *  @param  buff     A buffer to receive the data
     *  @param  offset   The offset to the first available byte in buff
     *  @param  mleng    The maximum number of bytes to read
     *  @param  timeout  The read timeout (ms)
     *  @return  The number of bytes read
     *  @throws  DriverException
     */
    private int read(byte[] buff, int offset, int mleng, int timeout) throws DriverException
    {
        if (isTelnet) {
            long endTime = System.currentTimeMillis() + timeout;
            int count = Math.min(mleng, buff.length - offset);
            int leng = 0;
            while (leng == 0 && System.currentTimeMillis() < endTime) {
                try {
                    leng = in.read(buff, offset, count);
                }
                catch (SocketTimeoutException e) {
                }
                catch (IOException e) {
                    throw new DriverException(e);
                }
                if (leng < 0) {
                    throw new DriverException(new SocketException("Connection closed remotely"));
                }
                try {
                    Thread.sleep(POLL_PERIOD);
                }
                catch (InterruptedException e) {
                }
            }
            if (leng == 0) {
                throw new DriverTimeoutException("Read timed out");
            }
            return leng;
        }
        else {
            asc.setTimeout(timeout);
            return asc.readBytes(buff, offset, mleng);
        }
    }

}
