package org.lsst.ccs.drivers.ascii;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverConstants;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;

/**
 *  General access routines for a device using Ascii commands.
 *
 *  @author Owen Saxton
 */

public class Ascii implements DriverConstants {

    /**
     *  Public constants.
     */
    /** Connection type - network */
    @Deprecated
    public final static int CONN_TYPE_NETWORK = 0;

    /** Connection type - serial via FTDI chip */
    @Deprecated
    public final static int CONN_TYPE_FTDI    = 1;

    /** Connection type - serial via serial port */
    @Deprecated
    public final static int CONN_TYPE_SERIAL  = 2;

    /**
     *  Public enumerations.
     */
    /** Constructor options */
    public enum Option {

        /** Network connection not allowed */
        NO_NET,

        /** Serial connection not allowed */
        NO_SERIAL;
    }

    /** Data characteristics - parity */
    public enum Parity {

        /** No parity */
        NONE(PARITY_NONE),

        /** Odd parity */
        ODD(PARITY_ODD),

        /** Even parity */
        EVEN(PARITY_EVEN),

        /** Mark parity */
        MARK(PARITY_MARK),

        /** Space parity */
        SPACE(PARITY_SPACE);

        int value;

        Parity(int value) {
            this.value = value;
        }

        private int getValue() {
            return value;
        }
    }

    /** Data characteristics - number of data bits */
    public enum DataBits {

        /** Seven data bits */
        SEVEN(DBITS_SEVEN),

        /** Eight data bits */
        EIGHT(DBITS_EIGHT);

        int value;

        DataBits(int value) {
            this.value = value;
        }

        private int getValue() {
            return value;
        }
    }

    /** Data characteristics - number of stop bits */
    public enum StopBits {

        /** One stop bit */
        ONE(SBITS_ONE),

        /** Two stop bits */
        TWO(SBITS_TWO);

        int value;

        StopBits(int value) {
            this.value = value;
        }

        private int getValue() {
            return value;
        }
    }

    /** Data characteristics - flow control */
    public enum FlowCtrl {

        /** No flow control */
        NONE(FLOW_NONE),

        /** RTS/CTS flow control */
        RTS(FLOW_RTS_CTS),

        /** DTR/DSR flow control (ignored for serial port) */
        DTR(FLOW_DTR_DSR),

        /** XON/XOFF flow control */
        XON(FLOW_XON_XOFF),

        /** RTS/CTS and DTR/DSR flow control */
        RTS_DTR(FLOW_RTS_CTS | FLOW_DTR_DSR),

        /** RTS/CTS and XON/XOFF flow control */
        RTS_XON(FLOW_RTS_CTS | FLOW_XON_XOFF),

        /** DTR/DSR and XON/XOFF flow control */
        DTR_XON(FLOW_DTR_DSR | FLOW_XON_XOFF),

        /** RTS/CTS, DTR/DSR and XON/XOFF flow control */
        RTS_DTR_XON(FLOW_RTS_CTS | FLOW_DTR_DSR | FLOW_XON_XOFF);

        int value;

        FlowCtrl(int value) {
            this.value = value;
        }

        private int getValue() {
            return value;
        }
    }

    /** Command or response terminators */
    public enum Terminator {

        /** Carriage return */
        CR("\r"),

        /** Line feed */
        LF("\n"),

        /** Carriage return/Line feed */
        CRLF("\r\n");

        String value;

        Terminator(String value) {
            this.value = value;
        }

        private String getValue() {
            return value;
        }
    }

    /**
     *  Private & package constants & fields.
     */
    final static int
        PARITY_NONE   = 0,
        PARITY_ODD    = 1,
        PARITY_EVEN   = 2,
        PARITY_MARK   = 3,
        PARITY_SPACE  = 4,
        DBITS_EIGHT   = 0,
        DBITS_SEVEN   = 1,
        SBITS_ONE     = 0,
        SBITS_TWO     = 1,
        FLOW_NONE     = 0,
        FLOW_RTS_CTS  = 0x01,
        FLOW_DTR_DSR  = 0x02,
        FLOW_XON_XOFF = 0x04;

    private final static Map<Integer, ConnType> typeMap = new HashMap<>();
    static {
        typeMap.put(CONN_TYPE_NETWORK, ConnType.NET);
        typeMap.put(CONN_TYPE_FTDI,    ConnType.FTDI);
        typeMap.put(CONN_TYPE_SERIAL,  ConnType.SERIAL);
    }

    private final byte[] buff = new byte[4096];
    private final boolean noNet, noSerial;
    private AsciiIO io;
    private String cmndTerm = "\r\n";
    private byte[] respTerm = {};
    private int timeout = 1000;
    private int buffIn, buffOut;


    /**
     *  Constructor.
     */
    public Ascii() {
        this.noNet = false;
        this.noSerial = false;
    }


    /**
     *  Constructor.
     *
     *  @param  option  Disallowed connections option
     */
    public Ascii(Option option) {
        this.noNet = (option == Option.NO_NET);
        this.noSerial = (option == Option.NO_SERIAL);
    }


    /**
     *  Opens a connection to the device.
     *
     *  This is the open method that should be overridden by extending
     *  classes. since it is called by all other open methods.
     *
     *  @param  type   The enumerated type of connection to make
     *  @param  ident  The device identifier:
     *                   host name or IP address for network;
     *                   serial number for FTDI device;
     *                   device name for serial
     *  @param  parm1  The first device parameter:
     *                   port number for network;
     *                   baud rate for FTDI or serial
     *  @param  parm2  The second device parameter:
     *                   unused for network;
     *                   encoded data characteristics for FTDI or serial:
     *                     0 sets 8-bit, no parity, 1 stop bit, no flow ctrl
     *  @throws  DriverException
     */
    public synchronized void open(ConnType type, String ident, int parm1,
                                  int parm2) throws DriverException
    {
        if (io != null) {
            throw new DriverException("Device already connected");
        }

        AsciiIO newIo;

        switch (type) {

        case NET:
            if (noNet) {
                throw new DriverException("Network connection not allowed");
            }
            newIo = new AsciiIONet();
            break;

        case FTDI:
            if (noSerial) {
                throw new DriverException("FTDI connection not allowed");
            }
            newIo = new AsciiIOFtdi();
            break;

        case SERIAL:
            if (noSerial) {
                throw new DriverException("Serial connection not allowed");
            }
            newIo = new AsciiIOSerial();
            break;

        default:
            throw new DriverException("Connection type not handled: " + type);
        }

        newIo.open(ident, parm1, parm2);
        io = newIo;
        io.setTimeout(timeout);
    }


    /**
     *  Opens a connection to the device.
     *
     *  @param  type   The enumerated type of connection to make
     *  @param  ident  The device identifier:
     *                   host name or IP address for network;
     *                   serial number for FTDI device;
     *                   device name for serial
     *  @param  parm   The device parameter:
     *                   port number for network;
     *                   baud rate for FTDI or serial
     *  @throws  DriverException
     */
    public void open(ConnType type, String ident, int parm)
        throws DriverException
    {
        open(type, ident, parm, 0);
    }


    /**
     *  Opens a connection to a network device.
     *
     *  @param  host  The host name or IP address
     *  @param  port  The port number
     *  @throws  DriverException
     */
    public void openNet(String host, int port) throws DriverException
    {
        open(ConnType.NET, host, port);
    }


    /**
     *  Opens a connection to a serial device.
     *
     *  @param  devcName  The device name
     *  @param  baudRate  The baud rate
     *  @param  dataChar  The encoded data characteristics:
     *                     0 sets 8-bit, no parity, 1 stop bit, no flow ctrl
     *  @throws  DriverException
     */
    public void openSerial(String devcName, int baudRate, int dataChar) throws DriverException
    {
        open(ConnType.SERIAL, devcName, baudRate, dataChar);
    }


    /**
     *  Opens a connection to a serial device.
     *
     *  @param  devcName  The device name
     *  @param  baudRate  The baud rate
     *  @throws  DriverException
     */
    public void openSerial(String devcName, int baudRate) throws DriverException
    {
        open(ConnType.SERIAL, devcName, baudRate);
    }


    /**
     *  Opens a connection to an FTDI device.
     *
     *  @param  serialNo  The USB serial number
     *  @param  baudRate  The baud rate
     *  @param  dataChar  The encoded data characteristics:
     *                     0 sets 8-bit, no parity, 1 stop bit, no flow ctrl
     *  @throws  DriverException
     */
    public void openFtdi(String serialNo, int baudRate, int dataChar) throws DriverException
    {
        open(ConnType.FTDI, serialNo, baudRate, dataChar);
    }


    /**
     *  Opens a connection to an FTDI device.
     *
     *  @param  serialNo  The USB serial number
     *  @param  baudRate  The baud rate
     *  @throws  DriverException
     */
    public void openFtdi(String serialNo, int baudRate) throws DriverException
    {
        open(ConnType.FTDI, serialNo, baudRate);
    }


    /**
     *  Opens a connection to the device (deprecated).
     *
     *  @param  type   The type of connection to make
     *  @param  ident  The device identifier:
     *                   host name or IP address for network;
     *                   [node:]serial number for FTDI device;
     *                   device name for serial
     *  @param  parm1  The first device parameter:
     *                   port number for network;
     *                   baud rate for FTDI or serial
     *  @param  parm2  The second device parameter:
     *                   unused for network;
     *                   encoded data characteristics for FTDI or serial:
     *                     0 sets 8-bit, no parity, 1 stop bit, no flow ctrl
     *  @throws  DriverException
     */
    @Deprecated
    public final void open(int type, String ident, int parm1, int parm2)
        throws DriverException
    {
        ConnType cType = typeMap.get(type);
        if (cType == null) {
            throw new DriverException("Invalid connection type: " + type);
        }
        open(cType, ident, parm1, parm2);
    }


    /**
     *  Opens a connection to the device (deprecated).
     *
     *  @param  type   The type of connection to make
     *  @param  ident  The device identifier:
     *                   host name or IP address for network;
     *                   [node:]serial number for FTDI device;
     *                   device name for serial
     *  @param  parm   The device parameter:
     *                   port number for network;
     *                   baud rate for FTDI or serial
     *  @throws  DriverException
     */
    @Deprecated
    public void open(int type, String ident, int parm) throws DriverException
    {
        open(type, ident, parm, 0);
    }


    /**
     *  Generates the serial data characteristics parameter.
     *
     *  @param  dataBits  The enumerated number of data bits
     *  @param  stopBits  The enumerated number of stop bits
     *  @param  parity    The enumerated parity
     *  @param  flowCtrl  The enumerated flow control
     *  @return  The encoded data characteristics
     */
    public static int makeDataCharacteristics(DataBits dataBits, StopBits stopBits,
                                              Parity parity, FlowCtrl flowCtrl)
    {
        return (dataBits.getValue() << 24) | (stopBits.getValue() << 16)
                 | (parity.getValue() << 8) | flowCtrl.getValue();
    }


    /**
     *  Closes the device connection.
     *
     *  @throws  DriverException
     */
    public synchronized void close() throws DriverException
    {
        checkOpen();
        try {
            io.close();
        }
        finally {
            io = null;
            buffIn = buffOut = 0;
        }
    }


    /**
     *  Closes the device connection silently.
     *
     *  @return  Whether or not the close caused an error
     */
    public boolean closeSilent()
    {
        try {
            close();
            return true;
        }
        catch (DriverException e) {
            return false;
        }
    }


    /**
     *  Writes a command.
     *
     *  @param  command  The command to write, excluding terminator
     *  @throws  DriverException
     */
    public synchronized void write(String command) throws DriverException
    {
        checkOpen();
        writeBytes((command + cmndTerm).getBytes());
    }


    /**
     *  Reads a terminated response.
     *
     *  The current version accepts only a single-byte or a two-byte terminator.
     *  The previous version accepted either a CR or LF as a terminator.  But
     *  this must remain supported for some time since a given program may 
     *  contain some code that depends on the old behavior and some that depends
     *  on the new.  The default behavior will be the old, indicated by a zero-
     *  byte terminator.  The new behavior will come into effect when the new
     *  setReceiveTerm method is called.
     *
     *  @return  The command response string
     *  @throws  DriverException
     *  @throws  DriverTimeoutException
     */
    public synchronized String read() throws DriverException
    {
        checkOpen();
        StringBuilder sBuff = null;
        int curr = buffOut, term = -1;
        while (true) {
            while (curr < buffIn) {
                byte buffByte = buff[curr++];
                if (respTerm.length == 0) {
                    if (buffByte != 0x0a && buffByte != 0x0d) continue;
                    if (curr == buffOut + 1) {
                        buffOut = curr;
                        continue;
                    }
                    term = curr - 1;
                }
                else {
                    if (term < 0) {
                        if (buffByte != respTerm[0]) continue;
                        term = curr - 1;
                        if (respTerm.length != 1) continue;
                    }
                    else {
                        if (buffByte != respTerm[1]) {
                            term = -1;
                            continue;
                        }
                    }
                }
                String resp = new String(buff, buffOut, term - buffOut);
                buffOut = term + respTerm.length;
                if (sBuff == null) {
                    return resp;
                }
                else {
                    return sBuff.append(resp).toString();
                }
            }
            if (buffOut > 0) {
                System.arraycopy(buff, buffOut, buff, 0, buffIn - buffOut);
                buffIn -= buffOut;
                term -= buffOut;
                buffOut = 0;
            }
            if (buffIn >= buff.length) {
                if (sBuff == null) {
                    sBuff = new StringBuilder(new String(buff, 0, buffIn - 1));
                }
                else {
                    sBuff.append(new String(buff, 0, buffIn - 1));
                }
                buff[0] = buff[buffIn - 1];
                term -= buffIn - 1;
                buffIn = 1;
            }
            curr = buffIn;
            try {
                buffIn += readBytes(buff, buffIn);
            }
            catch (DriverTimeoutException re) {
                throw re;
            }
            catch (DriverException re) {
                closeSilent();
                throw re;
            }
        }
    }


    /**
     *  Reads a response after writing a command.
     *
     *  @param  command  The command to write, excluding terminator
     *  @return  The command response string
     *  @throws  DriverException
     *  @throws  DriverTimeoutException
     */
    public synchronized String read(String command) throws DriverException
    {
        flush();
        write(command);
        return read();
    }


    /**
     *  Writes a command as bytes.
     *
     *  @param  command  The command to write, including any terminator
     *  @throws  DriverException
     */
    public void writeBytes(byte[] command) throws DriverException
    {
        checkOpen();
        writeBytes(command, 0, command.length);
    }


    /**
     *  Writes a command as bytes.
     *
     *  @param  command  The command to write, including any terminator
     *  @param  offset   The offset to the first byte to write
     *  @param  leng     The number of bytes to write
     *  @throws  DriverException
     */
    public void writeBytes(byte[] command, int offset, int leng)
        throws DriverException
    {
        checkOpen();
        io.write(command, offset, leng);
    }


    /**
     *  Reads available response data as bytes.
     *
     *  @param  buff    The buffer to receive the response data
     *  @param  offset  The offset to the first available byte in the buffer
     *  @return  The number of bytes read
     *  @throws  DriverException
     *  @throws  DriverTimeoutException
     */
    public int readBytes(byte[] buff, int offset) throws DriverException
    {
        checkOpen();
        return readBytes(buff, offset, buff.length - offset);
    }


    /**
     *  Reads available response data as bytes.
     *
     *  @param  buff    The buffer to receive the response data
     *  @param  offset  The offset to the first available byte in the buffer
     *  @param  leng    The maximum number of bytes to read
     *  @return  The number of bytes read
     *  @throws  DriverException
     *  @throws  DriverTimeoutException
     */
    public int readBytes(byte[] buff, int offset, int leng)
        throws DriverException
    {
        checkOpen();
        return io.read(buff, offset, leng);
    }


    /**
     *  Flushes any unread data.
     *
     *  @throws  DriverException
     */
    public synchronized void flush() throws DriverException
    {
        checkOpen();
        buffIn = buffOut;
        io.flush();
    }


    /**
     *  Sets the response terminator.
     *
     *  @param  term  The expected terminator for a response (CR, LF or CRLF)
     */
    public void setResponseTerm(Terminator term)
    {
        respTerm = term.getValue().getBytes();
    }


    /**
     *  Sets the command terminator.
     *
     *  @param  term  The terminator to be appended to each command (CR, LF or CRLF)
     */
    public void setCommandTerm(Terminator term)
    {
        cmndTerm = term.getValue();
    }


    /**
     *  Sets both the command & response terminators.
     *
     *  @param  term  The terminator to be used
     */
    public void setTerminator(Terminator term)
    {
        setCommandTerm(term);
        setResponseTerm(term);
    }


    /**
     *  Sets the command terminator.
     *
     *  @param  term  The terminator string to be used
     */
    @Deprecated
    public void setTerminator(String term)
    {
        cmndTerm = term;
    }


    /**
     *  Sets the read timeout.
     *
     *  @param  time  The read timeout (sec).  0 means no timeout.
     *  @throws  DriverException
     */
    public void setTimeout(double time) throws DriverException
    {
        setTimeout((int)(1000 * time));
    }


    /**
     *  Sets the read timeout.
     *
     *  @param  time  The read timeout (msec).  0 means no timeout.
     *  @throws  DriverException
     */
    public synchronized void setTimeout(int time) throws DriverException
    {
        timeout = time;
        if (io != null) {
            io.setTimeout(timeout);
        }
    }


    /**
     *  Checks that the connection is open.
     *
     *  @throws  DriverException
     */
    private void checkOpen() throws DriverException
    {
        if (io == null) {
            throw new DriverException("Device not connected");
        }
    }

}
