package org.lsst.ccs.drivers.ascii;

import java.io.PrintStream;
import jline.console.ConsoleReader;
import org.kohsuke.args4j.Argument;
import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;
import org.kohsuke.args4j.Option;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;

/**
 ************************************************************
 *
 *  Program to enable direct talking to a commanded device.
 *
 *  @author Owen Saxton
 *
 ************************************************************
 */
public class TalkAscii {

    private final static PrintStream out = System.out;
    private final static String[] terms = {"\r\n", "\r", "\n"};
    private final Thread readW = new Thread(new Reader());
    private final Ascii dev = new Ascii();
    private int nRead = 0, nByte = 0, nTimeout = 0;
    private boolean open;

    private enum Terminator {
        CRLF, CR, LF;
    }


   /**
    *  Inner class for holding command line options.
    */
    private static class Options {

        @Option(name="-c", metaVar="<connType>",
                usage="Connection type: net (default), ftdi or serial")
        private final Ascii.ConnType type = Ascii.ConnType.NET;

        @Option(name="-p", metaVar="<parity>",
                usage="Parity: none (default), odd, even, mark or space")
        private final Ascii.Parity parity = Ascii.Parity.NONE;

        @Option(name="-d", metaVar="<dbits>",
                usage="No. data bits: eight (default) or seven")
        private final Ascii.DataBits dataBits = Ascii.DataBits.EIGHT;

        @Option(name="-s", metaVar="<sbits>",
                usage="No. stop bits: one (default) or two")
        private final Ascii.StopBits stopBits = Ascii.StopBits.ONE;

        @Option(name="-f", metaVar="<flow>",
                usage="Flow control: none (default), rts, dtr or xon")
        private final Ascii.FlowCtrl flowCtrl = Ascii.FlowCtrl.NONE;

        @Option(name="-r", metaVar="<timeout>",
                usage="Read timeout (sec) (default = 0.2)")
        private final double timeout = 0.1;

        @Option(name="-t", metaVar="<term>",
                usage="Command terminator: crlf (default), cr or lf")
        private final Terminator term = Terminator.CRLF;

        @Option(name="-help", usage="Displays this help")
        private boolean showHelp;

        @Argument(index=0, required=true, metaVar="<ident>",
                  usage="Device identifier (node, serial no. or device name)")
        private String ident;

        @Argument(index=1, required=true, metaVar="<parm>",
                  usage="Device parameter (port number or baud rate)")
        private int parm;

    }


   /**
    *  Inner class to implement device reading thread.
    */
    private class Reader implements Runnable {

        @Override
        public void run()
        {
            while (open) {
                try {
                    try {
                        Thread.sleep(1);  // Avoid hogging the Ascii lock
                    }
                    catch (InterruptedException e) {}
                    String reply = dev.read();
                    out.println(reply);
                    nRead++;
                    nByte += reply.length();
                }
                catch (DriverTimeoutException e) {
                    nTimeout++;
                }
                catch (DriverException e) {
                    if (open) {
                        out.println(e);
                        System.exit(0);
                    }
                }
            }
        }

    }


   /**
    *  Main program.
    *
    *  @param  args  Command-line arguments
    */
    public static void main(String[] args)
    {
        Options optns = new Options();
        CmdLineParser parser = new CmdLineParser(optns);
        try {
            parser.parseArgument(args);
            if (optns.showHelp) {
                showHelp(parser);
            }
            else {
                (new TalkAscii()).run(optns);
            }
        }
        catch (CmdLineException e) {
            if (!optns.showHelp) {
                out.println(e.getMessage());
            }
            showHelp(parser);
        }
        catch (Exception e) {
            out.println(e);
        }
        System.exit(0);
    }


   /**
    *  Sends commands to the device.
    *
    *  Loops reading and processing each new typed command line.
    */
    private void run(Options optns) throws Exception
    {
        dev.open(optns.type, optns.ident, optns.parm,
                 Ascii.makeDataCharacteristics(optns.dataBits, optns.stopBits,
                                               optns.parity, optns.flowCtrl));
        open = true;
        dev.setTimeout(optns.timeout);
        dev.setTerminator(terms[optns.term.ordinal()]);
        readW.setDaemon(true);
        readW.start();
        ConsoleReader readC = new ConsoleReader();
        out.println("Connected. Type CTRL-D to exit, CR for read stats.");

        while (true) {
            String line = readC.readLine("");
            if (line == null) break;
            if (line.isEmpty()) {
                out.format("#reads: %s; #bytes: %s; #timeouts: %s %n",
                           nRead, nByte, nTimeout);
            }
            else {
                dev.write(line);
            }
        }

        open = false;
        dev.close();
    }


   /**
    *  Displays help.
    */
    private static void showHelp(CmdLineParser parser)
    {
        out.println("Usage: TalkAscii [options...] <ident> <parm>");
        out.println("Parameters & options:");
        parser.printUsage(out);
    }

}
