package org.lsst.ccs.drivers.ascii;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;

/**
 *****************************************************
 *
 *  Communications I/O routines for network devices.
 *
 *  @author Owen Saxton
 *
 *****************************************************
 */

public class AsciiIONet implements AsciiIO {

   /**
    *  Private fields.
    */
    private Socket sock;
    private InputStream in;
    private OutputStream out;


   /**
    *  Opens a connection.
    *
    *  @param  host   The host name or IP address
    *
    *  @param  port   The port number
    *
    *  @param  dummy  Dummy argument
    *
    *  @throws  DriverException
    */
    @Override
    public void open(String host, int port, int dummy) throws DriverException
    {
        try {
            sock = new Socket(host, port);
            in = sock.getInputStream();
            out = sock.getOutputStream();
        }
        catch (IOException e) {
            throw new DriverException(e);
        }
    }


   /**
    *  Closes the connection.
    *
    *  @throws  DriverException
    */
    @Override
    public void close() throws DriverException
    {
        try {
            sock.close();
        }
        catch (IOException e) {
            throw new DriverException(e);
        }
    }


   /**
    *  Writes command data.
    *
    *  @param  buff    The buffer containing the command to write
    *
    *  @param  offset  The offset to the command
    *
    *  @param  leng    The length of the command
    *
    *  @throws  DriverException
    */
    @Override
    public void write(byte[] buff, int offset, int leng)
        throws DriverException
    {
        try {
            out.write(buff, offset, leng);
        }
        catch (IOException e) {
            throw new DriverException(e);
        }
    }


   /**
    *  Reads available response data.
    *
    *  @param  buff    The buffer to receive the response data
    *
    *  @param  offset  The offset to the first available byte in the buffer:
    *                  must be < the buffer length
    *
    *  @param  mleng   The maximum number of bytes to read: must be > 0
    *
    *  @return  The number of bytes read
    *
    *  @throws  DriverException
    *  @throws  DriverTimeoutException
    */
    @Override
    public int read(byte[] buff, int offset, int mleng) throws DriverException
    {
        if (offset + mleng > buff.length) {
            mleng = buff.length - offset;
        }
        try {
            int leng = 0, count = in.available();
            if (count == 0) {
                leng = in.read(buff, offset, 1);
                if (leng <= 0) {
                    throw new DriverException("Connection closed");
                }
                offset++;
                mleng--;
                count = in.available();
            }
            if (count > mleng) {
                count = mleng;
            }
            if (count > 0) {
                leng += in.read(buff, offset, count);
            }
            return leng;
        }
        catch (SocketTimeoutException e) {
            throw new DriverTimeoutException("Read timed out");
        }
        catch (IOException e) {
            throw new DriverException(e);
        }
    }


   /**
    *  Flushes any available response data.
    *
    *  @throws  DriverException
    */
    @Override
    public void flush() throws DriverException
    {
        try {
            in.skip(in.available());
        }
        catch (IOException e) {
            throw new DriverException(e);
        }
    }

    
   /**
    *  Sets the receive timeout.
    *
    *  @param  timeout  The receive timeout (ms).  0 means no timeout.
    *
    *  @throws  DriverException
    */
    @Override
    public void setTimeout(int timeout) throws DriverException
    {
        try {
            sock.setSoTimeout(timeout);
        }
        catch (SocketException e) {
            throw new DriverException(e);
        }
    }

}
