package org.lsst.ccs.drivers.ascii;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.drivers.serial.SerialPort;

/**
 *****************************************************************************
 **
 **  Communications I/O routines for serial devices.
 **
 **  @author Owen Saxton
 **
 *****************************************************************************
 */

public class AsciiIOSerial implements AsciiIO {

   /**
    **************************************************************************
    **
    **  Private fields.
    **
    **************************************************************************
    */
    private final static Map<Integer, Integer> dbitsMap = new HashMap<>();
    static {
        dbitsMap.put(Ascii.DBITS_SEVEN, 7);
        dbitsMap.put(Ascii.DBITS_EIGHT, 8);
    }

    private final static Map<Integer, Integer> sbitsMap = new HashMap<>();
    static {
        sbitsMap.put(Ascii.SBITS_ONE, 1);
        sbitsMap.put(Ascii.SBITS_TWO, 2);
    }

    private final static Map<Integer, Integer> parityMap = new HashMap<>();
    static {
        parityMap.put(Ascii.PARITY_NONE,  0);
        parityMap.put(Ascii.PARITY_ODD,   1);
        parityMap.put(Ascii.PARITY_EVEN,  2);
        parityMap.put(Ascii.PARITY_MARK,  3);
        parityMap.put(Ascii.PARITY_SPACE, 4);
    }

    private final SerialPort port = new SerialPort();
    private int timeout;


   /**
    **************************************************************************
    **
    **  Opens a connection.
    **
    **  @param  dname  The serial device name
    **
    **  @param  baud   The baud rate
    **
    **  @param  dchar  The data characteristics
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void open(String dname, int baud, int dchar) throws DriverException
    {
        Integer dataBits = dbitsMap.get((dchar >> 24) & 0xff);
        if (dataBits == null) {
            throw new DriverException("Invalid number of data bits");
        }
        Integer stopBits = sbitsMap.get((dchar >> 16) & 0xff);
        if (stopBits == null) {
            throw new DriverException("Invalid number of stop bits");
        }
        Integer parity = parityMap.get((dchar >> 8) & 0xff);
        if (parity == null) {
            throw new DriverException("Invalid parity");
        }
        int fc = dchar & 0xff;
        int flowCtrl = ((fc & Ascii.FLOW_RTS_CTS) != 0 ? 0x03 : 0)
                        | ((fc & Ascii.FLOW_XON_XOFF) != 0 ? 0x0c : 0);
        port.openPort(dname);
        port.setParams(baud, dataBits, stopBits, parity);
        port.setFlowControlMode(flowCtrl);
    }


   /**
    **************************************************************************
    **
    **  Closes the connection.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void close() throws DriverException
    {
        port.closePort();
    }


   /**
    **************************************************************************
    **
    **  Writes command data.
    **
    **  @param  buff    The buffer containing the command to write
    **
    **  @param  offset  The offset to the command
    **
    **  @param  leng    The length of the command
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void write(byte[] buff, int offset, int leng) throws DriverException
    {
        port.writeBytes(buff, offset, leng);
    }


   /**
    **************************************************************************
    **
    **  Reads available response data.
    **
    **  @param  buff    The buffer to receive the response data
    **
    **  @param  offset  The offset to the first available byte in the buffer
    **
    **  @param  mleng   The maximum number of bytes to read
    **
    **  @return  The number of bytes read
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    @Override
    public int read(byte[] buff, int offset, int mleng) throws DriverException
    {
        return port.read(buff, offset, mleng, timeout);
    }


   /**
    **************************************************************************
    **
    **  Flushes any available response data.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void flush() throws DriverException
    {
        port.readBytes();
    }

    
   /**
    **************************************************************************
    **
    **  Sets the receive timeout.
    **
    **  @param  timeout  The receive timeout (ms).  0 means no timeout.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void setTimeout(int timeout) throws DriverException
    {
        this.timeout = timeout;
    }

}
