package org.lsst.ccs.drivers.ascii;

import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;

/**
 ****************************************************************************** 
 ** 
 **  Program to test the Ascii command device driver.
 ** 
 **  @author Owen Saxton
 ** 
 ****************************************************************************** 
 */
public class TestAscii {

    protected final Ascii dev;

    protected static enum ConnType {

        NET(Ascii.CONN_TYPE_NETWORK),
        FTDI(Ascii.CONN_TYPE_FTDI),
        SERIAL(Ascii.CONN_TYPE_SERIAL);

        int value;

        ConnType(int value)
        {
            this.value = value;
        }

        public int getValue()
        {
            return value;
        }

    }


   /**
    *************************************************************************** 
    ** 
    **  Constructor.
    ** 
    *************************************************************************** 
    */
    public TestAscii()
    {
        dev = new Ascii();
    }


   /**
    *************************************************************************** 
    ** 
    **  Constructor.
    **
    **  @param  dev  The device to use
    **
    *************************************************************************** 
    */
    public TestAscii(Ascii dev)
    {
        this.dev = dev;
    }


   /**
    *************************************************************************** 
    ** 
    **  Opens connection to a device.
    **
    **  @param  type   The connection type
    **
    **  @param  ident  The device identification
    **
    **  @param  param  The device parameter
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="open", description="Open connection to general device")
    public void open(@Argument(name="type",
                               description="Connection type: net, serial or ftdi")
                     ConnType type,
                     @Argument(name="ident", description="Device identifier")
                     String ident,
                     @Argument(name="param", description="Device parameter")
                     int param
                     ) throws DriverException
    {
        dev.open(type.getValue(), ident, param);
    }


   /**
    *************************************************************************** 
    ** 
    **  Opens connection to a serially-connected device.
    ** 
    **  @param  type    The connection type
    **
    **  @param  ident   The device identification
    **
    **  @param  baud    The baud rate
    **
    **  @param  dbits   The number of data bits (eight or seven)
    **
    **  @param  sbits   The number of stop bits (one or two)
    **
    **  @param  parity  The parity (none, odd, even, mark or space)
    **
    **  @param  flow    The flow control (none, xon, rts or dtr)
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="open", description="Open connection to serial device")
    public void open(@Argument(name="type",
                               description="Connection type: net, serial or ftdi")
                     ConnType type,
                     @Argument(name="ident", description="Device identifier")
                     String ident,
                     @Argument(name="baud", description="Baud rate")
                     int baud,
                     @Argument(name="dbits",
                               description="The number of data bits")
                     Ascii.DataBits dbits,
                     @Argument(name="sbits",
                               description="The number of stop bits")
                     Ascii.StopBits sbits,
                     @Argument(name="parity",
                               description="The parity")
                     Ascii.Parity parity,
                     @Argument(name="flow",
                               description="The flow control")
                     Ascii.FlowCtrl flow
                     ) throws DriverException
    {
        dev.open(type.getValue(), ident, baud,
                 Ascii.makeDataCharacteristics(dbits, sbits, parity, flow));
    }


   /**
    *************************************************************************** 
    ** 
    **  Closes connection to a device.
    ** 
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="close", description="Close connection to device")
    public void close() throws DriverException
    {
        dev.close();
    }


   /**
    *************************************************************************** 
    ** 
    **  Writes a command.
    **
    **  @param  command  The command to send (without terminator)
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="write", description="Write command to device")
    public void write(@Argument(name="command", description="Command to write")
                      String command
                      ) throws DriverException
    {
        dev.write(command);
    }


   /**
    *************************************************************************** 
    ** 
    **  Reads a response.
    **
    **  @return  The received response
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="read", description="Read response from device")
    public String read() throws DriverException
    {
        return dev.read();
    }


   /**
    *************************************************************************** 
    ** 
    **  Writes a command and reads the response.
    **
    **  @param  command  The command to send (without terminator)
    **
    **  @return  The response string
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="read", description="Write command and read response")
    public String read(@Argument(name="command", description="Command to write")
                       String command
                       ) throws DriverException
    {
        return dev.read(command);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the response timeout.
    **
    **  @param  time  The timeout (secs); 0 means no timeout
    **
    **  @throws  DriverException
    **
    *************************************************************************** 
    */
    @Command(name="timeout", description="Set read response timeout")
    public void timeout(@Argument(name="time", description="Timeout value (sec)")
                        double time
                        ) throws DriverException
    {
        dev.setTimeout(time);
    }


   /**
    *************************************************************************** 
    ** 
    **  Sets the command terminator.
    **
    **  @param  term  The terminator to set
    **
    *************************************************************************** 
    */
    @Command(name="setterm", description="Set command terminator")
    public void setTerm(@Argument(name="term", description="Terminator string")
                        String term)
    {
        char[] cTerm = new char[term.length()];
        term.getChars(0, cTerm.length, cTerm, 0);
        int jout = 0;
        for (int jin = 0; jin < cTerm.length; jin++) {
            if (cTerm[jin] == '\\' && jin < cTerm.length - 1) {
                jin++;
                switch (cTerm[jin]) {
                case 'r': cTerm[jin] = '\r'; break;
                case 'n': cTerm[jin] = '\n'; break;
                case 't': cTerm[jin] = '\t'; break;
                case 'b': cTerm[jin] = '\b'; break;
                case 'f': cTerm[jin] = '\f'; break;
                }
            }
            cTerm[jout++] = cTerm[jin];
        }
        dev.setTerminator(new String(cTerm, 0, jout));
    }

}
