package org.lsst.ccs.drivers.ascii;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;

/**
 *****************************************************************************
 **
 **  General access routines for a device using Ascii commands.
 **
 **  @author Owen Saxton
 **
 *****************************************************************************
 */

public class Ascii {

   /**
    **************************************************************************
    **
    **  Public constants.
    **
    **************************************************************************
    */
    /** Connection type - network */
    public final static int CONN_TYPE_NETWORK = 0;

    /** Connection type - serial via FTDI chip */
    public final static int CONN_TYPE_FTDI    = 1;

    /** Connection type - serial via serial port */
    public final static int CONN_TYPE_SERIAL  = 2;
    
   /**
    **************************************************************************
    **
    **  Public enumerations.
    **
    **************************************************************************
    */
    /** Data characteristics - parity */
    public enum Parity {

        /** No parity */
        NONE(PARITY_NONE),

        /** Odd parity */
        ODD(PARITY_ODD),

        /** Even parity */
        EVEN(PARITY_EVEN),

        /** Mark parity */
        MARK(PARITY_MARK),

        /** Space parity */
        SPACE(PARITY_SPACE);

        int value;

        Parity(int value) {
            this.value = value;
        }

        public int getValue() {
            return value;
        }
    }

    /** Data characteristics - number of data bits */
    public enum DataBits {

        /** Seven data bits */
        SEVEN(DBITS_SEVEN),

        /** Eight data bits */
        EIGHT(DBITS_EIGHT);

        int value;

        DataBits(int value) {
            this.value = value;
        }

        public int getValue() {
            return value;
        }
    }

    /** Data characteristics - number of stop bits */
    public enum StopBits {

        /** One stop bit */
        ONE(SBITS_ONE),

        /** Two stop bits */
        TWO(SBITS_TWO);

        int value;

        StopBits(int value) {
            this.value = value;
        }

        public int getValue() {
            return value;
        }
    }

    /** Data characteristics - flow control */
    public enum FlowCtrl {

        /** No flow control */
        NONE(FLOW_NONE),

        /** RTS/CTS flow control */
        RTS(FLOW_RTS_CTS),

        /** DTR/DSR flow control (ignored for serial port) */
        DTR(FLOW_DTR_DSR),

        /** XON/XOFF flow control */
        XON(FLOW_XON_XOFF),

        /** RTS/CTS and DTR/DSR flow control */
        RTS_DTR(FLOW_RTS_CTS | FLOW_DTR_DSR),

        /** RTS/CTS and XON/XOFF flow control */
        RTS_XON(FLOW_RTS_CTS | FLOW_XON_XOFF),

        /** DTR/DSR and XON/XOFF flow control */
        DTR_XON(FLOW_DTR_DSR | FLOW_XON_XOFF),

        /** RTS/CTS, DTR/DSR and XON/XOFF flow control */
        RTS_DTR_XON(FLOW_RTS_CTS | FLOW_DTR_DSR | FLOW_XON_XOFF);

        int value;

        FlowCtrl(int value) {
            this.value = value;
        }

        public int getValue() {
            return value;
        }
    }

   /**
    **************************************************************************
    **
    **  Private constants & fields.
    **
    **************************************************************************
    */
    final static int
        PARITY_NONE   = 0,
        PARITY_ODD    = 1,
        PARITY_EVEN   = 2,
        PARITY_MARK   = 3,
        PARITY_SPACE  = 4,
        DBITS_EIGHT   = 0,
        DBITS_SEVEN   = 1,
        SBITS_ONE     = 0,
        SBITS_TWO     = 1,
        FLOW_NONE     = 0,
        FLOW_RTS_CTS  = 0x01,
        FLOW_DTR_DSR  = 0x02,
        FLOW_XON_XOFF = 0x04;

    private final static byte CR = 0x0d, LF = 0x0a;
    private final byte[] buff = new byte[4096];
    private AsciiIO io;
    private String terminator = "\r\n";
    private int timeout = 1000;
    private int buffIn, buffOut;


   /**
    **************************************************************************
    **
    **  Opens a connection to the device.
    **
    **  @param  type   The type of connection to make
    **
    **  @param  ident  The device identifier:
    **                   host name or IP address for network;
    **                   [node:]serial number for FTDI device;
    **                   device name for serial
    **
    **  @param  parm   The device parameter:
    **                   port number for network;
    **                   baud rate for FTDI or serial
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    public void open(int type, String ident, int parm) throws DriverException
    {
        open(type, ident, parm, 0);
    }


   /**
    **************************************************************************
    **
    **  Opens a connection to the device.
    **
    **  @param  type   The type of connection to make
    **
    **  @param  ident  The device identifier:
    **                   host name or IP address for network;
    **                   [node:]serial number for FTDI device;
    **                   device name for serial
    **
    **  @param  parm1  The first device parameter:
    **                   port number for network;
    **                   baud rate for FTDI or serial
    **
    **  @param  parm2  The second device parameter:
    **                   unused for network;
    **                   encoded data characteristics for FTDI or serial:
    **                     0 sets 8-bit, no parity, 1 stop bit, no flow ctrl
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    public synchronized void open(int type, String ident, int parm1, int parm2)
        throws DriverException
    {
        if (io != null) {
            throw new DriverException("Device already connected");
        }

        AsciiIO newIo;

        switch (type) {

        case CONN_TYPE_NETWORK:
            newIo = new AsciiIONet();
            break;

        case CONN_TYPE_FTDI:
            newIo = new AsciiIOFtdi();
            break;

        case CONN_TYPE_SERIAL:
            newIo = new AsciiIOSerial();
            break;

        default:
            throw new DriverException("Invalid connection type: " + type);
        }

        newIo.open(ident, parm1, parm2);
        io = newIo;
        io.setTimeout(timeout);
    }


   /**
    **************************************************************************
    **
    **  Generates the serial data characteristics parameter.
    **
    **  @param  dataBits  The enumerated number of data bits
    **
    **  @param  stopBits  The enumerated number of stop bits
    **
    **  @param  parity    The enumerated parity
    **
    **  @param  flowCtrl  The enumerated flow control
    **
    **  @return  The encoded data characteristics
    **
    **************************************************************************
    */
    public static int makeDataCharacteristics(DataBits dataBits, StopBits stopBits,
                                              Parity parity, FlowCtrl flowCtrl)
    {
        return (dataBits.getValue() << 24) | (stopBits.getValue() << 16)
                 | (parity.getValue() << 8) | flowCtrl.getValue();
    }


   /**
    **************************************************************************
    **
    **  Closes the device connection.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    public synchronized void close() throws DriverException
    {
        checkOpen();
        try {
            io.close();
        }
        finally {
            io = null;
            buffIn = buffOut = 0;
        }
    }


   /**
    **************************************************************************
    **
    **  Closes the device connection silently.
    **
    **  @return  Whether or not the close caused an error
    **
    **************************************************************************
    */
    public boolean closeSilent()
    {
        try {
            close();
            return true;
        }
        catch (DriverException e) {
            return false;
        }
    }


   /**
    **************************************************************************
    **
    **  Writes a command.
    **
    **  @param  command  The command to write, excluding terminator
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    public synchronized void write(String command) throws DriverException
    {
        checkOpen();
        io.write((command + terminator).getBytes());
    }


   /**
    **************************************************************************
    **
    **  Reads a response.
    **
    **  @return  The command response string
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    public synchronized String read() throws DriverException
    {
        checkOpen();
        StringBuilder sBuff = null;
        try {
            int start = buffOut, end = buffIn, term = -1;
            while (true) {
                for (int j = start; j < end; j++) {
                    if (buff[j] == CR || buff[j] == LF) {
                        if (j == buffOut) {
                            buffOut++;
                        }
                        else {
                            term = j;
                            break;
                        }
                    }
                }
                if (term >= 0) {
                    String resp = new String(buff, buffOut, term - buffOut);
                    buffOut = term + 1;
                    if (sBuff == null) {
                        return resp;
                    }
                    else {
                        return sBuff.append(resp).toString();
                    }
                }
                if (buffOut > 0) {
                    System.arraycopy(buff, buffOut, buff, 0, buffIn - buffOut);
                    buffIn -= buffOut;
                    buffOut = 0;
                }
                if (buffIn >= buff.length) {
                    if (sBuff == null) {
                        sBuff = new StringBuilder(new String(buff));
                    }
                    else {
                        sBuff.append(new String(buff));
                    }
                    buffIn = buffOut = 0;
                }
                start = buffIn;
                buffIn += io.read(buff, buffIn);
                end = buffIn;
            }
        }
        catch (DriverTimeoutException re) {
            throw re;
        }
        catch (DriverException re) {
            closeSilent();
            throw re;
        }
    }


   /**
    **************************************************************************
    **
    **  Reads a response after writing a command.
    **
    **  @param  command  The command to write, excluding terminator
    **
    **  @return  The command response string
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    public synchronized String read(String command) throws DriverException
    {
        flush();
        write(command);
        return read();
    }


   /**
    **************************************************************************
    **
    **  Writes a command as bytes.
    **
    **  @param  command  The command to write, including any terminator
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    public void writeBytes(byte[] command) throws DriverException
    {
        checkOpen();
        io.write(command);
    }


   /**
    **************************************************************************
    **
    **  Reads available response data as bytes.
    **
    **  @param  buff    The buffer to receive the response data
    **
    **  @param  offset  The offset to the first available byte in the buffer
    **
    **  @return  The number of bytes read
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    public int readBytes(byte[] buff, int offset) throws DriverException
    {
        checkOpen();
        return io.read(buff, offset);
    }


   /**
    **************************************************************************
    **
    **  Flushes any unread data.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    public synchronized void flush() throws DriverException
    {
        checkOpen();
        buffIn = buffOut;
        io.flush();
    }


   /**
    **************************************************************************
    **
    **  Sets the command terminator.
    **
    **  @param  term  The terminator to be appended to each sent command
    **
    **************************************************************************
    */
    public void setTerminator(String term)
    {
        terminator = term;
    }


   /**
    **************************************************************************
    **
    **  Sets the read timeout.
    **
    **  @param  time  The read timeout (sec).  0 means no timeout.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    public synchronized void setTimeout(double time) throws DriverException
    {
        timeout = (int)(1000 * time);
        if (io != null) {
            io.setTimeout(timeout);
        }
    }


   /**
    **************************************************************************
    **
    **  Checks that the connection is open.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    private void checkOpen() throws DriverException
    {
        if (io == null) {
            throw new DriverException("Device not connected");
        }
    }

}
