package org.lsst.ccs.drivers.ascii;

import java.io.PrintStream;
import jline.console.ConsoleReader;
import org.kohsuke.args4j.Argument;
import org.kohsuke.args4j.CmdLineException;
import org.kohsuke.args4j.CmdLineParser;
import org.kohsuke.args4j.Option;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;

/**
 ******************************************************************************
 **
 **  Program to enable direct talking to a commanded device.
 **
 **  @author Owen Saxton
 **
 ******************************************************************************
 */
public class TalkAscii {

    private final static PrintStream out = System.out;
    private final static String[] terms = {"\r\n", "\r", "\n"};
    private final Thread readW = new Thread(new Reader());
    private final Ascii dev = new Ascii();
    private int nRead = 0, nByte = 0, nTimeout = 0;
    private boolean open;

    private enum ConnType {

        NET(Ascii.CONN_TYPE_NETWORK),
        FTDI(Ascii.CONN_TYPE_FTDI),
        SERIAL(Ascii.CONN_TYPE_SERIAL);

        int value;

        ConnType(int value)
        {
            this.value = value;
        }

        public int getValue()
        {
            return value;
        }

    }

    private enum Terminator {
        CRLF, CR, LF;
    }


   /**
    ***************************************************************************
    **
    **  Inner class to implement device reading thread.
    **
    ***************************************************************************
    */
    private class Reader implements Runnable {

        @Override
        public void run()
        {
            while (open) {
                try {
                    String reply = dev.read();
                    out.println(reply);
                    nRead++;
                    nByte += reply.length();
                }
                catch (DriverTimeoutException e) {
                    nTimeout++;
                }
                catch (DriverException e) {
                    if (open) {
                        out.println(e);
                        System.exit(0);
                    }
                }
            }
        }

    }


   /**
    ***************************************************************************
    **
    **  Inner class for holding command line options.
    **
    ***************************************************************************
    */
    private static class Options {

        @Option(name="-c", metaVar="<connType>",
                usage="connection type: NET (default), FTDI or SERIAL")
        private ConnType type = ConnType.NET;

        @Option(name="-r", metaVar="<timeout>",
                usage="read timeout (sec) (default = 0.2)")
        private double timeout = 0.2;

        @Option(name="-i", metaVar="<index>",
                usage="FTDI device index (default = 0)")
        private int index = 0;

        @Option(name="-t", metaVar="<term>",
                usage="command terminator: CRLF (default), CR or LF")
        private Terminator term = Terminator.CRLF;

        @Argument(index=0, required=true, metaVar="<ident>",
                  usage="device identifier (node, serial or device)")
        private String ident;

        @Argument(index=1, required=true, metaVar="<parm>",
                  usage="device parameter (port number or baud rate)")
        private int parm;

    }


   /**
    ***************************************************************************
    **
    **  Main program.
    **
    **  @param  args  Command-line arguments
    **
    ***************************************************************************
    */
    public static void main(String[] args)
    {
        Options optns = new Options();
        CmdLineParser parser = new CmdLineParser(optns);
        try {
            parser.parseArgument(args);
            (new TalkAscii()).run(optns);
        }
        catch (CmdLineException e) {
            out.println(e.getMessage());
            System.err.println("Usage: TalkAscii [options...] <ident> <parm>");
            parser.printUsage(System.err);
        }
        catch (Exception e) {
            out.println(e);
        }
        System.exit(0);
    }


   /**
    ***************************************************************************
    **
    **  Sends commands to the device.
    **
    **  Loops reading and processing each new typed command line.
    **
    ***************************************************************************
    */
    private void run(Options optns) throws Exception
    {
        dev.open(optns.type.getValue(), optns.ident, optns.parm, optns.index);
        open = true;
        dev.setTimeout(optns.timeout);
        dev.setTerminator(terms[optns.term.ordinal()]);
        readW.setDaemon(true);
        readW.start();
        ConsoleReader readC = new ConsoleReader();

        while (true) {
            String line = readC.readLine("");
            if (line == null) break;
            if (line.equals("")) {
                out.format("#reads: %s; #bytes: %s; #timeouts: %s %n",
                           nRead, nByte, nTimeout);
            }
            else {
                dev.write(line);
            }
        }

        open = false;
        dev.close();
    }

}
