package org.lsst.ccs.drivers.ascii;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.drivers.serial.SerialPort;

/**
 *****************************************************************************
 **
 **  Communications I/O routines for serial devices.
 **
 **  @author Owen Saxton
 **
 *****************************************************************************
 */

public class AsciiIOSerial implements AsciiIO {

   /**
    **************************************************************************
    **
    **  Private fields.
    **
    **************************************************************************
    */
    private final SerialPort port = new SerialPort();
    private int timeout;


   /**
    **************************************************************************
    **
    **  Opens a connection.
    **
    **  @param  device  The device name of the serial port
    **
    **  @param  baud    The baud rate
    **
    **  @param  dummy   Dummy argument
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void open(String device, int baud, int dummy) throws DriverException
    {
        port.openPort(device);
        port.setParams(baud, 8, 1, 0);
    }


   /**
    **************************************************************************
    **
    **  Closes the connection.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void close() throws DriverException
    {
        port.closePort();
    }


   /**
    **************************************************************************
    **
    **  Writes a command.
    **
    **  @param  command  The command to write, including the terminator
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void write(byte[] command) throws DriverException
    {
        port.writeBytes(command);
    }


   /**
    **************************************************************************
    **
    **  Reads available response data.
    **
    **  @param  buff    The buffer to receive the response data
    **
    **  @param  offset  The offset to the first available byte in the buffer
    **
    **  @return  The number of bytes read
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    @Override
    public int read(byte[] buff, int offset) throws DriverException
    {
        return port.read(buff, offset, timeout);
    }


   /**
    **************************************************************************
    **
    **  Flushes any available response data.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void flush() throws DriverException
    {
        port.readBytes();
    }

    
   /**
    **************************************************************************
    **
    **  Sets the receive timeout.
    **
    **  @param  timeout  The receive timeout (ms).  0 means no timeout.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void setTimeout(int timeout) throws DriverException
    {
        this.timeout = timeout;
    }

}
