package org.lsst.ccs.drivers.ascii;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.drivers.ftdi.Ftdi;

/**
 *****************************************************************************
 **
 **  Communications I/O routines for serial devices connected via FTDI.
 **
 **  @author Owen Saxton
 **
 *****************************************************************************
 */

public class AsciiIOFtdi implements AsciiIO {

   /**
    **************************************************************************
    **
    **  Private fields.
    **
    **************************************************************************
    */
    Ftdi ftdi = new Ftdi();


   /**
    **************************************************************************
    **
    **  Opens a connection.
    **
    **  @param  serial  The serial number string of the interface
    **
    **  @param  baud    The baud rate
    **
    **  @param  index   The index of the device
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void open(String serial, int baud, int index) throws DriverException
    {
        ftdi.open(index, serial);
        ftdi.setBaudrate(baud);
    }


   /**
    **************************************************************************
    **
    **  Closes the connection.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void close() throws DriverException
    {
        ftdi.close();
    }


   /**
    **************************************************************************
    **
    **  Writes a command.
    **
    **  @param  command  The command to write, including the terminator
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void write(byte[] command) throws DriverException
    {
        ftdi.write(command);
    }


   /**
    **************************************************************************
    **
    **  Reads available response data.
    **
    **  @param  buff    The buffer to receive the response data
    **
    **  @param  offset  The offset to the first available byte in the buffer
    **
    **  @return  The number of bytes read
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    @Override
    public int read(byte[] buff, int offset) throws DriverException
    {
        int leng = 0, count = ftdi.getQueueStatus();
        if (count == 0) {
            leng = ftdi.read(buff, offset, 1);
            if (leng == 0) {
                throw new DriverTimeoutException("Read timed out");
            }
            offset++;
            count = ftdi.getQueueStatus();
        }
        if (offset + count > buff.length) {
            count = buff.length - offset;
        }
        if (count > 0) {
            leng += ftdi.read(buff, offset, count);
        }
        return leng;
    }


   /**
    **************************************************************************
    **
    **  Flushes any available response data.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void flush() throws DriverException
    {
        int count = ftdi.getQueueStatus();
        if (count > 0) {
            byte[] buff = new byte[count];
            ftdi.read(buff, 0, count);
        }
    }

    
   /**
    **************************************************************************
    **
    **  Sets the receive timeout.
    **
    **  @param  timeout  The receive timeout (ms).  0 means no timeout.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    @Override
    public void setTimeout(int timeout) throws DriverException
    {
        ftdi.setTimeouts(timeout, 0);
    }

}
