package org.lsst.ccs.drivers.ascii;

import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;

/**
 *****************************************************************************
 **
 **  General access routines for a device using Ascii commands.
 **
 **  @author Owen Saxton
 **
 *****************************************************************************
 */

public class Ascii {

   /**
    **************************************************************************
    **
    **  Public constants.
    **
    **************************************************************************
    */
    /** Connection type - network */
    public final static int CONN_TYPE_NETWORK = 0;

    /** Connection type - serial via FTDI chip */
    public final static int CONN_TYPE_FTDI    = 1;

    /** Connection type - serial via serial port */
    public final static int CONN_TYPE_SERIAL  = 2;
    

   /**
    **************************************************************************
    **
    **  Private constants & fields.
    **
    **************************************************************************
    */
    private final static byte CR = 0x0d, LF = 0x0a;
    private final byte[] buff = new byte[65536];
    private AsciiIO io;
    private String terminator = "\r\n";
    private int timeout = 1000;
    private int buffIn, buffOut;


   /**
    **************************************************************************
    **
    **  Opens a connection to the device.
    **
    **  @param  type   The type of connection to make
    **
    **  @param  ident  The device identifier:
    **                   host name or IP address for network
    **                   serial number for FTDI device
    **                   device name for serial
    **
    **  @param  parm   The device parameter:
    **                   port number for network
    **                   baud rate for FTDI or serial
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    public void open(int type, String ident, int parm) throws DriverException
    {
        open(type, ident, parm, 0);
    }


   /**
    **************************************************************************
    **
    **  Opens a connection to the device.
    **
    **  @param  type   The type of connection to make
    **
    **  @param  ident  The device identifier:
    **                   host name or IP address for network
    **                   serial number for FTDI device
    **                   device name for serial
    **
    **  @param  parm1  The first device parameter:
    **                   port number for network
    **                   baud rate for FTDI or serial
     **
    **  @param  parm2  The second device parameter:
    **                   unused for network or serial
    **                   device index for FTDI
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    public synchronized void open(int type, String ident, int parm1, int parm2)
        throws DriverException
    {
        if (io != null) {
            throw new DriverException("Device already connected");
        }

        AsciiIO newIo;

        switch (type) {

        case CONN_TYPE_NETWORK:
            newIo = new AsciiIONet();
            break;

        case CONN_TYPE_FTDI:
            newIo = new AsciiIOFtdi();
            break;

        case CONN_TYPE_SERIAL:
            newIo = new AsciiIOSerial();
            break;

        default:
            throw new DriverException("Invalid connection type: " + type);
        }

        newIo.open(ident, parm1, parm2);
        io = newIo;
        io.setTimeout(timeout);
    }


   /**
    **************************************************************************
    **
    **  Closes the device connection.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    public synchronized void close() throws DriverException
    {
        checkOpen();
        try {
            io.close();
        }
        finally {
            io = null;
            buffIn = buffOut = 0;
        }
    }


   /**
    **************************************************************************
    **
    **  Closes the device connection silently.
    **
    **  @return  Whether or not the close caused an error
    **
    **************************************************************************
    */
    public boolean closeSilent()
    {
        try {
            close();
            return true;
        }
        catch (DriverException e) {
            return false;
        }
    }


   /**
    **************************************************************************
    **
    **  Writes a command.
    **
    **  @param  command  The command to write, excluding terminator
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    public synchronized void write(String command) throws DriverException
    {
        checkOpen();
        io.write((command + terminator).getBytes());
    }


   /**
    **************************************************************************
    **
    **  Reads a response.
    **
    **  @return  The command response string
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    public synchronized String read() throws DriverException
    {
        checkOpen();
        try {
            int start = buffOut, end = buffIn, term = -1;
            while (true) {
                for (int j = start; j < end; j++) {
                    if (buff[j] == CR || buff[j] == LF) {
                        if (j == buffOut) {
                            buffOut++;
                        }
                        else {
                            term = j;
                            break;
                        }
                    }
                }
                if (term >= 0) {
                    String resp = new String(buff, buffOut, term - buffOut);
                    buffOut = term + 1;
                    return resp;
                }
                if (buffOut > 0) {
                    System.arraycopy(buff, buffOut, buff, 0, buffIn - buffOut);
                    buffIn -= buffOut;
                    buffOut = 0;
                }
                if (buffIn >= buff.length) {
                    throw new DriverException("Input buffer overflow");
                }
                start = buffIn;
                buffIn += io.read(buff, buffIn);
                end = buffIn;
            }
        }
        catch (DriverTimeoutException re) {
            throw re;
        }
        catch (DriverException re) {
            closeSilent();
            throw re;
        }
    }


   /**
    **************************************************************************
    **
    **  Reads a response after writing a command.
    **
    **  @param  command  The command to write, excluding terminator
    **
    **  @return  The command response string
    **
    **  @throws  DriverException
    **  @throws  DriverTimeoutException
    **
    **************************************************************************
    */
    public synchronized String read(String command) throws DriverException
    {
        flush();
        write(command);
        return read();
    }


   /**
    **************************************************************************
    **
    **  Flushes any unread data.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    public synchronized void flush() throws DriverException
    {
        checkOpen();
        buffIn = buffOut;
        io.flush();
    }


   /**
    **************************************************************************
    **
    **  Sets the command terminator.
    **
    **  @param  term  The terminator to be appended to each sent command
    **
    **************************************************************************
    */
    public void setTerminator(String term)
    {
        terminator = term;
    }


   /**
    **************************************************************************
    **
    **  Sets the read timeout.
    **
    **  @param  time  The read timeout (sec).  0 means no timeout.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    public synchronized void setTimeout(double time) throws DriverException
    {
        timeout = (int)(1000 * time);
        if (io != null) {
            io.setTimeout(timeout);
        }
    }


   /**
    **************************************************************************
    **
    **  Checks that the connection is open.
    **
    **  @throws  DriverException
    **
    **************************************************************************
    */
    private void checkOpen() throws DriverException
    {
        if (io == null) {
            throw new DriverException("Device not connected");
        }
    }

}
