package org.lsst.ccs.drivers.archon;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.ShortBuffer;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import static java.nio.file.StandardOpenOption.READ;
import java.util.HashSet;
import java.util.Random;
import java.util.Set;
import org.lsst.ccs.utilities.ccd.CCDGeometry;
import org.lsst.ccs.utilities.ccd.CCDTransform;
import org.lsst.ccs.utilities.ccd.CCDTransform.PixelType;

/*
 * Typically, a host program will issue the FRAME command, and check the
 * returned status data to see if there are any frame buffers holding
 * completed frames with a frame number greater than what the program has
 * already captured. If a new frame exists, the host program issues a LOCK
 * command to prevent that buffer from being overwritten, and then fetches
 * the pixel data using a FETCH command. The number of bytes to fetch for a
 * complete frame is the frame width times the frame height times 2 (for
 * 16-bit samples) or 4 (for 32-bit samples). Divide the number of bytes by
 * 1024 and round up to determine the number of blocks to fetch. The
 * starting address is 0x20000000 for frame buffer 1, 0x40000000 for frame
 * buffer 2, and 0x60000000 for frame buffer 3.
 */
public class RawImageData {

    public enum Sample {

        BIT16(2), BIT32(4);
        private final int bytes;

        private Sample(int bytes) {
            this.bytes = bytes;
        }

        public int bytes() {
            return bytes;
        }

        public int bits() {
            return 8 * bytes;
        }
    };

    public enum FrameMode {

        TOP, BOTTOM, SPLIT
    }

    int width;
    int height;
    Sample sample;
    FrameMode frameMode;
    long timestamp;
    long frame;
    byte[] data;

    public int getWidth() {
        return width;
    }

    public int getHeight() {
        return height;
    }

    public Sample getSample() {
        return sample;
    }

    public FrameMode getFrameMode() {
        return frameMode;
    }

    public long getTimestamp() {
        return timestamp;
    }

    public byte[] getData() {
        return data;
    }

    public long getFrame() {
        return frame;
    }

    /**
     * Create a test image. Used for test cases and by the DummyArchonController.
     * @param geom The CCD geometry to use to create the test image.
     * @return A simulated RawImageData object
     */
    public static RawImageData createTestImage(CCDGeometry geom) {
        RawImageData testData = new RawImageData();
        testData.sample = Sample.BIT16;
        testData.frameMode = FrameMode.SPLIT;
        testData.timestamp = System.currentTimeMillis();
        testData.height = geom.getTotalParallelSize();
        testData.width = geom.getTotalSerialSize();
        testData.data = new byte[testData.width * testData.height * 2];
        CCDTransform t = geom.getGlobalTransform();
        ByteBuffer bb = ByteBuffer.wrap(testData.data);
        bb.order(ByteOrder.LITTLE_ENDIAN);
        ShortBuffer sb = bb.asShortBuffer();
        Random random = new Random();
        for (int j = 0; j < testData.height; j++) {
            for (int i = 0; i < testData.width; i++) {
                t.setXY(i,j);
                sb.put((t.getPixelType() == PixelType.ACTIVE ? (short) (100 + 2 * i + 3 * j) : (short) Math.round(100 + 10 * random.nextGaussian())));
            }
        }
        return testData;
    }
    
    /**
     * Creates a test image consisting of concentric rings. The non-active
     * area is filled with noise, and the active area is filled with data.
     * @param geom The CCD geometry to use to create the test image.
     * @return A simulated RawImageData object
     */
    public static RawImageData createRippledImage(CCDGeometry geom) {
        RawImageData testData = new RawImageData();
        testData.sample = Sample.BIT16;
        testData.frameMode = FrameMode.SPLIT;
        testData.timestamp = System.currentTimeMillis();
        testData.height = geom.getTotalParallelSize();
        testData.width = geom.getTotalSerialSize();
        testData.data = new byte[testData.width * testData.height * 2];
        ByteBuffer bb = ByteBuffer.wrap(testData.data);
        bb.order(ByteOrder.LITTLE_ENDIAN);
        ShortBuffer sb = bb.asShortBuffer();
        Random random = new Random();
        // Fill everything with noise
        while (sb.hasRemaining()) {
            sb.put((short) Math.round(100 + 10 * random.nextGaussian()));
        }
        sb.clear();
        // Fill active area with concentric rings
        CCDTransform t = geom.getActiveTransform();
        for (int j = 0; j < geom.getActiveSerialSize(); j++) {
            for (int i = 0; i < geom.getActiveParallelSize(); i++) {
                double r = Math.sqrt(Math.pow((j - geom.getActiveSerialSize() / 2), 2) + Math.pow((i - geom.getActiveParallelSize() / 2), 2));
                t.setXY(i,j);
                sb.put(t.getGlobalY()+t.getGlobalX()*geom.getTotalSerialSize(), (short) Math.round(1000 + 200 * Math.sin(r / 30)));
            }
        }
        return testData;
    }

    /**
     * Create a RawImageData object by reading a file of raw data.
     *
     * @param rawFile The file to read
     * @param geom The CCD geometry to use to create the test image.
     * @return The raw image
     * @throws java.io.IOException
     */
    public static RawImageData createTestImage(Path rawFile, CCDGeometry geom) throws IOException {
        RawImageData testData = new RawImageData();
        testData.sample = Sample.BIT16;
        testData.frameMode = FrameMode.SPLIT;
        testData.timestamp = System.currentTimeMillis();
        testData.height = geom.getTotalParallelSize();
        testData.width = geom.getTotalSerialSize();
        testData.data = new byte[testData.width * testData.height * 2];
        ByteBuffer bb = ByteBuffer.wrap(testData.data);
        bb.order(ByteOrder.LITTLE_ENDIAN);
        Set<OpenOption> options = new HashSet<>();
        options.add(READ);
        try (SeekableByteChannel newByteChannel = Files.newByteChannel(rawFile, options)) {
            assert(newByteChannel.size()==testData.width * testData.height * 2);
            newByteChannel.read(bb);
            return testData;
        }
    }
}
